// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================

package com.adobe.xmp.core.serializer;

import java.util.EnumSet;



/**
 * Options for {@link XMPSerializer}.
 * Default serialization tooks place if no options are set.
 */
public class SerializeOptions implements Cloneable
{
	/**
	 * Enum for all serialization options.
	 */
	public enum Option
	{
		/** Omit the XML packet wrapper. */
		NO_PACKET_WRAPPER, 

		/** Omit the &lt;x:xmpmeta&gt;-tag */
		NO_XMPMETA_ELEMENT,

		/** Omits the Toolkit version attribute, not published, only used for Unit tests. */
		NO_VERSION_ATTRIBUTE,

		/**
		 * Mark packet as read-only. Default is a writable packet.
		 * This option can only be applied 
		 * if the option NO_PACKET_WRAPPER isn't set.
		 */
		READONLY_PACKET,
		
		/**
		 * Serialize to the canonical form of RDF if set.
		 * The compact form is the default serialization format
		 * (if this option is not set).
		 * To serialize to the canonical form, set the flag USE_CANONICAL_FORMAT.
		 */
		CANONICAL_FORMAT,
		
		/**
		 * Include a padding allowance for a thumbnail image. If no <tt>xmp:Thumbnails</tt> property
		 * is present, the typical space for a JPEG thumbnail is used.
		 */
		INCLUDE_THUMBNAIL_PAD,

		/**
		 * If this option is set, 
		 * the padding parameter is interpreted to be the overall packet length. 
		 * The actual amount of padding is computed. 
		 * An exception is thrown if the packet exceeds this length with no padding.
		 */
		EXACT_PACKET_LENGTH,
		
		/** Sort the struct properties, unordered arrays and qualifiers before serializing */
		SORT,

		/** Serializes with UTF-16 Big Endian encoding.
		 *  UTF-8 is the default */
		ENCODE_UTF16BE,
		
		/** Serializes with UTF-16 Little Endian encoding.
		 *  UTF-8 is the default */
		ENCODE_UTF16LE
	}
	
	/** Holds a set of serialize options */
	private EnumSet<Option> options;

	/**
	 * The amount of padding to be added if a writeable XML packet is created. If zero is passed
	 * (the default) an appropriate amount of padding is computed.
	 */
	private int padding = 2048;

	/**
	 * The string to be used as a line terminator. If empty it defaults to; linefeed, U+000A, the
	 * standard XML newline.
	 */
	private String newline = "\n";

	/**
	 * The string to be used for each level of indentation in the serialized
	 * RDF. If empty it defaults to two ASCII spaces, U+0020.
	 */
	private String indent = "  ";

	/**
	 * The number of levels of indentation to be used for the outermost XML element in the
	 * serialized RDF. This is convenient when embedding the RDF in other text, defaults to 0.
	 */
	private int baseIndent = 0;
	
	
	/** 
	 * Creates an options instance with one or more options. 
	 * @param options a list of option parameters
	 * */
	public SerializeOptions(Option... options)
	{
		// create an empty enum set.
		this.options = EnumSet.of(Option.CANONICAL_FORMAT);
		this.options.remove(Option.CANONICAL_FORMAT);
		
		for (Option option : options)
		{
			this.options.add(option);
		}
	}


	/**
	 * Check if an options is set.
	 * @param option the option flag
	 * @return Returns true if the option is set, false otherwise.
	 */
	public boolean hasOption(Option option)
	{
		return this.options.contains(option);
	}
	
	
	/**
	 * Adds an option flag.
	 * @param option an option enum
	 */
	public void add(Option option)
	{
		this.options.add(option);
	}
	
	
	/**
	 * Removes an option flag.
	 * @param option an option enum
	 */
	public void remove(Option option)
	{
		this.options.remove(option);
	}
	
	
	/**
	 * @return the options
	 */
	public EnumSet<Option> getOptions()
	{
		return options;
	}


	/**
	 * @param options the options to set
	 */
	public void setOptions(EnumSet<Option> options)
	{
		this.options = options;
	}


	/**
	 * @return the padding
	 */
	public int getPadding()
	{
		return padding;
	}


	/**
	 * @param padding the padding to set
	 */
	public void setPadding(int padding)
	{
		this.padding = padding;
	}


	/**
	 * @return the newline
	 */
	public String getNewline()
	{
		return newline;
	}


	/**
	 * @param newline the newline to set
	 */
	public void setNewline(String newline)
	{
		this.newline = newline;
	}


	/**
	 * @return the indent
	 */
	public String getIndent()
	{
		return indent;
	}


	/**
	 * @param indent the indent to set
	 */
	public void setIndent(String indent)
	{
		this.indent = indent;
	}


	/**
	 * @return the baseIndent
	 */
	public int getBaseIndent()
	{
		return baseIndent;
	}


	/**
	 * @param baseIndent the baseIndent to set
	 */
	public void setBaseIndent(int baseIndent)
	{
		this.baseIndent = baseIndent;
	}


	/**
	 * @return Returns the encoding as Java encoding String.
	 */
	public String getEncoding()
	{
		if (options.contains(Option.ENCODE_UTF16BE))
		{
			return "UTF-16BE";
		}
		else if (options.contains(Option.ENCODE_UTF16LE))
		{
			return "UTF-16LE";
		}
		else
		{
			return "UTF-8";
		}
	}
	

	
	/**
	 *
	 * @return Returns clone of this SerializeOptions-object with the same options set.
	 * @throws CloneNotSupportedException Cannot happen in this place.
	 */
	public Object clone() throws CloneNotSupportedException
	{
		return super.clone();
	}
}
