/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.cloudservices.provisioning;

import java.util.Map;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.osgi.annotation.versioning.ConsumerType;

/**
 * A {@code ProvisioningHandler} defines the service API to be implemented by service providers extending the
 * provisioning mechanism. Service providers may register OSGi services of this type to be used by the provisioning
 * default servlet to handle the provisioning of a specific service integration.
 *
 * <p>The {@code ProvisioningHandler} service must be registered with a {@link ProvisioningHandler#CLOUDSERVICE_ID}
 * registration property giving the name of the service integration supported by the service. The names will be used by
 * the provisioning default servlet to match the request parameter {@code servicename} with the correct handler to
 * trigger the provisioning.</p>
 *
 */
@ConsumerType
public interface ProvisioningHandler {

    /**
     * The name of the service registration property indicating the name(s) of the service provided by the provisioning
     * implementation. The value of this service property must be a single String.
     */
    String CLOUDSERVICE_ID = "cloudservice.id";

    /**
     * Triggers the provisioning of a cloud service.
     *
     * @param resourceResolver
     *         A {@link org.apache.sling.api.resource.ResourceResolver} instance
     * @param properties
     *         A {@link java.util.Map} of the properties required to do the provisioning.
     * @return Newly created service configuration
     * @throws ProvisioningException
     *         if an error occurs during the setup
     */
    Resource setup(ResourceResolver resourceResolver, Map<String, Object> properties) throws ProvisioningException;

    /**
     * Is rolling back the setup step.
     *
     * @param resourceResolver
     *         A {@link org.apache.sling.api.resource.ResourceResolver} instance
     * @param properties
     *         A {@link java.util.Map} of the properties required to do the provisioning.
     * @throws ProvisioningException
     *         if an error occurs during the rollback
     */
    void rollback(ResourceResolver resourceResolver, Map<String, Object> properties) throws ProvisioningException;

    /**
     * Applies the {@code configuration} to the specified {@code paths}.
     *
     * @param resourceResolver
     *         A {@link org.apache.sling.api.resource.ResourceResolver} instance
     * @param paths
     *         An Array of paths to apply the {@code configuration} to
     * @param configuration
     *         Configuration {@link Resource} created by the {@link #setup(ResourceResolver, Map)} method.
     * @throws ProvisioningException
     *         if an error occurs during application
     */
    void applyToPaths(ResourceResolver resourceResolver, String[] paths, Resource configuration) throws ProvisioningException;

    /**
     * Revokes the {@code configuration} from the specified {@code paths}.
     *
     * @param resourceResolver
     *         A {@link org.apache.sling.api.resource.ResourceResolver} instance
     * @param paths
     *         the paths to revoke the {@code configuration} from
     * @param configuration
     *         configuration {@link Resource}
     * @throws ProvisioningException
     *         if an error occurs during the revoke
     */
    void revokeFromPaths(ResourceResolver resourceResolver, String[] paths, Resource configuration) throws ProvisioningException;

}
