/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.security.user.util;

import java.util.ArrayList;
import java.util.List;

import com.adobe.granite.security.user.UserProperties;
import com.adobe.granite.security.user.UserPropertiesManager;
import com.adobe.granite.security.user.UserPropertiesService;
import com.adobe.granite.security.user.internal.AggregatedUserProperties;
import com.adobe.granite.security.user.internal.AuthorizableInfo;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

/**
 * AuthorizableUtil...
 */
public class AuthorizableUtil {

    private static Logger log = LoggerFactory.getLogger(AuthorizableUtil.class);

    static final String FAMILY_NAME = UserPropertiesService.PROFILE_PATH + "/familyName";
    static final String GIVEN_NAME = UserPropertiesService.PROFILE_PATH + "/givenName";

    /**
     * @deprecated no longer stored in the authorizable properties but only in the profile
     */
    @Deprecated
    private static final String PROPERTY_NAME = "rep:fullname";
    /**
     * @deprecated no longer stored in the authorizable properties but only in the profile
     */
    @Deprecated
    private static final String PROPERTY_DESCRIPTION = "rep:description";

    @Deprecated
    private static final String PROPERTY_FIRST_NAME = "cq:first-name";
    /**
     * @deprecated no longer stored in the authorizable properties but only in the profile
     */
    @Deprecated
    private static final String PROPERTY_LAST_NAME = "cq:last-name";

    public static UserProperties getProfile(@Nonnull UserPropertiesManager userPropertiesManager, @Nonnull String authorizableId) throws RepositoryException {
        UserProperties profiles = userPropertiesManager.getUserProperties(authorizableId, UserPropertiesService.PROFILES_ROOT, UserPropertiesManager.DESCENDING_RANKING_COMPARATOR);
        UserProperties deprecatedProfile = userPropertiesManager.getUserProperties(authorizableId, UserPropertiesService.PROFILE_PATH);

        List<UserProperties> agg = profiles.getAggregatedUserProperties();
        if (agg.isEmpty()) {
            return deprecatedProfile;
        } else {
            AuthorizableInfo info = new AuthorizableInfo(profiles.getAuthorizableID(), profiles.getAuthorizablePath(), profiles.isGroupProperties(), profiles.getPrincipalName());
            List<UserProperties> l = new ArrayList<UserProperties>();
            if (deprecatedProfile != null) {
                l.add(deprecatedProfile);
            }
            l.addAll(agg);
            return new AggregatedUserProperties(info, profiles.getNode(), l);
        }
    }

    /**
     * Utility to retrieve the name of an authorizable that takes deprecated
     * user properties such as present in older versions if CQ into account.
     *
     * @param authorizable the authorizable
     * @return the name of the specified authorizable
     * @throws RepositoryException in case of a failure
     * @deprecated since AEM6.4 because it is not multiple profile aware. Please use {@link #getFormattedName(ResourceResolver, Authorizable)}
     */
    @Deprecated
    public static String getName(Authorizable authorizable) throws RepositoryException {
        String name = getProperty(authorizable, GIVEN_NAME, PROPERTY_FIRST_NAME);
        if (authorizable.isGroup()) {
            if (name != null && !name.isEmpty()) {
                return name;
            }
            name = getProperty(authorizable, PROPERTY_NAME);
            if (name != null && !name.isEmpty()) {
                return name;
            }
            name = getProperty(authorizable, PROPERTY_DESCRIPTION);
            if (name != null && !name.isEmpty()) {
                return name;
            }
        } else {
            StringBuilder buf = new StringBuilder();
            if (name != null) {
                buf.append(name);
            }
            String familyName = getProperty(authorizable, FAMILY_NAME, PROPERTY_LAST_NAME);
            if (familyName != null && !familyName.isEmpty()) {
                buf.append(' ').append(familyName);
            }
            if (buf.length() > 0) {
                return buf.toString();
            }
        }
        // fallback: return the ID.
        return authorizable.getID();
    }

    /**
     * Utility to retrieve the formatted name of a user. Always use the faster
     * {@link #getFormattedName(ResourceResolver, Authorizable)} if possible.
     *
     * @param resolver Resource resolver to use to get the {@code UserPropertiesManager}
     * @param userId   ID of the user to get the display name from
     * @return User's display name or its id if an error occurred
     */
    public static String getFormattedName(ResourceResolver resolver, String userId) {
        return getFormattedName(resolver, null, userId, null);
    }

    /**
     * Utility to retrieve the formatted name of a user.
     *
     * @param resolver Resource resolver to use to get the {@code UserPropertiesManager}
     * @param authorizable the authorizable
     * @return User's display name or its id if an error occurred
     */
    public static String getFormattedName(ResourceResolver resolver, Authorizable authorizable) {
        return getFormattedName(resolver, authorizable, null, null);
    }

    /**
     * Utility to retrieve the formatted name of a user. Always use the faster
     * {@link #getFormattedName(ResourceResolver, Authorizable, String)} if possible.
     *
     * @param resolver Resource resolver to use to get the {@code UserPropertiesManager}
     * @param userId   ID of the user to get the display name from
     * @param nameDisplayOrder  Order of given, middle and family names.
     * Western name order should be "givenName middleName familyName",
     * Eastern name order should be "familyName givenName middleName".
     * @return User's display name or its id if an error occurred
     */
    public static String getFormattedName(ResourceResolver resolver, String userId, String nameDisplayOrder) {
        return getFormattedName(resolver, null, userId, nameDisplayOrder);
    }

    /**
     * Utility to retrieve the formatted name of a user
     *
     * @param resolver Resource resolver to use to get the {@code UserPropertiesManager}
     * @param authorizable the authorizable
     * @param nameDisplayOrder  Order of given, middle and family names.
     * Western name order should be "givenName middleName familyName",
     * Eastern name order should be "familyName givenName middleName".
     * @return User's display name or its id if an error occurred
     */
    public static String getFormattedName(ResourceResolver resolver, Authorizable authorizable,
                                          String nameDisplayOrder) {
        return getFormattedName(resolver, authorizable, null, nameDisplayOrder);
    }

    private static String getFormattedName(ResourceResolver resolver, Authorizable authorizable, String userId,
                                           String nameDisplayOrder) {
        if (resolver != null && (authorizable != null || !StringUtils.isEmpty(userId))) {
            try {
                UserPropertiesManager upm = resolver.adaptTo(UserPropertiesManager.class);
                if (upm != null) {
                    UserProperties props = (authorizable != null) ? AuthorizableUtil.getProfile(upm, authorizable.getID()) :
                            AuthorizableUtil.getProfile(upm, userId);
                    if (props != null) {
                        String displayName = props.getDisplayName(nameDisplayOrder);
                        if (StringUtils.isNotEmpty(displayName)) {
                            return displayName;
                        }
                    }
                }
            } catch (RepositoryException e) {
                log.debug("Unable to get display name for {}: {}.", userId, e.getMessage());
            }
        }
        try {
            return (authorizable != null) ? authorizable.getID() : userId;
        } catch (RepositoryException e) {
            log.error("Unable to get authorizable id: {}", e.getMessage());
            return null;
        }
    }

    private static String getProperty(Authorizable authorizable, String propertyName) {
        return getProperty(authorizable, propertyName, null);
    }

    private static String getProperty(Authorizable authorizable, String propertyName, String fallbackName) {
        try {
            if (authorizable.hasProperty(propertyName)) {
                Value[] vs = authorizable.getProperty(propertyName);
                if (vs != null && vs.length > 0) {
                    return vs[0].getString();
                }
            }
            // try fallback property name
            if (fallbackName != null && authorizable.hasProperty(fallbackName)) {
                Value[] vs = authorizable.getProperty(fallbackName);
                if (vs != null && vs.length > 0) {
                    return vs[0].getString();
                }
            }
        } catch (RepositoryException e) {
            // cannot retrieve property
        }

        // no such property.
        return null;
    }
}
