/*
 * File: CustomFilterRegistry.java
 *  
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2008 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import com.adobe.internal.pdftoolkit.core.filter.spi.CustomDecodeFilter;
import com.adobe.internal.pdftoolkit.core.filter.spi.CustomEncodeFilter;
import com.adobe.internal.pdftoolkit.core.filter.spi.CustomFilter;
import com.adobe.internal.pdftoolkit.core.filter.spi.CustomFilterException;
import com.adobe.internal.pdftoolkit.core.types.ASName;


/**
 * A registry that enables clients to register their custom stream filters
 * with us.
 * The filters will be used when encoding and decoding the CosStreams
 * containing relevant content.
 *  
 * @author mdharan
 *
 */
public class CustomFilterRegistry 
{
	//TODO : Populate the Gibson standard filters into the custom filter registry.
	//TODO : Instead of accepting a list during registration, force the registry of each encode/decode filter separately. 
	//This allows clients to have tighter control over what filters registered succesfully.
	
	private HashMap<ASName, CustomDecodeFilter> decodeFilters = new HashMap<ASName, CustomDecodeFilter>();
	private HashMap<ASName, CustomEncodeFilter> encodeFilters = new HashMap<ASName, CustomEncodeFilter>();

	public CustomFilterRegistry() {
		super();		
	}
	
	/**
	 * Register the given filter with the registry.
	 * @param customFilter
	 */
	public void registerFilter(CustomFilter customFilter)
	{
		if (customFilter instanceof CustomEncodeFilter)
			registerEncodeFilter((CustomEncodeFilter) customFilter);
		if (customFilter instanceof CustomDecodeFilter)
			registerDecodeFilter((CustomDecodeFilter) customFilter);	
	}
	

	/**
	 * Deregister your custom filter implementation
	 * @param filter
	 */
	protected void unregister(CustomFilter filter)
	{
		
		if (decodeFilters != null)
		{
			if (decodeFilters.containsKey(filter.getName()))
					decodeFilters.remove(filter.getName());
		}
		if (encodeFilters != null)
		{
			if (encodeFilters.containsKey(filter.getName()))
					encodeFilters.remove(filter.getName());
		}
	}
	
	/**
	 * Checks if a custom encode filter is registered
	 * @param filterName
	 * @return true if a custom encode filter is registered
	 */
	public boolean isEncodeFilterRegistered(ASName filterName)
	{
		if (encodeFilters != null)
		{
			if (encodeFilters.containsKey(filterName))
					return true;
		}
		return false;
	}
	
	/**
	 * Checks if a custom decode filter is registered
	 * @param filterName
	 * @return true if a custom decode filter is registered
	 */
	public boolean isDecodeFilterRegistered(ASName filterName)
	{
		if (decodeFilters != null)
		{
			if (decodeFilters.containsKey(filterName))
					return true;
		}
		return false;
	}
	
	/**
	 * Returns the list of all registered encode filters
	 * @return Collection of encode filters
	 */
	public Collection<CustomEncodeFilter> getEncodeFilters()
	{
		return  encodeFilters.values();
	}
	
	/**
	 * Returns the list of all registered decode filters
	 * @return Collection of decode filters
	 */
	public Collection<CustomDecodeFilter> getDecodeFilters()
	{
		return decodeFilters.values();
	}
		
	/**
	 * Register your custom encode filter implementation
	 * @param filter
	 */
	public void registerEncodeFilter( CustomEncodeFilter filter)
	{
		encodeFilters.put(filter.getName(), filter);
	}
	
	
	/**
	 * Register your custom decode filter implementation
	 * @param filter
	 */
	public void registerDecodeFilter( CustomDecodeFilter filter)
	{
		decodeFilters.put(filter.getName(), filter);
	}
	

	/**
	 * Decode the stream using the filters registered for decode.
	 * @param filterName
	 * @param inStm
	 * @param params
	 * @return InputStream containing the decoded bytes.
	 * @throws CustomFilterException
	 */
	protected InputStream decode(ASName filterName, InputStream inStm,
			FilterParams params) throws CustomFilterException 
	{
		InputStream decodedStream = null;
		
		CustomDecodeFilter filterToUse = decodeFilters.get(filterName);
		if (filterToUse != null)
		{
			decodedStream = filterToUse.decode(inStm, params);		
		}
			
		return decodedStream;
	}
	
	/**
	 * Encode the stream using the filters registered for encode.
	 * @param filterName
	 * @param dstStm
	 * @param params
	 * @return InputStream containing the decoded bytes.
	 * @throws CustomFilterException
	 */
	public OutputStream encode(ASName filterName, OutputStream dstStm,
			FilterParams params) throws CustomFilterException 
	{
		OutputStream encodedStream = null;
		
		CustomEncodeFilter filterToUse = encodeFilters.get(filterName);
		if (filterToUse != null)
		{
			encodedStream = filterToUse.encode(dstStm, params);		
		}
			
		return encodedStream;
	}

	/**
	 * Updates the filterParams based on capability of the custom encode filter  
	 * @param filterName
	 * @param params
	 * @return updated filterParams
	 */
	protected FilterParams updateFilterParams(ASName filterName, FilterParams params) 
	{
		FilterParams newParams = params;
		
		CustomEncodeFilter filterToUse = encodeFilters.get(filterName);
		if (filterToUse != null)
		{
			newParams = filterToUse.updateFilterParams(params);		
		}
			
		return newParams;
	}

	/**
	 * Returns list of all the registered filters
	 * @return list of registered filters
	 */
	public List getRegisteredFilters() 
	{
		List customFilterList = new ArrayList();
		Collection<CustomDecodeFilter> decodeFilterList = getDecodeFilters();
		Collection<CustomEncodeFilter> encodeFilterList = getEncodeFilters();
		if (decodeFilterList != null)
		{
			customFilterList.addAll(decodeFilterList);
		}
		if (encodeFilterList != null)
		{
			customFilterList.addAll(encodeFilterList);
		}
		
		return customFilterList;
	}
	
	/**
	 * Returns a list of filter names representing decode filters present in Gibson
	 * @return list of decode filters present in Gibson by default.
	 */
	public static List/*<ASName>*/ getDefaultDecodeFilters()
	{
		return java.util.Arrays.asList(new ASName[] {ASName.k_FlateDecode,ASName.k_Fl,ASName.k_ASCII85Decode,ASName.k_A85,
				ASName.k_ASCIIHexDecode,ASName.k_AHx,ASName.k_LZWDecode,ASName.k_LZW,ASName.k_Crypt,ASName.k_CCITTFaxDecode,
				ASName.k_CCF,ASName.k_DCTDecode,ASName.k_DCT,ASName.k_RunLengthDecode,ASName.k_RL});
		
	}
	
	/**
	 * Returns a list of filter names representing encode filters present in Gibson
	 * @return list of encode filters present in Gibson by default.
	 */
	public static List/*<ASName>*/ getDefaultEncodeFilters()
	{
		return java.util.Arrays.asList(new ASName[] {ASName.k_FlateDecode,ASName.k_ASCII85Decode,ASName.k_ASCIIHexDecode,ASName.k_LZWDecode,
				ASName.k_RunLengthDecode,ASName.k_Crypt});
		
	}
	
	/**
	 * searches registered custom decode filter by name and returns it, if present. 
	 * @param filterName
	 * @return CustomDecodeFilter
	 */
	public CustomDecodeFilter getRegisteredDecodeFilterByName(ASName filterName){
		return decodeFilters.get(filterName);
	}
	
	/**
	 * searches registered custom encode filter by name and returns it, if present. 
	 * @param filterName
	 * @return CustomEncodeFilter
	 */
	public CustomEncodeFilter getRegisteredEncodeFilterByName(ASName filterName){
		return encodeFilters.get(filterName);
	}
	
}