/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.result;

import com.day.crx.statistics.Report;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import java.util.Iterator;
import java.util.Calendar;
import java.util.List;
import java.util.ArrayList;

import org.apache.jackrabbit.util.Text;

/**
 * <code>ResultQualityReport</code> returns a historical list of average
 * positions in a query result list for a given node.
 *
 * @author mreutegg
 */
public class ResultQualityReport extends Report {

    /**
     * Default period is 12 months
     */
    private int period = 12;

    /**
     * The path of the result node.
     */
    private final String path;

    /**
     * Creates a new report.
     *
     * @param dataPath the location where the result statistics are stored.
     * @param path the path of a node.
     */
    public ResultQualityReport(String dataPath, String path) {
        super(dataPath);
        this.path = path;
    }

    /**
     * {@inheritDoc}
     * Returns result rows with the following objects:
     * <ul>
     * <li>Path <code>String</code> indicating the month</li>
     * <li>Average position as a <code>Long</code></li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        ResultSelected rs = new ResultSelected(getDataPath(), path, 1, "");
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.add(Calendar.MONTH, -(getPeriod() - 1));
        List data = new ArrayList();
        for (int i = 0; i < getPeriod(); i++) {
            rs.setTimestamp(cal.getTimeInMillis());
            String p = Text.getRelativeParent(rs.getPath(), 1);
            long count = 0;
            double position = 0;
            if (session.itemExists(p)) {
                Item item = session.getItem(p);
                if (item.isNode()) {
                    Node month = (Node) item;
                    for (NodeIterator days = month.getNodes(); days.hasNext(); ) {
                        Node d = days.nextNode();
                        long c = d.getProperty(ResultSelected.COUNT).getLong();
                        double ap = d.getProperty(ResultSelected.AVG_POSITION).getDouble();
                        count += c;
                        position += ap * c;
                    }
                }
            }
            Long avgPosition = new Long(0);
            if (count != 0) {
                avgPosition = new Long((long) (position / count));
            }
            data.add(new Object[]{p, avgPosition});
            cal.add(Calendar.MONTH, 1);
        }
        return data.iterator();
    }

    /**
     * @return the report period in number of days.
     */
    public int getPeriod() {
        return period;
    }

    /**
     * @param period the report period in number of days.
     */
    public void setPeriod(int period) {
        this.period = period;
    }
}
