/*
 *    File: Cmap.java
 *
 *    ADOBE CONFIDENTIAL
 *    ___________________
 *
 *    Copyright 2004-2006 Adobe Systems Incorporated
 *    All Rights Reserved.
 *
 *    NOTICE: All information contained herein is, and remains the property of
 *    Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *    and technical concepts contained herein are proprietary to Adobe Systems
 *    Incorporated and its suppliers and may be covered by U.S. and Foreign
 *    Patents, patents in process, and are protected by trade secret or
 *    copyright law. Dissemination of this information or reproduction of this
 *    material is strictly forbidden unless prior written permission is obtained
 *    from Adobe Systems Incorporated.
 */

package com.adobe.fontengine.font.opentype;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CoderMalfunctionError;
import java.nio.charset.CoderResult;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import com.adobe.agl.charset.CharsetEncoderICU;
import com.adobe.agl.text.UTF16;
import com.adobe.fontengine.CharsetUtil;
import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.SubsetDefaultImpl;
import com.adobe.fontengine.font.SubsetSimpleTrueType;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.Name.MicrosoftEncodingId;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to a 'cmap' table. 
 * 
 * <h4>Encoding forms support</h4>
 * 
 * <p>A number of subtable formats are designed to combine the decoding of a 
 * character stream and the computation of the glyph ID. For example,
 * format 8 is geared to support encoding forms with one or two 
 * 16 bits code units (such as UTF-16) and has a 64K bit array to
 * indicate those code units which represent a code point on their
 * own and those code units which are the first of a pair. The idea
 * is that the details of the encoding form can be captured in 
 * the font.</p>
 * 
 * <p>Our point of view is that encoding forms are not the business 
 * of fonts, so we ignore those parts of those formats.</p>
 * 
 * <h4>Version handling</h4>
 * 
 * <p>'cmap' tables have only a minor version number.
 * This implementation:
 * <ul> 
 * <li>fully supports version 0 tables</li>
 * <li>interprets version x > 0 tables as version 0 tables</li>
 * </ul>
 */

final public class Cmap extends Table {

	//-------------------------------------------------------------- constants ---

	/** Enum classes for the various platformIDs and encodingIDs. */

	final public static class PlatformID {
		public static final int UNICODE = 0;
		public static final int MACINTOSH = 1;
		public static final int ISO = 2;
		public static final int MICROSOFT = 3;
		public static final int CUSTOM = 4;
	}

	final public static class UnicodeEncodingID {
		public static final int ID_1_0 = 0;
		public static final int ID_1_1 = 1;
		public static final int ID_10646_1993 = 2;
		public static final int ID_2_0_BMP = 3;
		public static final int ID_2_0_FULL = 4;
		public static final int UNI_VARIATION_SEQ = 5;
	}

	final public static class MS_EncodingID {
		public static final int SYMBOL = 0;
		public static final int UNICODE_BMP = 1;
		public static final int SHIFTJIS = 2;
		public static final int PRC = 3;
		public static final int BIG5 = 4;
		public static final int WANSUNG = 5;
		public static final int JOHAB = 6;
		public static final int UNICODE_FULL = 10;
	}

	final public static class MacEncodingID {
		public static final int ROMAN = 0;
		public static final int JAPANESE = 1;
		public static final int CHINESE_TRADITIONAL = 2;
		public static final int KOREAN = 3;
		public static final int ARABIC = 4;
		public static final int HEBREW = 5;
		public static final int GREEK = 6;
		public static final int RUSSIAN = 7;
		public static final int RSYMBOL = 8;
		public static final int DEVANAGARI = 9;
		public static final int GURMUKHI = 10;
		public static final int GUJARATI = 11;
		public static final int ORIYA = 12;
		public static final int BENGALI = 13;
		public static final int TAMIL = 14;
		public static final int TELUGU = 15;
		public static final int KANNADA = 16;
		public static final int MALAYALAM = 17;
		public static final int SINHALESE = 18;
		public static final int BURMESE = 19;
		public static final int KHMER = 20;
		public static final int THAI = 21;
		public static final int LAOTIAN = 22;
		public static final int GEORGIAN = 23;
		public static final int ARMENIAN = 24;
		public static final int CHINESE_SIMPLIFIED = 25;
		public static final int TIBETAN = 26;
		public static final int MONGOLIAN = 27;
		public static final int GEEZ = 28;
		public static final int SLAVIC = 29;
		public static final int VIETNAMESE = 30;
		public static final int SINDHI = 31;
		public static final int UNINTERPRETED = 32;
	}

	/**
	 * Class used to receive the list of recognized subtables in this cmap table.
	 */
	public static interface CmapSelector {
		/**
		 * Called for each recognized subtable. The platform ids will always come
		 * from the list defined in class PlatformID. The encoding ids will always
		 * come from the lists defined in the classes MS_EncodingID and
		 * UnicodeEncodingID. The index tells which subtable is being enumerated.
		 */
		public void cmapFound(int platformID, int platformEncoding, int index)
		throws InvalidFontException, UnsupportedFontException;
	}


	//---------------------------------------------------------------- members ---

	/*
	 * To support the function {@link unicodeChar2glyph}, we somehow need to
	 * select a cmap subtable through which to look up the character.
	 * 
	 * When the font contains at least one cmap subtable for Unicode, we can
	 * simply use one of those cmap subtables; we assume that if there is some
	 * character 'usv' and some character set 's' such that cmap (s, unicode2s
	 * (usv)) == g != 0, then the font designer also arranged for cmap (unicode,
	 * usv) == g), i.e. at least one of the Unicode subtables is as complete as
	 * any of the non-Unicode subtables. Furthermore, we assume that the various
	 * cmap subtables for Unicode are compatible, i.e. that if cmap (x, usv) == g !=
	 * 0 for some Unicode subtable x, then for any Unicode subtable y that can
	 * deal with usv, the font designer also arranged for cmap (y, usv) == g. This
	 * condition allows us to rank the various Unicode subtables, because they
	 * should differ only in coverage. We use the following order, starting with
	 * the best to use: MS/UNICODE_FULL, Unicode/ID_2_0_FULL, MS/UNICODE_BMP,
	 * Unicode/ID_2_0_BMP, Unicode/ID_10646_1993. The bottom line is that by just
	 * looking at which cmap subtables are present, we can decide that we are
	 * always going to use one specific subtable, and when we use it, we simply
	 * look up the scalar value, without convertion. We also expect most fonts to
	 * have a Unicode subtable, so we can make that determination at construction
	 * time, record the subtable to use in a final member, and we then never need
	 * synchronization in {@link unicodeChar2glyph}.
	 * 
	 * Our next approach is to use a MS/SYMBOL subtable if there is one.
	 * Furthermore, if the code point we want to use is not mapped and is in
	 * the range 0x00..0xFF, we also try to lookup the entry 0xF0xy.
	 * 
	 * When the font does not contain any cmap subtable for Unicode, choosing a
	 * priori is more problematic. May be there is a MacDevanagari and a
	 * MacJapanese subtable, and which one we should use depend on the character
	 * we need to lookup. Of course, this particular scenario is unlikely; a more
	 * common case is to have a MacRoman and a MacJapanese subtable. What we can
	 * do in advance is select the order in which we are going to try the
	 * subtables. If we cannot convert the Unicode character to the character set
	 * associated with a subtable (because it does not exist in that character
	 * set), we simply move to the next subtable. If we can convert it, then we
	 * use that subtable (even if it maps to .notdef - we assume that the various
	 * subtables are compatible in that sense). The motivation for the ordering of
	 * the subtables is to handle the common case of MacRoman+MacJapanese: most
	 * characters are likely to be handled via the MacJapanese subtable only, so
	 * we may as well start there. Just like in the Unicode subtable case, we make
	 * that determination at construction time to avoid synchronization.
	 * 
	 * We treat Unicode/ID_1_0 and Unicode/ID_1_1 as non-Unicode character sets,
	 * since code points have moved between those Unicode versions and Unicode as
	 * we know it today.
	 */

	/** The offset of the Unicode subtable to use.
	 * If no such subtable should be used, -1. */
	protected final int unicodeSubtableOffset; 

	/** The offset of the "old" (1.0 or 1.1) Unicode subtable to use.
	 * If no such subtable should be used, -1.
	 */
	protected final int oldUnicodeSubtableOffset;

	/** Describes how to use a non-Unicode cmap.
	 */
	static class NonUnicodeCmap 
	{
		/* The index of the non-Unicode cmap */
		final int subtableOffset;
		/* The corresponding charset; null means no conversion. */
		final String charsetName;
		/* A decoder for that charset. This is set on demand, 
		 * and is protected by 'this'. */
		Charset charset = null;

		public NonUnicodeCmap (int subtableOffset, String charsetName) 
		{
			this.subtableOffset = subtableOffset;
			this.charsetName = charsetName;
		}

		// only the charset is cached as the encoder is not threadsafe
		public CharsetEncoder getCharsetEncoder() 
		throws UnsupportedCharsetException, IllegalCharsetNameException
		{
			// set the charset the first time
			if (this.charset == null && this.charsetName != null) 
			{
				synchronized (this) 
				{
					if (this.charset == null && this.charsetName != null) 
					{
						this.charset = CharsetUtil.forNameICU(this.charsetName);
					}
				}
			}

			// must create a charset encoder for each caller as it in NOT threadsafe
			CharsetEncoder encoder = null;
			if (this.charset != null)
			{
				encoder = this.charset.newEncoder();
				if (encoder instanceof CharsetEncoderICU)
				{
					((CharsetEncoderICU) encoder).setFallbackUsed(true);
				}
			}
			return encoder; 
		}
	}

	/** The non-Unicode cmaps to use. */
	protected final NonUnicodeCmap[] nonUnicodeSubtableIndices;

	/** The offset of the symbol subtable. */
	protected final int symbolSubtableOffset;

	/** Padding bytes used with symbol subtables */
	protected int paddingByte = 0; 


	//------------------------------------------------------------ constructor ---

	protected Cmap (FontByteArray buffer) 
	throws IOException, InvalidFontException, UnsupportedFontException 
	{
		super (buffer);

		int offset;

		if ((offset = probe (PlatformID.MICROSOFT, MS_EncodingID.UNICODE_FULL)) != -1) {
			unicodeSubtableOffset = offset; }
		else if ((offset = probe (PlatformID.UNICODE, UnicodeEncodingID.ID_2_0_FULL)) != -1) {
			unicodeSubtableOffset = offset; }
		else if ((offset = probe (PlatformID.MICROSOFT, MS_EncodingID.UNICODE_BMP)) != -1) {
			unicodeSubtableOffset = offset; }
		else if ((offset = probe (PlatformID.UNICODE, UnicodeEncodingID.ID_2_0_BMP)) != -1) {
			unicodeSubtableOffset = offset; }
		else {
			unicodeSubtableOffset = -1; }

		if ((offset = probe (PlatformID.UNICODE, UnicodeEncodingID.ID_10646_1993)) != -1) {
			oldUnicodeSubtableOffset = offset; }
		else if ((offset = probe (PlatformID.UNICODE, UnicodeEncodingID.ID_1_1)) != -1) {
			oldUnicodeSubtableOffset = offset; }
		else if ((offset = probe (PlatformID.UNICODE, UnicodeEncodingID.ID_1_0)) != -1) {
			oldUnicodeSubtableOffset = offset; }
		else {
			oldUnicodeSubtableOffset = -1; }

		if ((offset = probe (PlatformID.MICROSOFT, MS_EncodingID.SYMBOL)) != -1) {
			symbolSubtableOffset = offset;
			paddingByte = computeSymbolPad(symbolSubtableOffset);}
		else {
			symbolSubtableOffset = -1; }

		ArrayList /*<NonUnicodeCmap>*/ v = new ArrayList /*<NonUnicodeCmap>*/ ();

		if ((offset = probe (PlatformID.MICROSOFT, MS_EncodingID.SHIFTJIS)) != -1) {
			v.add (new NonUnicodeCmap (offset, "windows-932")); }
		if ((offset = probe (PlatformID.MICROSOFT, MS_EncodingID.PRC)) != -1) {
			v.add (new NonUnicodeCmap (offset, "windows-936")); }
		if ((offset = probe (PlatformID.MICROSOFT, MS_EncodingID.BIG5)) != -1) {
			v.add (new NonUnicodeCmap (offset, "windows-950")); }
		if ((offset = probe (PlatformID.MICROSOFT, MS_EncodingID.WANSUNG)) != -1) {
			v.add (new NonUnicodeCmap (offset, "windows-949")); }
		if ((offset = probe (PlatformID.MICROSOFT, MS_EncodingID.JOHAB)) != -1) {
			v.add (new NonUnicodeCmap (offset, "ms1361")); }

		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.ROMAN)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacRoman")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.JAPANESE)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacJapanese")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.CHINESE_TRADITIONAL)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacChineseTraditional")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.KOREAN)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacKorean")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.ARABIC)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacArabic")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.HEBREW)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacHebrew")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.GREEK)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacGreek")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.RUSSIAN)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacCyrillic")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.RSYMBOL)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacSymbol")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.THAI)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacThai")); }
		if ((offset = probe (PlatformID.MACINTOSH, MacEncodingID.CHINESE_SIMPLIFIED)) != -1) {
			v.add (new NonUnicodeCmap (offset, "MacChineseSimplified")); }

		nonUnicodeSubtableIndices = (NonUnicodeCmap[]) v.toArray (new NonUnicodeCmap [0]);
	}

	/** Heuristically determine what padding is used by a symbol cmap, using
	 * the same method that CoolType does (StrDecod.cpp, ComputeTTSymbolPadByte)
	 */
	private int computeSymbolPad(int stOffset) 
	throws UnsupportedFontException, InvalidFontException
	{
		final int[] possiblePadding = {0xF000, 0xF100, 0xF200, 0};

		for (int i = 0; i < possiblePadding.length; i++) {
			int gid = getMapping(stOffset, 'a' + possiblePadding[i]);
			if (gid != 0) {
				return possiblePadding[i]; }}

		int code = getLowestMappedCode(stOffset);
		if (code >= 0xF000 && code < 0xF300)
			return code & 0xFF00;
		return 0;
	}

	/** Return the offset of a give cmap subtable if present, -1 otherwise. */
	public int probe (int platformID, int encodingID) 
	throws InvalidFontException, UnsupportedFontException
	{

		int numTables = this.data.getuint16 (2);

		for (int i = 0; i < numTables; i++) {
			int p = this.data.getuint16 (4 + 8*i + 0);
			int e = this.data.getuint16 (4 + 8*i + 2);
			if (p == platformID && e == encodingID) {
				return this.data.getuint32asint (4 + 8*i + 4, "Offset to cmap subtable is big"); }}

		return -1;
	}

	/** Tell whether the font is symbolic.
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	public boolean isSymbolic ()
	throws UnsupportedFontException, InvalidFontException
	{
		return symbolSubtableOffset != -1;
	}

	/**
	 * Enumerate the cmaps contained in the cmap table. Call
	 * the selector's cmapFound method for each recognized platform id/encoding id
	 * combination. 
	 * @param selector
	 */
	public void enumerateCmaps(CmapSelector selector)
	throws UnsupportedFontException, InvalidFontException 
	{
		int numTables = this.data.getuint16 (2);

		for (int i = 0; i < numTables; i++) {
			int p = this.data.getuint16 (4 + 8*i + 0);
			int e = this.data.getuint16 (4 + 8*i + 2);

			switch (p) {
			case PlatformID.MACINTOSH: {
				switch (e) {
				case MacEncodingID.ROMAN:
				case MacEncodingID.JAPANESE:
				case MacEncodingID.CHINESE_TRADITIONAL:
				case MacEncodingID.KOREAN:
				case MacEncodingID.HEBREW:
				case MacEncodingID.GREEK:
				case MacEncodingID.RUSSIAN:
				case MacEncodingID.RSYMBOL:
				case MacEncodingID.THAI:
				case MacEncodingID.CHINESE_SIMPLIFIED:
				case MacEncodingID.ARABIC: {
					selector.cmapFound(p, e, i);
					break; }}
				break; }

			case PlatformID.MICROSOFT: {
				switch (e) {
				case MS_EncodingID.UNICODE_BMP:
				case MS_EncodingID.BIG5:
				case MS_EncodingID.JOHAB:
				case MS_EncodingID.PRC:
				case MS_EncodingID.SHIFTJIS:
				case MS_EncodingID.SYMBOL:
				case MS_EncodingID.UNICODE_FULL:
				case MS_EncodingID.WANSUNG: {
					selector.cmapFound (p,e, i);
					break; }}
				break; }

			case PlatformID.UNICODE: {
				switch (e) {
				case UnicodeEncodingID.ID_10646_1993:
				case UnicodeEncodingID.ID_1_0:
				case UnicodeEncodingID.ID_1_1:
				case UnicodeEncodingID.ID_2_0_BMP:
				case UnicodeEncodingID.ID_2_0_FULL: 
				case UnicodeEncodingID.UNI_VARIATION_SEQ: {
					selector.cmapFound (p,e, i);
					break; }}
				break; }}}
	}

	public int getCmapSubtableIndex (int platformId, int encodingId)
	throws InvalidFontException 
	{
		int numTables = this.data.getuint16 (2);
		for (int i = 0; i < numTables; i++) {
			int p = this.data.getuint16 (4 + 8*i + 0);
			int e = this.data.getuint16 (4 + 8*i + 2);
			if (p == platformId && e == encodingId) {
				return i; }}
		return -1;
	}

	/**
	 * Modify the input character so that it does not have the modifier necessary to go through the cmap.
	 * @throws InvalidFontException 
	 * @throws UnsupportedFontException 
	 */
	int removeSymbolModifier(int usv) 
	throws UnsupportedFontException, InvalidFontException
	{
		if (symbolSubtableOffset != -1) {  
			int topByte = usv & 0xff00;
			if (topByte == paddingByte) {
				return usv & 0xff; }}	  
		return usv;
	}

	/** Get the glyph mapped from a given character.
   @param usv 
   the Unicode scalar value to look up

   @return 
   the gid of the glyph mapped from <code>usv</code>, 0 if not mapped. Note
   that this is the true OpenType gid, even in the case of cid cff otf. It is
   NOT the cid.
	 */
	public int unicodeChar2glyph (int usv) 
	throws UnsupportedFontException, InvalidFontException 
	{
		if (symbolSubtableOffset != -1) {  
			int m = getMapping (symbolSubtableOffset, usv);
			if (m == 0 && 0x00 <= usv && usv <= 0xff) {
				m = getMapping (symbolSubtableOffset, paddingByte | usv); }
			return m; }

		if (unicodeSubtableOffset != -1) {
			// 0xffff is a "noncharacter" according to http://www.unicode.org/charts/PDF/UFFF0.pdf, 
			// yet several fonts incorrectly map it to a glyph when using 
			// format 4 cmaps (because format 4 requires
			// it to be present though allows it to be mapped to notdef). Every other
			// piece of software I can find (sfntdump, spot, cooltype, batik) explicitly
			// throws out this code point to avoid mapping it to a glyph. We'll do so too.
			if (usv == 0xffff)
				return 0;

			return getMapping (unicodeSubtableOffset, usv); }    

		if (nonUnicodeSubtableIndices != null) {
			return getNonUnicodeMapping (usv); }

		return 0;
	}

	/** Approximates CoolType's interpretation of cmaps.
	 * 
	 * This approximation is fairly incomplete: for example, 
	 * CoolType has some heuristics to interpret the cmaps of Win 3.1 ME fonts
	 * (which have an MS/Symbol cmap that follows some conventions and are marked
	 * with high order bits in OS/2.fsSelection).
	 */
	public int coolTypeUnicodeChar2glyph (int usv) 
	throws UnsupportedFontException, InvalidFontException
	{
		if (unicodeSubtableOffset != -1) {
			return getMapping (unicodeSubtableOffset, usv); }

		if (oldUnicodeSubtableOffset != -1) {
			// CoolType does not even both to convert from Unicode to 
			// Unicode 1.0/1.0 before looking up the cmap. On the other hand,
			// we do not use this computation in general, only a very small
			// set of charaters, to derive metrics from their glyphs, so
			// such a conversion is not necessary.
			return getMapping (oldUnicodeSubtableOffset, usv); }

		if (symbolSubtableOffset != -1) {  
			int m = getMapping (symbolSubtableOffset, usv);
			if (m == 0 && 0x00 <= usv && usv <= 0xff) {
				m = getMapping (symbolSubtableOffset, paddingByte | usv); }
			return m; }

		if (nonUnicodeSubtableIndices != null) {
			return getNonUnicodeMapping (usv); }

		return 0;
	}

	int getNonUnicodeMapping (int usv)
	throws UnsupportedFontException, InvalidFontException 
	{

		char[] c = new char [2];
		UTF16.append (c, 0, usv);
		CharBuffer cb = CharBuffer.wrap(c, 0, (c[1] == 0) ? 1 : 2);

		ByteBuffer bb = ByteBuffer.allocate(10);

		for (NonUnicodeCmap cmap : nonUnicodeSubtableIndices)
		{
			try {
				CharsetEncoder encoder = cmap.getCharsetEncoder();
				if (encoder == null) 
				{
					int gid = getMapping (cmap.subtableOffset,	usv);
					if (gid != 0) 
					{
						return gid; 
					}
				}
				else if (encoder.canEncode((char)usv)) 
				{
					CoderResult coderResult = CharsetUtil.encodeLoopNoExceptions(cb, bb, encoder);
					if (!coderResult.isError())
					{
						if (bb.position() == 1)
						{
							int cp = bb.get(0) & 0xff;
							return getMapping (cmap.subtableOffset, cp); 
						}
						else if (bb.position() == 2) 
						{
							int cp = ((bb.get(0) & 0xff) << 8) | (bb.get(1) & 0xff); 
							return getMapping (cmap.subtableOffset, cp); 
						}
					} else {
						// ignore errors
					}
				}
			} catch (UnsupportedCharsetException e) {
				/* just ignore this subtable */ 
			} catch (IllegalCharsetNameException e) {
				/* just ignore this subtable */ 
			}
		}

		return 0;
	}

	public int char2glyph (int ch, int index)
	throws UnsupportedFontException, InvalidFontException 
	{
		int o = this.data.getuint32asint (4 + 8*index + 4, "Offset to cmap subtable is big");
		return getMapping (o, ch); 
	}

	/**
	 * Get the character code that is the lowest that is mapped by the cmap found at
	 * stOffset. 
	 */
	public int getLowestMappedCode (int stOffset) 
	throws UnsupportedFontException, InvalidFontException 
	{
		int format = this.data.getuint16 (stOffset);
		switch (format) {
		case 0: return getLowestMappedCodeFormat0 (stOffset);
		case 2: return getLowestMappedCodeFormat2 (stOffset);
		case 4: return getLowestMappedCodeFormat4 (stOffset);
		case 6: return getLowestMappedCodeFormat6 (stOffset);
		case 8: return getLowestMappedCodeFormat8 (stOffset);
		case 10: return getLowestMappedCodeFormat10 (stOffset);
		case 12: return getLowestMappedCodeFormat12 (stOffset);
		default: throw new UnsupportedFontException("cmap subtable format " + format); }
	}

	/**
	 * Get the character code that is the highest that is mapped by the cmap found at
	 * stOffset. If max2bytes is true, the highest code that is at most 2 bytes is returned.
	 */
	public int getHighestMappedCode (int stOffset, boolean max2bytes) 
	throws UnsupportedFontException, InvalidFontException 
	{
		int format = this.data.getuint16 (stOffset);
		switch (format) {
		case 0: return getHighestMappedCodeFormat0 (stOffset);
		case 2: return getHighestMappedCodeFormat2 (stOffset);
		case 4: return getHighestMappedCodeFormat4 (stOffset);
		case 6: return getHighestMappedCodeFormat6 (stOffset);
		case 8: return getHighestMappedCodeFormat8 (stOffset, max2bytes);
		case 10: return getHighestMappedCodeFormat10 (stOffset, max2bytes);
		case 12: return getHighestMappedCodeFormat12 (stOffset, max2bytes);
		default: throw new UnsupportedFontException("cmap subtable format " + format); }
	}

	/** Get the glyphID for charCode using the subtable at stOffset.
	 * @param stOffset the offset (in the cmap table) of the subtable to use
	 * @param charCode the character code to map
	 * @return the glyphID for <code>charCode</code>
	 **/
	public int getMapping (int stOffset, int charCode) 
	throws UnsupportedFontException, InvalidFontException 
	{
		int format = this.data.getuint16 (stOffset);
		switch (format) {
		case 0: return getMappingFormat0 (stOffset, charCode);
		case 2: return getMappingFormat2 (stOffset, charCode);
		case 4: return getMappingFormat4 (stOffset, charCode);
		case 6: return getMappingFormat6 (stOffset, charCode);
		case 8: return getMappingFormat8 (stOffset, charCode);
		case 10: return getMappingFormat10 (stOffset, charCode);
		case 12: return getMappingFormat12 (stOffset, charCode);
		default: throw new UnsupportedFontException 
		("cmap subtable format " + format); }
	}

	public int getOffset(int subtableIndex) throws InvalidFontException, UnsupportedFontException
	{
		return data.getuint32asint(4 + subtableIndex*8 + 4, "Offset to cmap subtable is big");
	}

	/** Get the inverted mapping from glyphs to characters for one subtable.
	 * The subtable, identified by <code>index</code> must 
	 * be present.
	 * @param numGlyphs the number of glyphs in the font
	 * @param index the index of the cmap to invert
	 * @return  array of <code>numGlyphs<code> integers, mapping
	 * glyphIDs to characters; if a glyphs is not mapped by the 
	 * subtable, the value in this array is -1. 
	 */ 
	public int[] glyph2char (int numGlyphs, int index) 
	throws UnsupportedFontException, InvalidFontException
	{

		int[] map = new int [numGlyphs];
		for (int i = 0; i < numGlyphs; i++) {
			map [i] = -1; }

		int o = this.data.getuint32asint (4 + 8*index + 4, "Offset to cmap subtable is big");

		int format = this.data.getuint16 (o);
		switch (format) {
		case 0: getInvertedMappingFormat0 (map, o); break;
		case 2: getInvertedMappingFormat2 (map, o); break;
		case 4: getInvertedMappingFormat4 (map, o); break;
		case 6: getInvertedMappingFormat6 (map, o); break;
		case 8: getInvertedMappingFormat8 (map, o); break;
		case 10: getInvertedMappingFormat10 (map, o); break;
		case 12: getInvertedMappingFormat12 (map, o); break;
		default: throw new UnsupportedFontException 
		("cmap subtable format " + format); }
		return map; 
	}

	/** Get the lowest code mapped by a format 0 cmap at stOffset*/
	protected int getLowestMappedCodeFormat0(int stOffset) 
	throws InvalidFontException 
	{
		int nbCodes = this.data.getuint16 (stOffset + 2) - 6 - 1;
		for (int currentCode = 0; currentCode < nbCodes; currentCode++)
		{
			int glyphid = this.data.getuint8 (stOffset + 6 + currentCode);
			if (glyphid > 0)
				return currentCode;
		}
		return nbCodes;
	}

	/** Get the highest code mapped by a format 0 cmap at stOffset*/
	protected int getHighestMappedCodeFormat0(int stOffset)
	throws InvalidFontException
	{
		int nbCodes = this.data.getuint16 (stOffset + 2) - 6 - 1;
		for (;nbCodes >= 0; nbCodes--)
		{
			int glyphid = this.data.getuint8 (stOffset + 6 + nbCodes);
			if (glyphid > 0)
				return nbCodes;
		}
		return 0;
	}

	/** Get the glyphID for charCode using the cmap subtable
	 *  in format 0 at stOffset.*/
	protected int getMappingFormat0 (int stOffset, int charCode)
	throws InvalidFontException 
	{
		// The size is apriori redundant and should always be equal to 6 + 256;
		// however, we guard against font compilers which generate smaller
		// tables (with the missing entries implicitly equal to 0) and indicate 
		// they do so by having a smaller size field

		int nbCodes = this.data.getuint16 (stOffset + 2) - 6;
		if (0 <= charCode && charCode < nbCodes) {
			return this.data.getuint8 (stOffset + 6 + charCode); }
		else {
			return 0; }
	}

	/** Invert the cmap subtable in format 0 at stOffset.
	 */
	protected void getInvertedMappingFormat0 (int[] map, int stOffset)
	throws InvalidFontException 
	{
		int nbCodes = this.data.getuint16 (stOffset + 2) - 6;
		for (int charCode = 0; charCode < nbCodes; charCode++) {
			int gid = this.data.getuint8 (stOffset + 6 + charCode); 
			map [gid] = charCode; }
	}

	/** Get the lowest code mapped by a format 2 cmap at stOffset*/
	protected int getLowestMappedCodeFormat2(int stOffset) 
	throws InvalidFontException
	{
		int key;
		for (key=0; key <= 255; key++) {
			int subHeaderKey = this.data.getuint16 (stOffset + 6 + 2*key) / 8;
			int subHeaderOffset = stOffset + 518 + 8*subHeaderKey;
			int firstCode = this.data.getuint16 (subHeaderOffset);
			int lastCode = this.data.getuint16 (subHeaderOffset + 2) + firstCode - 1;

			if (firstCode > key || key > lastCode)
				continue;

			int idDelta = this.data.getuint16 (subHeaderOffset + 4);
			int idRange = this.data.getuint16 (subHeaderOffset + 6);
			int glyphIndexOffset = subHeaderOffset + 6 + idRange;
			int g = this.data.getuint16 (glyphIndexOffset 
					+ 2 * (key - firstCode));

			if (g != 0) {
				g = (g + idDelta) % 0x10000;
				if (g != 0) {
					return key; }}}


		for (key=1; key <= 255; key++) {
			int subHeaderKey = this.data.getuint16 (stOffset + 6 + 2*key) / 8;
			int subHeaderOffset = stOffset + 518 + 8*subHeaderKey;
			int currentCode;
			int firstCode = currentCode = this.data.getuint16 (subHeaderOffset);
			int lastCode = this.data.getuint16 (subHeaderOffset + 2) + currentCode - 1;

			for (; currentCode <= lastCode; currentCode++) {
				int idDelta = this.data.getuint16 (subHeaderOffset + 4);
				int idRange = this.data.getuint16 (subHeaderOffset + 6);
				int glyphIndexOffset = subHeaderOffset + 6 + idRange;
				int g = this.data.getuint16 (glyphIndexOffset 
						+ 2 * (currentCode - firstCode));

				if (g != 0) {
					g = (g + idDelta) % 0x10000;
					if (g != 0) {
						return key << 8 | currentCode; }}}}

		return 0;
	}

	/** Get the highest code mapped by a format 2 cmap at stOffset*/
	protected int getHighestMappedCodeFormat2(int stOffset) 
	throws InvalidFontException
	{
		int key;
		for (key=255; key > 0; key--) {
			int subHeaderKey = this.data.getuint16 (stOffset + 6 + 2*key) / 8;
			int subHeaderOffset = stOffset + 518 + 8*subHeaderKey;
			int firstCode = this.data.getuint16 (subHeaderOffset);
			int currentCode = this.data.getuint16 (subHeaderOffset + 2) + firstCode - 1;

			for (; firstCode <= currentCode; currentCode--) {
				int idDelta = this.data.getuint16 (subHeaderOffset + 4);
				int idRange = this.data.getuint16 (subHeaderOffset + 6);
				int glyphIndexOffset = subHeaderOffset + 6 + idRange;
				int g = this.data.getuint16 (glyphIndexOffset 
						+ 2 * (currentCode - firstCode));

				if (g != 0) {
					g = (g + idDelta) % 0x10000;
					if (g != 0) {
						return key << 8 | currentCode; }}}}


		for (key=255; key >= 0; key--) {
			int subHeaderKey = this.data.getuint16 (stOffset + 6 + 2*key) / 8;
			int subHeaderOffset = stOffset + 518 + 8*subHeaderKey;
			int firstCode = this.data.getuint16 (subHeaderOffset);
			int lastCode = this.data.getuint16 (subHeaderOffset + 2) + firstCode - 1;

			if (firstCode > key || key > lastCode)
				continue;

			int idDelta = this.data.getuint16 (subHeaderOffset + 4);
			int idRange = this.data.getuint16 (subHeaderOffset + 6);
			int glyphIndexOffset = subHeaderOffset + 6 + idRange;
			int g = this.data.getuint16 (glyphIndexOffset 
					+ 2 * (key - firstCode));

			if (g != 0) {
				g = (g + idDelta) % 0x10000;
				if (g != 0) {
					return key; }}}

		return 0;
	}

	/** Get the glyphID for charCode using the cmap subtable
	 *  in format 2 at stOffset.*/
	/** Get the glyphID for charCode using the cmap subtable
	 *  in format 2 at stOffset.*/
	protected int getMappingFormat2 (int stOffset, int charCode)
	throws InvalidFontException 
	{
		int headerIndexByte;
		int glyphIndexArrayIndexByte;

		{ int highByte = (charCode >> 8) & 0xff;
		int lowByte = (charCode) & 0xff;

		// From AOTS:
		//  <para>This cmap subtable format is really supporting two
		//  operations: find the bytes that represent a character in the
		//  input stream and convert those to a glyphID. We do not really
		//  want to support the first part in our interface: the client
		//  must already determine the boundaries of characters, and
		//  assemble the bytes in code points.</para>
		//  
		//  <para>This creates a little difficulty: consider the case
		//  where we get at our interface a code point
		//  <emphasis>c</emphasis> between 0x00 and 0xff. Was it the
		//  result of a single byte <emphasis>c</emphasis> (in which case
		//  we should index in subHeaderKeys with that value) or was it
		//  the result of a two byte sequence 0x00 <emphasis>c</emphasis>
		//  (in which case we should index subHeaderKeys with 0)?
		//  Fortunately, we can inspect subHeaderKeys[c] to distinguish
		//  the cases. The situation does not happen for code points above
		//  0xff, as those are by necessity on two bytes.</para>

		//  <para>It is worth noting that the confusion is unlikely in
		//  practice, as the byte 0x00 is representing entirely a
		//  character in essentially all one/two bytes encodings. However,
		//  the is is not implied by the cmap subtable format, so we need
		//  to handle the situation.</para>

		if (highByte == 0 && this.data.getuint16 (stOffset + 6 + 2*lowByte) == 0) {
			// lowByte is a single byte character
			headerIndexByte = lowByte; }
		else {
			headerIndexByte = highByte; }
		glyphIndexArrayIndexByte = lowByte; }

		int subHeaderKey = this.data.getuint16 (stOffset + 6 + 2*headerIndexByte) / 8;
		int subHeaderOffset = stOffset + 518 + 8*subHeaderKey;
		int firstCode = this.data.getuint16 (subHeaderOffset);
		int entryCount = this.data.getuint16 (subHeaderOffset + 2);

		if (   glyphIndexArrayIndexByte < firstCode 
				|| firstCode + entryCount <= glyphIndexArrayIndexByte) {
			return 0; }

		int idDelta = this.data.getuint16 (subHeaderOffset + 4);
		int idRange = this.data.getuint16 (subHeaderOffset + 6);
		int glyphIndexOffset = subHeaderOffset + 6 + idRange;
		int g = this.data.getuint16 (glyphIndexOffset 
				+ 2 * (glyphIndexArrayIndexByte - firstCode));

		if (g == 0) {
			return 0; }
		else {
			return (g + idDelta) % 0x10000; }
	}

	/** Invert the cmap subtable in format 2 at stOffset.
	 */
	protected void getInvertedMappingFormat2 (int[] map, int stOffset)
	throws InvalidFontException 
	{
		for (int firstByte = 0; firstByte <= 0xff; firstByte++) {
			int subHeaderIndex = this.data.getuint16 (stOffset + 6 + 2*firstByte) / 8;
			int subHeaderOffset = stOffset + 518 + subHeaderIndex * 8;
			int firstCode = this.data.getuint16 (subHeaderOffset);
			int entryCount = this.data.getuint16 (subHeaderOffset + 2);
			int idDelta = this.data.getuint16 (subHeaderOffset + 4);
			int idRange = this.data.getuint16 (subHeaderOffset + 6);
			int glyphIndexOffset = subHeaderOffset + 6 + idRange;

			if (subHeaderIndex == 0) {
				int g = this.data.getuint16 (glyphIndexOffset + 2 * (firstByte - firstCode));
				if (g != 0) {
					map [(g + idDelta) % 0x10000] = firstByte; }}

			else {
				for (int secondByte = firstCode; 
				secondByte < firstCode + entryCount; 
				secondByte++) {
					int g = this.data.getuint16 (glyphIndexOffset + 2 * (secondByte - firstCode));
					if (g != 0) {
						map [(g + idDelta) % 0x10000] = (firstByte << 8) + secondByte; }}}}
	}

	/** Get the lowest code mapped by a format 4 cmap at stOffset*/
	protected int getLowestMappedCodeFormat4(int stOffset)
	throws InvalidFontException 
	{
		int segCount = this.data.getuint16 (stOffset + 6) / 2;
		int currentGroup = segCount - 1;

		for (currentGroup = 0; currentGroup <= segCount - 1; currentGroup++) {
			try {
				int currentCode;
				int firstCode = currentCode = this.data.getuint16 (stOffset + 16 + 2*segCount + 2*currentGroup);
				int lastCode = this.data.getuint16 (stOffset + 14 + 2*currentGroup);

				for (; currentCode <= lastCode; currentCode++) {
					try {
						int idRangeOffset = this.data.getuint16 (stOffset + 16 + 6*segCount + 2*currentGroup);
						int idDelta = this.data.getint16 (stOffset + 16 + 4*segCount + 2*currentGroup);

						if (idRangeOffset == 0) {
							int gid = (currentCode + idDelta) & 0xffff;
							if (gid != 0) {
								return currentCode; }}

						int offset = stOffset + 16 + 6*segCount + 2*currentGroup 
						+ idRangeOffset 
						+ 2*(currentCode - firstCode);
						int glyphIdArrayValue = this.data.getuint16 (offset);

						if (glyphIdArrayValue != 0) {	
							int gid = (glyphIdArrayValue + idDelta) & 0xffff;
							if (gid != 0) {
								return currentCode; }}}
					catch (InvalidFontException e){}}}
			catch(InvalidFontException e) {}}


		return 0; 
	}

	/** Get the highest code mapped by a format 4 cmap at stOffset*/
	protected int getHighestMappedCodeFormat4(int stOffset)
	throws InvalidFontException 
	{
		int segCount = this.data.getuint16 (stOffset + 6) / 2;
		int currentGroup = segCount - 1;

		for (;0 <= currentGroup; currentGroup--) {
			try {
				int startCode = this.data.getuint16 (stOffset + 16 + 2*segCount + 2*currentGroup);
				int currentCode = this.data.getuint16 (stOffset + 14 + 2*currentGroup);

				for (; currentCode >= startCode; currentCode--) {
					try {
						int idRangeOffset = this.data.getuint16 (stOffset + 16 + 6*segCount + 2*currentGroup);
						int idDelta = this.data.getint16 (stOffset + 16 + 4*segCount + 2*currentGroup);

						if (idRangeOffset == 0) {
							int gid = (currentCode + idDelta) & 0xffff;
							if (gid != 0) {
								return currentCode; }}

						int offset = stOffset + 16 + 6*segCount + 2*currentGroup 
						+ idRangeOffset 
						+ 2*(currentCode - startCode);
						int glyphIdArrayValue = this.data.getuint16 (offset);

						if (glyphIdArrayValue != 0) {	
							int gid = (glyphIdArrayValue + idDelta) & 0xffff;
							if (gid != 0) {
								return currentCode; }}}
					catch (InvalidFontException e){}}}
			catch(InvalidFontException e) {}}


		return 0; 
	}

	/** Get the glyphID for charCode using the cmap subtable
	 *  in format 4 at stOffset.*/
	protected int getMappingFormat4 (int stOffset, int charCode) 
	{
		try {
			int segCount = this.data.getuint16 (stOffset + 6) / 2;
			int min = 0;
			int max = segCount - 1;

			while (min <= max) {
				int s = (min + max) / 2;
				int startCount = this.data.getuint16 (stOffset + 16 + 2*segCount + 2*s);
				int endCount = this.data.getuint16 (stOffset + 14 + 2*s);

				if (charCode < startCount) {
					max = s - 1; }
				else if (endCount < charCode) {
					min = s + 1; }
				else {
					int idRangeOffset = this.data.getuint16 (stOffset + 16 + 6*segCount + 2*s);
					int idDelta = this.data.getint16 (stOffset + 16 + 4*segCount + 2*s);

					if (idRangeOffset == 0) {
						return (charCode + idDelta) & 0xffff; }

					int offset = stOffset + 16 + 6*segCount + 2*s 
					+ idRangeOffset 
					+ 2*(charCode - startCount);
					int glyphIdArrayValue = this.data.getuint16 (offset);

					if (glyphIdArrayValue == 0) {
						return 0; }

					return (glyphIdArrayValue + idDelta) & 0xffff; }}

			return 0;
		} catch (InvalidFontException e) {
			// If we go past the end of the table, consider the glyph unmapped.
			return 0;  } 
	}

	/** Invert the cmap subtable in format 4 at stOffset.
	 */
	protected void getInvertedMappingFormat4 (int[] map, int stOffset)
	throws InvalidFontException 
	{
		int segCount = this.data.getuint16 (stOffset + 6) / 2;
		for (int s = 0; s < segCount; s++) {

			int startCount = this.data.getuint16 (stOffset + 16 + 2*segCount + 2*s);
			int endCount = this.data.getuint16 (stOffset + 14 + 2*s);
			int idRangeOffsetOffset = stOffset + 16 + 6*segCount + 2*s;
			int idRangeOffset = this.data.getuint16 (idRangeOffsetOffset);
			int idDelta = this.data.getint16 (stOffset + 16 + 4*segCount + 2*s);

			for (int c = startCount; c <= endCount; c++) {
				if (idRangeOffset == 0) {
					int glyphID = (c + idDelta) & 0xffff;
					if (glyphID != 0 && glyphID < map.length) {
						map [glyphID] = c; }}
				else {
					int offset = stOffset + 16 + 6*segCount + 2*s 
					+ idRangeOffset 
					+ 2*(c - startCount);
					int glyphIdArrayValue = this.data.getuint16 (offset);

					if (glyphIdArrayValue != 0) {
						int glyphID = (glyphIdArrayValue + idDelta) & 0xffff;
						if (glyphID != 0 && glyphID < map.length) {
							map [glyphID] = c; }}}}}
	}

	/** Get the lowest code mapped by a format 6 cmap at stOffset*/
	protected int getLowestMappedCodeFormat6(int stOffset) 
	throws InvalidFontException
	{
		int currentCode;
		int firstCode = currentCode = this.data.getuint16 (stOffset + 6);
		int lastCode = this.data.getuint16 (stOffset + 8) + currentCode - 1;

		for (;currentCode <= lastCode; currentCode++) {
			int gid = this.data.getuint16 (stOffset + 10 + (currentCode-firstCode)*2);
			if (gid != 0){
				return currentCode;}}

		return 0;
	}

	/** Get the highest code mapped by a format 6 cmap at stOffset*/
	protected int getHighestMappedCodeFormat6(int stOffset) 
	throws InvalidFontException
	{
		int firstCode = this.data.getuint16 (stOffset + 6);
		int currentCode = this.data.getuint16 (stOffset + 8) + firstCode - 1;

		for (;firstCode <= currentCode; currentCode--) {
			int gid = this.data.getuint16 (stOffset + 10 + (currentCode-firstCode)*2);
			if (gid != 0){
				return currentCode;}}

		return 0;
	}

	/** Get the glyphID for charCode using the cmap subtable
	 *  in format 6 at stOffset.*/
	protected int getMappingFormat6 (int stOffset, int charCode)
	throws InvalidFontException 
	{
		int firstCode = this.data.getuint16 (stOffset + 6);
		int entryCount = this.data.getuint16 (stOffset + 8);

		if (firstCode <= charCode && charCode < firstCode + entryCount) {
			return this.data.getuint16 (stOffset + 10 + (charCode-firstCode)*2); }
		else {
			return 0; }
	}

	/** Invert the cmap subtable in format 6 at stOffset.
	 */
	protected void getInvertedMappingFormat6 (int[] map, int stOffset)
	throws InvalidFontException 
	{
		int firstCode = this.data.getuint16 (stOffset + 6);
		int entryCount = this.data.getuint16 (stOffset + 8);

		for (int i = 0; i < entryCount; i++) {
			int code = firstCode + i;
			map [this.data.getuint16 (stOffset + 10 + i*2)] = code; }
	}

	/** Get the lowest code mapped by a format 8 cmap at stOffset*/
	protected int getLowestMappedCodeFormat8(int stOffset) 
	throws UnsupportedFontException, InvalidFontException
	{
		int nGroups = this.data.getuint32asint (stOffset + 8204, 
		"cmap subtable, format 8, nGroups is big");
		int currentGroup;
		int lastCode = 0xffff;

		for (currentGroup = 0; currentGroup < nGroups; currentGroup++) {
			int currentCode;
			int firstCode = this.data.getuint32asint (stOffset + 8208 + 12*currentGroup,
			"cmap subtable, format 8, startCharCode is big");
			lastCode = this.data.getuint32asint (stOffset + 8208 + 12*currentGroup + 4,
			"cmap subtable, format 8, endCharCode is big");

			for (currentCode = firstCode;currentCode <= lastCode; currentCode++){
				int gid = this.data.getuint32asint (stOffset + 8208 + 12*currentGroup + 8, 
				"cmap subtable, format 8, startGlyphID is big") + currentCode - firstCode;
				if (gid != 0 && (currentCode <= 0xffff))
					return currentCode;}}

		return lastCode; 
	}

	/** Get the highest code mapped by a format 8 cmap at stOffset*/
	protected int getHighestMappedCodeFormat8(int stOffset, boolean max2bytes) 
	throws UnsupportedFontException, InvalidFontException 
	{
		int nGroups = this.data.getuint32asint (stOffset + 8204, 
		"cmap subtable, format 8, nGroups is big");
		int currentGroup = nGroups - 1;

		for (; 0 <= currentGroup; currentGroup--) {
			int startCharCode = this.data.getuint32asint (stOffset + 8208 + 12*currentGroup,
			"cmap subtable, format 8, startCharCode is big");
			int currentCharCode = this.data.getuint32asint (stOffset + 8208 + 12*currentGroup + 4,
			"cmap subtable, format 8, endCharCode is big");

			for (;currentCharCode >= startCharCode; currentCharCode--){
				int gid = this.data.getuint32asint (stOffset + 8208 + 12*currentGroup + 8, 
				"cmap subtable, format 8, startGlyphID is big") + currentCharCode - startCharCode;
				if (gid != 0 && (!max2bytes || currentCharCode <= 0xffff))
					return currentCharCode;}}

		return 0; 
	}

	/** Get the glyphID for charCode using the cmap subtable
	 *  in format 8 at stOffset.*/
	protected int getMappingFormat8 (int stOffset, int charCode)
	throws UnsupportedFontException, InvalidFontException 
	{
		int nGroups = this.data.getuint32asint (stOffset + 8204, 
		"cmap subtable, format 8, nGroups is big");
		int min = 0;
		int max = nGroups - 1;

		while (min <= max) {
			int s = (min + max) / 2;
			int startCharCode = this.data.getuint32asint (stOffset + 8208 + 12*s,
			"cmap subtable, format 8, startCharCode is big");
			int endCharCode = this.data.getuint32asint (stOffset + 8208 + 12*s + 4,
			"cmap subtable, format 8, endCharCode is big");

			if (charCode < startCharCode) {
				max = s - 1; }
			else if (endCharCode < charCode) {
				min = s + 1; }
			else {
				int startGlyphID = this.data.getuint32asint (stOffset + 8208 + 12*s + 8, 
				"cmap subtable, format 8, startGlyphID is big");
				return charCode - startCharCode + startGlyphID; }}

		return 0;
	}

	/** Invert the cmap subtable in format 8 at stOffset.
	 */
	protected void getInvertedMappingFormat8 (int[] map, int stOffset)
	throws UnsupportedFontException, InvalidFontException
	{
		int nGroups = this.data.getuint32asint (stOffset + 8204, 
		"cmap subtable, format 8, nGroups is big");

		for (int g = 0; g < nGroups; g++) {
			int startCharCode = this.data.getuint32asint (stOffset + 8208 + 12*g,
			"cmap subtable, format 8, startCharCode is big");
			int endCharCode = this.data.getuint32asint (stOffset + 8208 + 12*g + 4,
			"cmap subtable, format 8, endCharCode is big");
			int startGlyphId = this.data.getuint32asint (stOffset + 8208 + 12*g + 8, 
			"cmap subtable, format 8, startGlyphID is big");

			for (int c = startCharCode; c <= endCharCode; c++) {
				map [startGlyphId + c - startCharCode] = c; }}
	}

	/** Get the lowest code mapped by a format 10 cmap at stOffset*/
	protected int getLowestMappedCodeFormat10(int stOffset) 
	throws InvalidFontException, UnsupportedFontException 
	{
		int firstCode = this.data.getuint32asint (stOffset + 12,
		"cmap subtable, format 10, startCharCode is big");
		int lastCode = this.data.getuint32asint (stOffset + 16, 
		"cmap subtable, format 10, numChars is big") + firstCode - 1;

		for (int currentCode = firstCode; currentCode <= lastCode; currentCode++) {
			int glyphID = this.data.getuint16 (stOffset + 20 + 2 * (currentCode - firstCode));
			if (glyphID > 0){
				return currentCode; }}

		return lastCode; 
	}

	/** Get the highest code mapped by a format 10 cmap at stOffset*/
	protected int getHighestMappedCodeFormat10(int stOffset, boolean max2bytes) 
	throws InvalidFontException, UnsupportedFontException 
	{
		int startCharCode = this.data.getuint32asint (stOffset + 12,
		"cmap subtable, format 10, startCharCode is big");
		int numChars = this.data.getuint32asint (stOffset + 16, 
		"cmap subtable, format 10, numChars is big");
		int charCode = startCharCode + numChars - 1;
		for (;startCharCode <= charCode; charCode--) {
			int glyphID = this.data.getuint16 (stOffset + 20 + 2 * (charCode - startCharCode));
			if (glyphID > 0 && (!max2bytes || charCode <= 0xffff)){
				return charCode; }}

		return 0; 
	}

	/** Get the glyphID for charCode using the cmap subtable
	 *  in format 10 at stOffset.*/
	protected int getMappingFormat10 (int stOffset, int charCode) 
	throws UnsupportedFontException, InvalidFontException 
	{
		int startCharCode = this.data.getuint32asint (stOffset + 12,
		"cmap subtable, format 10, startCharCode is big");
		int numChars = this.data.getuint32asint (stOffset + 16, 
		"cmap subtable, format 10, numChars is big");
		if (startCharCode <= charCode && charCode < startCharCode + numChars) {
			return this.data.getuint16 (stOffset + 20 + 2 * (charCode - startCharCode)); }

		return 0;
	}

	/** Invert the cmap subtable in format 10 at stOffset.
	 */
	protected void getInvertedMappingFormat10 (int[] map, int stOffset) 
	throws UnsupportedFontException, InvalidFontException 
	{
		int startCharCode = this.data.getuint32asint (stOffset + 12,
		"cmap subtable, format 10, startCharCode is big");
		int numChars = this.data.getuint32asint (stOffset + 16, 
		"cmap subtable, format 10, numChars is big");
		for (int c = startCharCode; c < startCharCode + numChars; c++) {
			map [this.data.getuint16 (stOffset + 20 + 2 * (c - startCharCode))] = c; }

	}

	/** Get the lowest code mapped by a format 12 cmap at stOffset*/
	protected int getLowestMappedCodeFormat12(int stOffset) 
	throws InvalidFontException, UnsupportedFontException 
	{
		int nGroups = (int)this.data.getuint32 (stOffset + 12);
		int lastCode = 0xffff;

		for (int currentGroup = 0; currentGroup < nGroups; currentGroup++) {
			int firstCode = (int) this.data.getuint32 (stOffset + 16 + 12*currentGroup);
			lastCode = (int) this.data.getuint32 (stOffset + 16 + 12*currentGroup + 4);

			for (int currentCode = firstCode; currentCode <= lastCode; currentCode++) {
				int startGlyphID = (int) this.data.getuint32 (stOffset + 16 + 12*currentGroup + 8);
				startGlyphID = (startGlyphID + (currentCode - firstCode));  
				if (startGlyphID > 0) { 
					return currentCode;}}}
		return lastCode;  
	}

	/** Get the highest code mapped by a format 12 cmap at stOffset*/
	protected int getHighestMappedCodeFormat12(int stOffset, boolean max2bytes) 
	throws InvalidFontException, UnsupportedFontException 
	{
		int nGroups = (int)this.data.getuint32 (stOffset + 12);
		int currentGroup = nGroups - 1;
		for (;0 <= currentGroup; currentGroup--) {
			int startCharCode = (int) this.data.getuint32 (stOffset + 16 + 12*currentGroup);
			int currentCharCode = (int) this.data.getuint32 (stOffset + 16 + 12*currentGroup + 4);

			for (;currentCharCode >= startCharCode; currentCharCode--) {
				int startGlyphID = (int) this.data.getuint32 (stOffset + 16 + 12*currentGroup + 8);
				startGlyphID = (startGlyphID + (currentCharCode - startCharCode));  
				if (startGlyphID > 0 && (!max2bytes || currentCharCode <= 0xffff)) { 
					return currentCharCode;}}}
		return 0;  
	}

	/** Get the glyphID for charCode using the cmap subtable
	 *  in format 12 at stOffset.*/
	protected int getMappingFormat12 (int stOffset, int charCode)
	throws InvalidFontException
	{

		int nGroups = (int)this.data.getuint32 (stOffset + 12);
		int min = 0;
		int max = nGroups - 1;
		while (min <= max) {
			int s = (min + max) / 2;
			int startCharCode = (int) this.data.getuint32 (stOffset + 16 + 12*s);
			int endCharCode = (int) this.data.getuint32 (stOffset + 16 + 12*s + 4);

			if (charCode < startCharCode) {
				max = s - 1; }
			else if (endCharCode < charCode) {
				min = s + 1; }
			else {
				int startGlyphID = (int) this.data.getuint32 (stOffset + 16 + 12*s + 8);
				return (startGlyphID + (charCode - startCharCode)); }}
		return 0;
	}

	/** Invert the cmap subtable in format 12 at stOffset.
	 */
	protected void getInvertedMappingFormat12 (int[] map, int stOffset)
	throws InvalidFontException 
	{
		int nGroups = (int)this.data.getuint32 (stOffset + 12);
		for (int g = 0; g < nGroups; g++) {
			int startCharCode = (int) this.data.getuint32 (stOffset + 16 + 12*g);
			int endCharCode = (int) this.data.getuint32 (stOffset + 16 + 12*g + 4);
			int startGlyphID = (int) this.data.getuint32 (stOffset + 16 + 12*g + 8);
			for (int c = startCharCode; c <= endCharCode; c++) {
				map [startGlyphID + (c - startCharCode)] = c; }}
	}

	/**
	 * In the SubsetSimpleTrueType case, the client will have provided a cmap
	 * and a set of codepoints to map against it. We will convert these to relate
	 * to the new subset GlyphIDs stored in the OTSubset. We don't bother to emit
	 * cmaps in any other case than the SubsetSimpleTrueType case.
	 */
	public void subsetAndStream(Subset subset, SubsetSimpleTrueType ttSubset, Map tables)
	throws UnsupportedFontException, InvalidFontException
	{
		if (ttSubset == null)
			return;
		int platformID = ttSubset.getPlatformID();
		int platSpecID = ttSubset.getPlatformSpecificID();
		int offset = probe(platformID, platSpecID);
		if (offset < 0)
			throw new UnsupportedFontException("Unsupported cmap ID");
		int format = data.getuint16(offset);
		if (format != 0 && format != 4 && format != 12)
			throw new UnsupportedFontException("Unsupported cmap format");
		if (format == 12)
			format = 4;
		ArrayList mapList = new ArrayList();
		int index = offsetToIndex(offset);
		int[] cps = ttSubset.getCodePoints();
		for (int i = 0; i < cps.length; i++) {
			int cp = cps[i];
			int gid = char2glyph(cp, index);
			int subsetGid = subset.getExistingSubsetGid(gid);
			if (subsetGid <= 0)
				throw new InvalidFontException("Subset does not contain required codepoint");
			MapElement element = new MapElement(cp, subsetGid);
			mapList.add(element);
		}
		if (mapList.size() == 0)
			return;
		MapElement lastElem = (MapElement)mapList.get(0);
		if (lastElem.mCodePoint > 0xFFFF)
			format = 12;
		for (int i = 1; i < mapList.size();) {
			MapElement thisElem = (MapElement)mapList.get(i);
			if (thisElem.mCodePoint == lastElem.mCodePoint) {
				mapList.remove(i);
			} else {
				if (thisElem.mCodePoint > 0xFFFF)
					format = 12;
				lastElem = thisElem;
				i++;
			}
		}
		OTByteArrayBuilder cmapData = OTByteArray.getOTByteArrayBuilderInstance(Math.max(1024, mapList.size()*32));
		cmapData.setuint16(0, 0);							// Format number, always zero
		cmapData.setuint16(2, 1);							// Number of cmaps, always one
		int curOffset = 4;								// Base of per-cmap header
		cmapData.setuint16(curOffset, platformID);
		cmapData.setuint16(curOffset+2, platSpecID);
		cmapData.setuint32(curOffset+4, curOffset+8);					// Actual cmap data begins here (offset 12)
		curOffset += 8;
		int baseOffset = curOffset;							// Base of per-cmap data
		ArrayList segList = new ArrayList();
		int anchor = 0;
		lastElem = (MapElement)mapList.get(anchor);
		int curEntry = 1;
		if (format == 0) {
			cmapData.setuint16(curOffset, 0);					// Subtable format (0)
			cmapData.setuint16(curOffset+2, 2 + 2 + 2 + 256);			// Size is constant
			cmapData.setuint16(curOffset+4, 0);					// Subtable version or language
			curOffset += 6;								// Base of character array
			cmapData.setuint8(curOffset + 255, 0);					// Set high-water mark
			for (int i = 0; i < mapList.size(); i++) {
				MapElement elem = (MapElement)mapList.get(i);
				cmapData.setuint8(curOffset + elem.mCodePoint, elem.mGlyphID);
			}
		} else if (format == 4) {
			int ordered = 0;
			if (((MapElement)mapList.get(mapList.size() - 1)).mCodePoint != 0xFFFF)
				mapList.add(new MapElement(0xFFFF, 0));
			while (curEntry < mapList.size()) {
				MapElement curElem = (MapElement)mapList.get(curEntry);
				if (curElem.mCodePoint == lastElem.mCodePoint + 1) {
					if (curElem.mGlyphID == lastElem.mGlyphID + 1) {
						if (++ordered > 3 && ordered < curEntry - anchor) {
							segList.add(new MapSegment(anchor, curEntry - ordered, false));
							anchor = curEntry - ordered;
						}
					} else {
						if (ordered > 3) {
							segList.add(new MapSegment(anchor, curEntry, true));
							anchor = curEntry;
						}
						ordered = 0;
					}
				} else {
					segList.add(new MapSegment(anchor, curEntry, curEntry - anchor == ordered + 1));
					anchor = curEntry;
					ordered = 0;
				}
				lastElem = curElem;
				curEntry++;
			}
			segList.add(new MapSegment(anchor, curEntry, curEntry - anchor == ordered + 1));
			cmapData.setuint16(curOffset, 4);					// Subtable format (4)
			int nSegs = segList.size();						// Update total size later
			cmapData.setuint16(curOffset+4, 0);					// Subtable version or language
			cmapData.setuint16(curOffset+6, nSegs*2);				// Segment count times two
			int nBits = 0;
			while (nSegs != 0) {
				nSegs >>= 1;
			nBits++;
			}
			nSegs = segList.size();
			cmapData.setuint16(curOffset+8, 1 << nBits);				// Search range
			cmapData.setuint16(curOffset+10, nBits - 1);				// Entry selector
			cmapData.setuint16(curOffset+12, nSegs*2 - (1 << nBits));		// Range shift
			curOffset += 14;
			for (index = 0; index < nSegs; index++) {
				MapSegment seg = (MapSegment)segList.get(index);
				MapElement elem = (MapElement)mapList.get(seg.mEndIndex - 1);
				cmapData.setuint16(curOffset + index*2, elem.mCodePoint);	// Segment end code point
			}
			curOffset += index*2;
			cmapData.setuint16(curOffset, 0);					// Padding, zero
			curOffset += 2;
			for (index = 0; index < nSegs; index++) {
				MapSegment seg = (MapSegment)segList.get(index);
				MapElement elem = (MapElement)mapList.get(seg.mFirstIndex);
				cmapData.setuint16(curOffset + index*2, elem.mCodePoint);	// Segment begin code point
			}
			curOffset += index*2;
			for (index = 0; index < nSegs; index++) {
				MapSegment seg = (MapSegment)segList.get(index);
				offset = 0;
				if (seg.mOrdered) {
					MapElement elem = (MapElement)mapList.get(seg.mFirstIndex);
					offset = elem.mGlyphID - (short)elem.mCodePoint;	// Segment GlyphID offset
				}
				cmapData.setuint16(curOffset + index*2, offset);
			}
			curOffset += index*2;
			int gidArrayBase = curOffset + index*2;
			int gidArrayIndex = 0;
			for (index = 0; index < nSegs; index++) {
				MapSegment seg = (MapSegment)segList.get(index);
				offset = 0;
				if (!seg.mOrdered) {
					offset = (nSegs - index + gidArrayIndex) * 2;
					for (int i = seg.mFirstIndex; i < seg.mEndIndex; i++) {
						MapElement elem = (MapElement)mapList.get(i);
						cmapData.setuint16(gidArrayBase + gidArrayIndex*2, elem.mGlyphID);
						gidArrayIndex++;
					}
				}
				cmapData.setuint16(curOffset + index*2, offset);
			}
			cmapData.setuint16(baseOffset+2, gidArrayBase + gidArrayIndex*2 - baseOffset);	// Set total table size
		} else {
			while (curEntry < mapList.size()) {
				MapElement curElem = (MapElement)mapList.get(curEntry);
				if (curElem.mCodePoint != lastElem.mCodePoint + 1 || curElem.mGlyphID != lastElem.mGlyphID + 1) {
					segList.add(new MapSegment(anchor, curEntry, true));
					anchor = curEntry;
				}
				lastElem = curElem;
				curEntry++;
			}
			segList.add(new MapSegment(anchor, curEntry, true));
			int nSegs = segList.size();
			cmapData.setuint16(curOffset, 12);					// Subtable format (12)
			cmapData.setuint16(curOffset+2, 0);					// Padding, zero
			cmapData.setuint32(curOffset+4, 16 + nSegs*12);				// Table size in bytes
			cmapData.setuint32(curOffset+8, 0);					// Subtable version or language
			cmapData.setuint32(curOffset+12, nSegs);				// Segment count
			curOffset += 16;
			for (index = 0; index < nSegs; index++) {
				MapSegment seg = (MapSegment)segList.get(index);
				MapElement firstElem = (MapElement)mapList.get(seg.mFirstIndex);
				MapElement endElem = (MapElement)mapList.get(seg.mEndIndex - 1);
				cmapData.setuint32(curOffset + index*12, firstElem.mCodePoint);		// Segment start code point
				cmapData.setuint32(curOffset + index*12 + 4, endElem.mCodePoint);	// Segment end code point
				cmapData.setuint32(curOffset + index*12 + 8, firstElem.mGlyphID);	// Segment start GlyphID
			}
		}
		tables.put(new Integer(Tag.table_cmap), cmapData);
	}

	public void subsetAndStreamForSWF(TreeSet codePoints, Subset subset, Map tables, boolean includeVariationCmap) 
	throws UnsupportedFontException, InvalidFontException
	{
		Iterator iter = codePoints.iterator();
		boolean bmpOnly = true;
		int numberOfSubtables = 1;
		OTByteArrayBuilder builder = OTByteArray.getOTByteArrayBuilderInstance();
		int firstSubtableSize = 0;

		if (includeVariationCmap)
			numberOfSubtables = 2;

		while (iter.hasNext())
		{
			if (((Integer)iter.next()).intValue() > 0xffff)
			{
				bmpOnly = false;
				break;
			}
		}

		builder.ensureCapacity(4 + 8*numberOfSubtables);
		builder.setuint16(0, 0);
		builder.setuint16(2, numberOfSubtables);
		int offsetToFirst = 4 + 8*numberOfSubtables;

		if (bmpOnly)
		{
			builder.setuint16(4, PlatformID.MICROSOFT);
			builder.setuint16(6, symbolSubtableOffset != -1 ? MS_EncodingID.SYMBOL : MS_EncodingID.UNICODE_BMP);
			builder.setuint32(8, offsetToFirst); // offset to subtable

			firstSubtableSize = writeFormat4Subtable(builder, offsetToFirst, codePoints, subset, paddingByte);		
		} else
		{
			builder.setuint16(4, PlatformID.MICROSOFT);
			builder.setuint16(6, MS_EncodingID.UNICODE_FULL);
			builder.setuint32(8, offsetToFirst); // offset to subtable

			firstSubtableSize = writeFormat12Subtable(builder, offsetToFirst, codePoints, subset);	
		}

		if (includeVariationCmap)
		{
			builder.setuint16(12, PlatformID.UNICODE);
			builder.setuint16(14, UnicodeEncodingID.UNI_VARIATION_SEQ);
			builder.setuint32(16, offsetToFirst + firstSubtableSize); // offset to subtable

			firstSubtableSize = writeFormat14Subtable(builder, offsetToFirst + firstSubtableSize, codePoints, subset,
					probe(PlatformID.UNICODE, UnicodeEncodingID.UNI_VARIATION_SEQ));	
		}

		tables.put(new Integer(Tag.table_cmap), builder);
	}

	private class SegmentInfo
	{
		boolean canUseIDDelta = true;
		int idDelta;
		int startCode;
		int endCode;
	}

	private int computeCodePointToEmit(int cp, int symbolPad)
	{
		if (symbolPad == 0)
			return cp;

		return (symbolPad | (cp&0xff));
	}

	private int writeFormat4Subtable(OTByteArrayBuilder builder, int offset, TreeSet codepoints, Subset subset, int symbolPad) 
	throws UnsupportedFontException, InvalidFontException
	{
		Iterator iter = codepoints.iterator();
		int lastCP = 0;
		List /*<SegmentInfo>*/ segmentList = new ArrayList();
		SegmentInfo currentSegment = null;
		boolean addTerminalSegment = true;

		if (iter.hasNext())
		{
			lastCP = ((Integer)iter.next()).intValue();
			currentSegment = new SegmentInfo();
			currentSegment.startCode = lastCP;
			int thisGid = subset.getExistingSubsetGid(unicodeChar2glyph(lastCP));
			currentSegment.idDelta = thisGid - computeCodePointToEmit(lastCP, symbolPad);
			segmentList.add(currentSegment);

			if (lastCP == 0xffff)
				addTerminalSegment = false;

		}

		while (iter.hasNext())
		{
			int cp = ((Integer)iter.next()).intValue();
			if (cp == 0xffff)
				addTerminalSegment = false;

			if (cp != lastCP + 1)
			{
				currentSegment.endCode = lastCP;
				currentSegment = new SegmentInfo();
				currentSegment.startCode = cp;
				int thisGid = subset.getExistingSubsetGid(unicodeChar2glyph(cp));
				currentSegment.idDelta = thisGid - computeCodePointToEmit(cp, symbolPad);
				segmentList.add(currentSegment);
			} else if (currentSegment.canUseIDDelta)
			{
				int thisGid = subset.getExistingSubsetGid(unicodeChar2glyph(cp));
				if (thisGid - computeCodePointToEmit(cp, symbolPad) != currentSegment.idDelta)
				{
					currentSegment.canUseIDDelta = false;
				} 
			}
			lastCP = cp;	
		}

		if (currentSegment != null)
			currentSegment.endCode = lastCP;

		int segmentCount = segmentList.size();

		if (addTerminalSegment)
			segmentCount++; // add 1 for the terminal segment

		int subtableSize = 16 + 8*segmentCount; // everything but the glyphIDArray is included in this.
		builder.ensureCapacity(offset+subtableSize);

		builder.setuint16(offset, 4); //format
		builder.setuint16(offset+4, 0); //language
		builder.setuint16(offset+6, segmentCount*2); //segCountX2
		int entrySelector = (int)Math.floor(log2(segmentCount));
		int searchRange = 2*(int)(Math.pow(2, entrySelector));
		builder.setuint16(offset + 8, searchRange);
		builder.setuint16(offset+10, entrySelector);
		builder.setuint16(offset+12, 2*segmentCount - searchRange);
		builder.setuint16(offset + 14 + segmentCount*2, 0); // padding byte

		int currGlyphOffset = 0;
		for (int i = 0; i < segmentList.size(); i++)
		{
			currentSegment = (SegmentInfo)segmentList.get(i);
			builder.setuint16(offset + 14 + 2*i, computeCodePointToEmit(currentSegment.endCode, symbolPad));
			builder.setuint16(offset + 16 + 2*segmentCount + 2*i, computeCodePointToEmit(currentSegment.startCode, symbolPad));
			builder.setint16(offset + 16 + 4*segmentCount + 2*i, 
					currentSegment.canUseIDDelta ?  currentSegment.idDelta : 0);
			builder.setuint16(offset + 16 + 6*segmentCount + 2*i,  
					currentSegment.canUseIDDelta ?  0 : (currGlyphOffset + segmentCount - i)*2);

			if (!currentSegment.canUseIDDelta)
			{
				builder.ensureCapacity(offset + 16 + 8*segmentCount + 2*(currGlyphOffset + currentSegment.endCode - currentSegment.startCode + 1));
				for (int j = currentSegment.startCode; j <= currentSegment.endCode; j++, currGlyphOffset++)
				{
					int thisGid = subset.getExistingSubsetGid(unicodeChar2glyph(j));
					builder.setuint16(offset + 16 + 8*segmentCount + 2*currGlyphOffset, thisGid);
				}		
			}
		}

		if (addTerminalSegment)
		{
			// the required terminal segment (if not included in the codepoints set)
			builder.setuint16(offset + 14 + 2*segmentList.size(), 0xffff);
			builder.setuint16(offset + 16 + 4*segmentCount - 2, 0xffff);
			builder.setuint16(offset + 16 + 6*segmentCount - 2, 1);
			builder.setuint16(offset + 16 + 8*segmentCount - 2, 0);
		}

		subtableSize += 2*currGlyphOffset;
		builder.setuint16(offset+2, subtableSize); // subtable length
		if (subtableSize >= 0x10000)
			throw new UnsupportedFontException("Format 4 cmap subtable is too big for format");
		return subtableSize;
	}

	private static double log2(double d) {
		return Math.log(d)/Math.log(2.0);
	}


	private int writeFormat12Subtable(OTByteArrayBuilder builder, int offset,TreeSet codepoints, Subset subset) 
	throws UnsupportedFontException, InvalidFontException
	{
		Iterator iter = codepoints.iterator();	
		int lastCP = 0;
		int lastGID = 0;
		int numGroups = 0;
		int tableSize = 16;
		builder.ensureCapacity(offset + tableSize);

		if (iter.hasNext())
		{
			lastCP = ((Integer)iter.next()).intValue();
			lastGID = subset.getExistingSubsetGid(unicodeChar2glyph(lastCP));
			tableSize += 12;
			builder.ensureCapacity(offset + tableSize);
			builder.setuint32(offset + 16 + 12*numGroups, lastCP); //startCharCode
			builder.setuint32(offset + 16 + 12*numGroups + 8, lastGID); // startGlyph
			numGroups++;
		}

		while (iter.hasNext())
		{
			int cp = ((Integer)iter.next()).intValue();
			int gid = subset.getExistingSubsetGid(unicodeChar2glyph(cp));

			if (cp != lastCP + 1 || gid != lastGID + 1)
			{
				builder.setuint32(offset + 16 + 12*(numGroups-1) + 4, lastCP); //endCharCode of the last group
				tableSize += 12;
				builder.ensureCapacity(offset + tableSize);
				builder.setuint32(offset + 16 + 12*numGroups, cp); // startCharCode
				builder.setuint32(offset + 16 + 12*numGroups + 8, gid); // startGlyph
				numGroups++;
			}
			lastCP = cp;
			lastGID = gid;
		}

		if (numGroups > 0) // if we've written any groups, then the last one doesn't have the endCode written yet.
			builder.setuint32(offset + 16 + 12*(numGroups-1) + 4, lastCP);

		builder.setuint16(offset, 12); // format
		builder.setuint16(offset+2, 0); // reserved
		builder.setuint32(offset+4, tableSize); // length
		builder.setuint32(offset + 8, 0); // language
		builder.setuint32(offset+12, numGroups); //nGroups

		return tableSize;
	}

	interface Format14Consumer
	{
		boolean defaultUV(int variationSelector, int uv);
		boolean nonDefaultUV(int variationSelector, int uv, int gid) throws UnsupportedFontException, InvalidFontException;
	}

	/**
	 * Iterates the variation sequences in the following order: for each variation selector,
	 * iterate the default uvs in sequential order followed by the non-default uvs in sequential order.
	 * @param stOffset
	 * @param consumer
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	void iterateFormat14(int stOffset, Format14Consumer consumer) 
	throws UnsupportedFontException, InvalidFontException
	{
		int numVarSelectorRecords = data.getuint32asint(stOffset + 6, "Only 2^31 selectors supported");

		for (int i = 0; i < numVarSelectorRecords; i++)
		{
			boolean stopThisIteration = false;
			int variationSelector = data.getuint24(stOffset + 10 + 11*i);

			// harvest default variation mappings.
			int uvsOffset = data.getuint32asint(stOffset + 10 + 11*i + 3, "Only 2^31 offsets supported");
			if (uvsOffset != 0)
			{
				uvsOffset += stOffset;

				int numUVRanges = data.getuint32asint(uvsOffset, "Only 2^31 UV ranges supported");
				for (int j = 0; j < numUVRanges; j++)
				{
					int startUV = data.getuint24(uvsOffset + 4 + 4*j);
					int addlCount = data.getuint8(uvsOffset + 4 + 4*j + 3);
					for (int k = 0; k <= addlCount; k++)
					{
						if (!consumer.defaultUV(variationSelector, startUV + k))
						{
							stopThisIteration = true;
							break;
						}
					}

					if (stopThisIteration)
						break;
				}
			}
			uvsOffset = data.getuint32asint(stOffset + 10 + 11*i + 7, "Only 2^31 offsets supported");
			if (uvsOffset != 0)
			{
				uvsOffset += stOffset;
				int numUVRanges = data.getuint32asint(uvsOffset, "Only 2^31 UV ranges supported");
				for (int j = 0; j < numUVRanges; j++)
				{
					int uv = data.getuint24(uvsOffset + 4 + 5*j);
					int gid = data.getuint16(uvsOffset + 4 + 5*j + 3);
					if (!consumer.nonDefaultUV(variationSelector, uv, gid))
					{
						break;
					}
				}
			}
		}
	}

	private static class Format14Writer implements Format14Consumer
	{
		private final Subset subset;
		private final Set codepoints;
		private final OTByteArrayBuilder builder;
		private final int newSTOffset;
		private Integer lastVS = new Integer(-1);
		private Map vsToUnicode /* <Integer, List<Integer>>*/ = new LinkedHashMap();
		int stSize = 0;

		static class UVAndGid 
		{
			UVAndGid(int uv, int gid)
			{
				this.uv = uv;
				this.gid = gid;
			}

			int uv;
			int gid;
		}

		Format14Writer(Subset subset, Set codepoints, OTByteArrayBuilder builder, int newSTOffset)
		{
			this.subset = subset;
			this.codepoints = codepoints;
			this.builder = builder;
			this.newSTOffset = newSTOffset;
		}

		public boolean defaultUV(int variationSelector, int uv) 
		{
			if (codepoints.contains(new Integer(uv)))
			{
				if (lastVS.intValue() != variationSelector)
				{
					lastVS = new Integer(variationSelector);
				}

				List uvList = (List)vsToUnicode.get(lastVS);
				if (uvList == null)
				{
					uvList = new ArrayList();
				}

				UVAndGid uvAndGid = new UVAndGid(uv, -1);

				uvList.add(uvAndGid);
				vsToUnicode.put(lastVS, uvList);
			}
			return true;
		}

		public boolean nonDefaultUV(int variationSelector, int uv, int gid) 
		{
			if (codepoints.contains(new Integer(uv)))
			{
				if (lastVS.intValue() != variationSelector)
				{
					lastVS = new Integer(variationSelector);
				}

				List uvList = (List)vsToUnicode.get(lastVS);
				if (uvList == null)
				{
					uvList = new ArrayList();
				}

				UVAndGid uvAndGid = new UVAndGid(uv, gid);

				uvList.add(uvAndGid);
				vsToUnicode.put(lastVS, uvList);
			}
			return true;
		}

		void writeSubtable() throws UnsupportedFontException, InvalidFontException
		{
			stSize = 10 + 11*vsToUnicode.size();
			builder.ensureCapacity(this.newSTOffset + stSize);
			builder.setuint16(this.newSTOffset, 14);
			builder.setuint32(this.newSTOffset + 6, vsToUnicode.size());

			int i = 0;
			Iterator iter = vsToUnicode.keySet().iterator();
			while (iter.hasNext())
			{
				Integer vs = (Integer)iter.next();
				int thisSize = 0;
				List uvAndGidList = (List)vsToUnicode.get(vs);
				thisSize = writeDefaultUVSTable(newSTOffset + stSize, uvAndGidList);
				builder.setuint24(this.newSTOffset + 10 + 11*i, vs.intValue());
				builder.setuint32(this.newSTOffset + 10 + 11*i + 3, thisSize == 0 ? 0:stSize);
				stSize += thisSize;

				thisSize = writeNonDefaultUVSTable(newSTOffset + stSize, uvAndGidList);
				builder.setuint32(this.newSTOffset + 10 + 11*i + 7, thisSize == 0 ? 0:stSize);
				stSize += thisSize;
				i++;
			}

			builder.setuint32(this.newSTOffset + 2, stSize);
		}

		private int writeDefaultUVSTable(int newOffset, List uvAndGidList)
		{
			int lastUV = -1;
			int numUVRanges = 0;
			Iterator iter = uvAndGidList.iterator();
			int thisRangeCount = 0;

			while (iter.hasNext())
			{
				UVAndGid uvAndGid = (UVAndGid)iter.next();
				if (uvAndGid.gid == -1)
				{
					if (lastUV +1 != uvAndGid.uv)
					{
						builder.ensureCapacity(newOffset + 4 + 4*(numUVRanges+1));
						if (numUVRanges > 0)
						{
							builder.setuint8(newOffset + 4 + 4*(numUVRanges-1) + 3, thisRangeCount);
						}
						builder.setuint24(newOffset + 4 + 4*numUVRanges, uvAndGid.uv);

						thisRangeCount = 0;
						numUVRanges++;
					} else
						thisRangeCount++;
				}
				else
				{
					break; // we've reached the end of the default uvs.
				}
			}

			if (numUVRanges > 0)
			{
				builder.setuint8(newOffset + 4 + 4*(numUVRanges-1) + 3, thisRangeCount);

				builder.setuint32(newOffset, numUVRanges);
				return 4 + 4*numUVRanges;
			}

			return 0;
		}

		private int writeNonDefaultUVSTable(int newOffset, List uvAndGidList) 
		throws UnsupportedFontException, InvalidFontException
		{
			int numUVRanges = 0;
			Iterator iter = uvAndGidList.iterator();

			while (iter.hasNext())
			{
				UVAndGid uvAndGid = (UVAndGid)iter.next();
				if (uvAndGid.gid != -1)
				{
					builder.ensureCapacity(newOffset + 4 + 5*(numUVRanges+1));
					builder.setuint24(newOffset + 4 + 5*numUVRanges, uvAndGid.uv);
					builder.setuint16(newOffset + 4 + 5*numUVRanges + 3, subset.getExistingSubsetGid(uvAndGid.gid));
					numUVRanges++;
				}
			}
			if (numUVRanges > 0)
			{
				builder.setuint32(newOffset, numUVRanges);
				return 4 + 5*numUVRanges;
			}
			return 0;
		}
	}

	private int writeFormat14Subtable(OTByteArrayBuilder builder, int offset, TreeSet codepoints, Subset subset, int originalOffset) 
	throws UnsupportedFontException, InvalidFontException
	{
		Format14Writer writer = new Format14Writer(subset, codepoints, builder, offset);
		iterateFormat14(originalOffset, writer);

		writer.writeSubtable();

		return writer.stSize;
	}

	private static class Format14Harvester implements Format14Consumer
	{
		private final Set harvestedCPs;
		private final Subset harvestedGids;
		final Set variationUVs;
		boolean variationSequencesFound = false;

		public Format14Harvester(Set harvestedCPs, Subset harvestedGids) 
		{
			this.harvestedCPs = harvestedCPs;
			this.harvestedGids = harvestedGids;
			variationUVs = new HashSet();
		}

		public boolean defaultUV(int variationSelector, int uv) 
		{
			if (harvestedCPs.contains(new Integer(uv)))
			{
				variationUVs.add(new Integer(variationSelector));
				variationSequencesFound = true;
				return false;
			}
			return true;
		}

		public boolean nonDefaultUV(int variationSelector, int uv, int gid) 
		throws UnsupportedFontException, InvalidFontException 
		{
			if (harvestedCPs.contains(new Integer(uv)))
			{
				Integer vs = new Integer(variationSelector);
				if (!variationUVs.contains(vs))
				{
					variationUVs.add(vs);
					variationSequencesFound = true;
				}
				harvestedGids.getSubsetGid(gid);
			}
			return true;
		}
	}

	boolean gatherPossibleMappings(Iterator inputCPs, Set harvestedCPs, Subset harvestedGids) 
	throws UnsupportedFontException, InvalidFontException
	{
		// harvest non-variation mappings.
		while (inputCPs.hasNext())
		{
			Integer cp = (Integer)inputCPs.next();

			int gid = unicodeChar2glyph(cp.intValue());
			if (gid != 0)
			{
				harvestedCPs.add(cp);
				harvestedGids.getSubsetGid(gid);
			}
		}

		// harvest variation mappings.
		int variationSubtableOffset = probe(PlatformID.UNICODE, UnicodeEncodingID.UNI_VARIATION_SEQ);
		if (variationSubtableOffset != -1)
		{
			Format14Harvester harvester = new Format14Harvester(harvestedCPs, harvestedGids);
			iterateFormat14(variationSubtableOffset, harvester);

			// For all of the variation selectors, see if they are also 
			// available "standalone." If so, add them to the list of supported CPs.
			Iterator iter = harvester.variationUVs.iterator();
			while (iter.hasNext())
			{
				Integer cp = (Integer)iter.next();
				int gid = unicodeChar2glyph(cp.intValue());
				if (gid != 0)
				{
					harvestedCPs.add(cp);
					harvestedGids.getSubsetGid(gid);
				}
			}

			return harvester.variationSequencesFound;
		}

		return false;

	}

	public void stream(Map tables) 
	{
		OTByteArrayBuilder newData = this.getDataAsByteArray();
		tables.put (new Integer (Tag.table_cmap), newData);
	}

	private class SWFCmapSelector implements CmapSelector
	{
		class CmapInfo {
			final int platformId;
			final int platformEncoding;
			final int offset;
			CmapInfo(int platformId, int platformEncoding, int offset)
			{
				this.platformId = platformId;
				this.platformEncoding = platformEncoding;
				this.offset = offset;
			}
		}

		boolean foundRequiredCmap = false;
		List /*<CmapInfo>*/ cmapsToKeep = new ArrayList /*<CmapInfo>*/();

		private boolean alreadyOneWithThisOffset(int offset)
		{
			Iterator iter = cmapsToKeep.iterator();
			while (iter.hasNext())
			{
				CmapInfo info = (CmapInfo)iter.next();
				if (info.offset == offset)
					return true;
			}
			return false;
		}

		public void cmapFound(int platformID, int platformEncoding, int index) 
		throws InvalidFontException, UnsupportedFontException 
		{
			// Keep one copy of each relevant subtable.

			int offset = Cmap.this.getOffset(index);
			switch (platformID)
			{
			case 0:
				switch (platformEncoding) 
				{
				case 3:
					if (!alreadyOneWithThisOffset(offset))
					{
						foundRequiredCmap = true;
						cmapsToKeep.add(new CmapInfo(platformID, platformEncoding, offset));
					}
					break;
				case 4:
					if (!alreadyOneWithThisOffset(offset))
					{
						foundRequiredCmap = true;
						cmapsToKeep.add(new CmapInfo(platformID, platformEncoding, offset));
					}
					break;
				case 5: // we keep these, but they don't meet the DefineFont4 requirements, so don't set 'foundRequiredCmap'.
					cmapsToKeep.add(new CmapInfo(platformID, platformEncoding, offset));
					break;
				}
				break;
			case 3:
				switch (platformEncoding) 
				{
				case 0:
				case 1:
					if (!alreadyOneWithThisOffset(offset))
					{
						foundRequiredCmap = true;
						cmapsToKeep.add(new CmapInfo(platformID, platformEncoding, offset));
					}
					break;
				case 10:
					if (!alreadyOneWithThisOffset(offset))
					{
						foundRequiredCmap = true;
						cmapsToKeep.add(new CmapInfo(platformID, platformEncoding, offset));
					}
					break;
				}
				break;
			}

		}
	}

	private int generateRequiredCmap(OTByteArrayBuilder builder, int stOffset, int numGlyphs) 
	throws UnsupportedFontException, InvalidFontException
	{
		if (nonUnicodeSubtableIndices.length == 0)
			throw new UnsupportedFontException("Could not generate required cmap subtables");

		TreeSet /*<Integer>*/ newCmap = new TreeSet();

		for (int j = 0; j < nonUnicodeSubtableIndices.length; j++)
		{
			if (nonUnicodeSubtableIndices[j].charsetName.equals("MacSymbol"))
			{
				// TODO handle symbol cmap
				if (nonUnicodeSubtableIndices.length == 1)
					throw new UnsupportedFontException("Cannot generate a required cmap");
				continue;
			}

			Charset cs = null;
			try {
				cs = CharsetUtil.forNameICU(nonUnicodeSubtableIndices[j].charsetName);
			} catch (UnsupportedCharsetException e) {
				throw new UnsupportedFontException("Could not generate required cmap subtables", e);
			} catch (IllegalCharsetNameException e) {
				throw new UnsupportedFontException("Could not generate required cmap subtables", e);
			}
			CharsetDecoder decoder = cs.newDecoder();

			//conv.setSubstitutionMode (false);

			int maxChar = getHighestMappedCode(nonUnicodeSubtableIndices[j].subtableOffset, false);
			int minChar = getLowestMappedCode(nonUnicodeSubtableIndices[j].subtableOffset);
			int index = this.offsetToIndex(nonUnicodeSubtableIndices[j].subtableOffset);

			CharBuffer targetChars = CharBuffer.allocate(1);
			ByteBuffer bb = ByteBuffer.allocate(2);
			final boolean warnOnFailedConversion = false;

			for (int i = minChar; i <= maxChar; i++)
			{
				bb.rewind();
				targetChars.rewind();
				int glyphId = char2glyph(i, index);
				if (glyphId != 0)
				{
					if (i < 256) 
					{
						bb.put((byte)(i & 0xff));
					} 
					else if (i < 65536) 
					{
						bb.put((byte)(i >> 8));
						bb.put((byte)(i & 0xff));
					} else{
						throw new UnsupportedFontException("Only 2-byte encodings supported");
					} 

					bb.rewind();
					CoderResult coderResult = null;
					try {
						coderResult = CharsetUtil.decodeLoopNoExceptions(bb, targetChars, decoder);
					} catch (IllegalStateException e) {

					} catch (CoderMalfunctionError e) {

					}
					if (!coderResult.isError())
					{
						targetChars.rewind();
						newCmap.add(new Integer(targetChars.get()));
					} else if (warnOnFailedConversion) {
						// Some badly made fonts encode things not in the official charset. For
						// example, 0a\0a115bc7a49d685fd49649c1ddb4bfc05478fd7a\GCSUN02M.TTF
						// encodes 0x100, which when converted to unicode, converts to 0x1 0x0. 
						// Alternatively, there are code points that fail to convert to unicode altogether
						// since they aren't supposed to be in the charset.
						System.err.println("Couldn't convert to unicode: " + i);
					}
				}
			}
		}

		return writeFormat4Subtable(builder, stOffset, newCmap, new SubsetDefaultImpl(numGlyphs, false), 0);
	}

	private int copySubtable(OTByteArrayBuilder builder, int stOffset, int originalSTOffset) 
	throws InvalidFontException, UnsupportedFontException
	{
		int origFormat = this.data.getuint16(originalSTOffset);
		int length;
		switch(origFormat)
		{
		case 0:
		case 2:
		case 4:
		case 6:
			length = this.data.getuint16(originalSTOffset + 2);
			break;
		case 8: 
		case 10:
			length = this.data.getuint32asint(originalSTOffset + 4, "Only signed int length cmap subtables supported");
			break;
		case 12:
			// The length field isn't as reliable as the nGroups field. Use it to compute the length.
			length = 16 + 12*this.data.getuint32asint(originalSTOffset + 12, "Only signed int length cmap subtables supported");
			break;
		case 14:
			length = this.data.getuint32asint(originalSTOffset + 2, "Only signed int length cmap subtables supported");
			break;
		default:
			throw new UnsupportedFontException("Unsupported cmap format: " + origFormat);
		}
		try {
			builder.replace(stOffset, this.data, originalSTOffset, length);
		} catch (ArrayIndexOutOfBoundsException e)
		{
			builder.replace(stOffset, this.data, originalSTOffset, this.data.getSize() - originalSTOffset);
		}

		return length;
	}

	public void streamForSWF(Map tables, int numGlyphs) 
	throws UnsupportedFontException, InvalidFontException
	{
		SWFCmapSelector selector = new SWFCmapSelector();
		enumerateCmaps(selector);

		int encodingRecordOffset = 4;
		int numTables = selector.cmapsToKeep.size() + (selector.foundRequiredCmap ? 0:1);
		int stOffset = encodingRecordOffset + 8*numTables;
		OTByteArrayBuilder builder = OTByteArray.getOTByteArrayBuilderInstance(stOffset);
		builder.setuint16(0, 0);
		if (!selector.foundRequiredCmap)
		{
			builder.setuint16(encodingRecordOffset, PlatformID.MICROSOFT);
			builder.setuint16(encodingRecordOffset + 2, MicrosoftEncodingId.UTF16_BE_BMP_ONLY);
			builder.setuint32(encodingRecordOffset + 4, stOffset);
			encodingRecordOffset += 8;
			stOffset += generateRequiredCmap(builder, stOffset, numGlyphs);
		}

		Iterator /*<CmapInfo>*/ iter = selector.cmapsToKeep.iterator();
		while (iter.hasNext())
		{
			SWFCmapSelector.CmapInfo cmapInfo = (SWFCmapSelector.CmapInfo)iter.next();

			int length = copySubtable(builder, stOffset, cmapInfo.offset);
			if (length > 0)
			{
				builder.setuint16(encodingRecordOffset, cmapInfo.platformId);
				builder.setuint16(encodingRecordOffset + 2, cmapInfo.platformEncoding);
				builder.setuint32(encodingRecordOffset + 4, stOffset);
				stOffset += length;
				encodingRecordOffset += 8;
			} else
				numTables--;
		}

		builder.setuint16(2, numTables);

		tables.put (new Integer (Tag.table_cmap), builder);
	}

	int offsetToIndex(int offset)
	throws InvalidFontException
	{
		int nTables = data.getuint16(2);
		for (int i = 0; i < nTables; i++) {
			if (offset == data.getuint32(4 + i*8 + 4))
				return i;
		}
		return -1;
	}

	private final class MapElement implements Comparable
	{
		int mCodePoint;
		int mGlyphID;

		MapElement(int cp, int gid) {
			mCodePoint = cp;
			mGlyphID = gid;
		}

		public int compareTo(Object obj)
		{
			return mCodePoint - ((MapElement)obj).mCodePoint;
		}
	}

	private final class MapSegment
	{
		int mFirstIndex;
		int mEndIndex;
		boolean mOrdered;

		MapSegment(int firstIndex, int endIndex, boolean ordered)
		{
			mFirstIndex = firstIndex;
			mEndIndex = endIndex;
			mOrdered = ordered;
		}
	}
}
