/*
*
*	File: Vhea.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'vhea' table.
 * 
 * <h4>Version handling</h4>
 * 
 * <p>'vhea' tables have a major/minor version number.
 * This implementation:
 * <ul> 
 * <li>fully supports version 1.0 and 1.1 tables,</li>
 * <li>interprets 1.x (x>1) tables as 1.1 tables,</li>
 * <li>rejects other versions with an <code>UnsupportedFontException</code> 
 * at construction time.</li>
 * </ul>
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

final public class Vhea extends Table {

  protected Vhea (FontByteArray buffer) 
      throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);

    int majorVersion = getTableMajorVersion ();
    if (majorVersion != 1) {
      throw new UnsupportedFontException 
                   ("'vhea' tables with major version " 
                        + majorVersion + "are not supported"); }
  }

  /** Get the major version of the table */
  public int getTableMajorVersion () throws InvalidFontException {
    return this.data.getuint16 (0);
  }
  
  /** get the minor version of the table */
  public int getTableMinorVersion () throws InvalidFontException {
    return this.data.getuint16 (2);
  }
  
  public int getAscent() throws InvalidFontException {
	  return this.data.getint16(4);
  }

  /** Get the number of vMetrics */
  public int getNumberOfVMetrics () throws InvalidFontException {
    return this.data.getuint16 (34);
  }
  
  public void subsetAndStream (Subset subset, Map tables) throws InvalidFontException {
	  OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(this.data);
	  newData.setuint16 (34, subset.getNumGlyphs ()); // the number of glyphs is at byte 34 regardless of table format.
	  tables.put (new Integer (Tag.table_vhea), newData); 
  }
  
  public void stream(Map tables) {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_vhea), newData);
  }
}