/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.taskmanagement;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * used to specify searching constraints for TaskManagement queries.  see {@link TaskManager#getTasks(Filter, int, int)}
 */
public class Filter {

    /**
     * The task type name
     */
    private String[] taskTypeNames;

    /**
     * The root task Id for where to start searching for tasks.
     */
    private String parentTaskId;

    /**
     * true to recurse through subtasks, false otherwise..
     */
    private boolean returnFlatStructure = false;

    /**
     * indicates whether or not to return the child task structure under a task
     */
    private boolean returnTaskStructure = false;

    /**
     * A List of conditions for  this Filter instance
     */
    private List<Condition> conditions;

    /**
     * Construct an empty Filter
     */
    public Filter() {
        this(null);
    }

    /**
     * Construct a Filter object with null conditions and the specified taskTypes
     *
     * @param taskTypes the taskTypes for this filter
     */
    public Filter(String ... taskTypes) {
        copyTaskTypeArray(taskTypes);
        this.parentTaskId = null;
    }

    /**
     * Sets the taskTypes to be returned by a query
     *
     * @param taskTypes task type names specified for this filter
     */
    public void setTaskTypes(String... taskTypes) {
        copyTaskTypeArray(taskTypes);
    }

    /**
     * helper method to set the array of taskType names
     * @param taskTypes array to copy
     */
    private void copyTaskTypeArray(String[] taskTypes) {
        if (taskTypes != null) {
            taskTypeNames = new String[taskTypes.length];
            System.arraycopy(taskTypes, 0, taskTypeNames, 0, taskTypes.length);
        }
    }

    /**
     * Gets the task type names associated with this filter
     *
     * @return A String[] representing the task type name or null if
     *         no task type name has been set
     */
    public String[] getTaskTypes() {
        // return a copy
        if (taskTypeNames != null) {
            String[] tempList = new String[taskTypeNames.length];
            System.arraycopy(taskTypeNames, 0, tempList, 0, taskTypeNames.length);
            return tempList;
        } else {
            return null;
        }
    }

    /**
     * Set the List of Condition objects associated with this filter.  The
     * List must be set before using the filter.
     *
     * @param condition conditions to add to this filter
     */
    public void addCondition(Condition ... condition) {
        if (condition == null) {
            return;
        }

        if (this.conditions == null) {
            this.conditions = new ArrayList<Condition>();
        }
        for (Condition cond : condition) {
            this.conditions.add(cond);
        }
    }

    /**
     * Returns a List of Condition objects associated with this filter.
     *
     * @return A List of Condition objects or null if no conditions have been set.
     */
    public Iterator<Condition> getConditions() {
        if (this.conditions == null) {
            this.conditions = new ArrayList<Condition>();
        }
        return this.conditions.iterator();
    }

    /**
     * Specify the parent task Id for this filter.
     *
     * @param parentTaskId the id of the task at which we want to start searching for tasks.
     */
    public void setParentTaskId(String parentTaskId) {
        this.parentTaskId = parentTaskId;
    }

    /**
     * Return the parent task Id specified for this filter.
     * @return the parent task Id specified for this filter
     */
    public String getParentTaskId() {
        return this.parentTaskId;
    }

    /**
     * If true the search will recurse the subtasks to return all tasks matching this filter, false will only check the level specified by the parentTaskid.
     * The tasks will be returned in a flat structure, not in the parent/child structure. <br>
     * Consider only using #setReturnFlatStructure(true) or #setReturnTaskStructure(true), not both
     *
     * @param returnFlatStructure indicates to return all tasks regardless of where they fall in the structure
     */
    public void setReturnFlatStructure(boolean returnFlatStructure) {
        this.returnFlatStructure = returnFlatStructure;
    }

    /**
     * indicates if the tasks will be returned in a flattened format
     * @return true if tasks are to be returned flattened
     */
    public boolean isReturnFlatStructure() {
        return this.returnFlatStructure;
    }

    /**
     * if true child tasks are returned as a structure under a parent task
     * Consider only using #setReturnFlatStructure(true) or #setReturnTaskStructure(true), not both
     *
     * @param returnTaskStructure indicates to return tasks in their parent/child structure
     */
    public void setReturnTaskStructure(boolean returnTaskStructure) {
        this.returnTaskStructure = returnTaskStructure;
    }

    /**
     * indicates if task structure is being returned
     * @return true if the task structure is being returned.
     */
    public boolean isReturnTaskStructure() {
        return returnTaskStructure;
    }


}
