/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.translation.core.common;

import com.adobe.granite.translation.api.TranslationConfig;
import com.adobe.granite.translation.api.TranslationConstants.TranslationMethod;
import com.adobe.granite.translation.api.TranslationException;
import com.adobe.granite.translation.api.TranslationService;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.Map;

public abstract class AbstractTranslationService implements TranslationService {

    private static final Logger log = LoggerFactory.getLogger(AbstractTranslationService.class);

    protected TranslationServiceInfo translationInfo;

    protected TranslationConfig tc;

    protected String defaultCategory;

    /**
     * Object containing list of supported language codes and their appropriate mappings. This is a one to one mapping
     * list in which the "key" is the accepted language code and corresponds to a node name from
     * /libs/wcm/core/resources/languages. The "value" is the corresponding mapping for the current connector engine.
     * Example: map.put("en_us", "eng") --&gt; "en_us" corresponds to /libs/wcm/core/resources/languages/en_us and
     * "eng corresponds to the expected value in this MT engine for "English"
     */
    protected Map<String, String> availableLanguageMap;

    /**
     * Object containing list of supported category codes and their appropriate mappings. This is a one to one mapping
     * list in which the "key" is the accepted category code and corresponds to a node name from
     * /libs/granite/translation/resources/categories. The "value" is the corresponding mapping for the current
     * connector engine. Example: map.put("education", "EDUCATI") --&gt; "education" corresponds to
     * /libs/granite/translation/resources/categories/education and "EDUCATI" corresponds to the expected value in
     * this MT engine for "Education"
     */
    protected Map<String, String> availableCategoryMap;

    protected AbstractTranslationService(Map<String, String> availableLanguageMap,
        Map<String, String> availableCategoryMap, String name, String label, String attribution,
        String translationCloudConfigRootPath, TranslationMethod supportedTranslationMethod, TranslationConfig tc) {
        log.trace("Starting Constructor for: AbstractTranslationService");

        this.tc = tc;
        this.availableLanguageMap = availableLanguageMap;
        this.availableCategoryMap = availableCategoryMap;
        translationInfo =
            new TranslationServiceInfoImpl(name, label, attribution, translationCloudConfigRootPath,
                supportedTranslationMethod);
    }

    @Override
    public TranslationServiceInfo getTranslationServiceInfo() {
        return translationInfo;
    }

    @Override
    public String getDefaultCategory() {
        return defaultCategory;
    }

    @Override
    public void setDefaultCategory(String defaultCategory) {
        this.defaultCategory = defaultCategory;
    }

    @Override
    public void updateDueDate(String strTranslationJobID, Date date) throws TranslationException {
        throw new TranslationException("Due date update is not supported",
                TranslationException.ErrorCode.SERVICE_NOT_IMPLEMENTED);
    }

    protected boolean validateLanguageCode(String langCode) throws TranslationException {
        log.debug("In function: validateLanguageCode");
        for (String languageKey : tc.getLanguages().keySet()) {
            if (StringUtils.equalsIgnoreCase(langCode, languageKey)) {
                return true;
            }
        }
        return false;
    }

    public class TranslationServiceInfoImpl implements TranslationServiceInfo {

        private String attribution;
        private String translationServiceName;
        private String translationServiceLabel;
        private String translationCloudConfigRootPath;
        private TranslationMethod supportedTranslationMethod;

        TranslationServiceInfoImpl(String name, String label, String attribution,
            String translationCloudConfigRootPath, TranslationMethod supportedTranslationMethod) {
            this.attribution = attribution;
            this.translationServiceName = name;
            this.translationServiceLabel = label;
            this.supportedTranslationMethod = supportedTranslationMethod;
            this.translationCloudConfigRootPath = translationCloudConfigRootPath;
        }

        @Override
        public String getTranslationServiceAttribution() {
            log.trace("Starting function: getAttribution");
            return attribution;
        }

        @Override
        public String getTranslationServiceLabel() {
            log.trace("Starting function: getTranslationServiceLabel");
            return translationServiceLabel;
        }

        @Override
        public String getTranslationServiceName() {
            log.trace("Starting function: getTranslationServiceName");
            return translationServiceName;
        }

        @Override
        public TranslationMethod getSupportedTranslationMethod() {
            return supportedTranslationMethod;
        }

        @Override
        public String getServiceCloudConfigRootPath() {
            return translationCloudConfigRootPath;
        }
    }
}