/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.ut.ExErrItem;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ISODate;
import com.adobe.xfa.ut.ISOTime;
import com.adobe.xfa.ut.LcDate;
import com.adobe.xfa.ut.LcLocale;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.StringUtils;

/**
 * Class LogMessage represents a collection of one or more log message(s). Log
 * messages may be constructed from existing jfException and jfFormatPos objects
 * or by specifying a message Id along with a string representing the message
 * text. If a log message is constructed through the default constructor then
 * call insertMessage to insert a new message into the LogMessage object. As
 * with the constructors, new log messages constructed from calling
 * insertMessage may be constructed from existing jfException and jfFormatPos
 * objects or by specifying a message Id along with a string representing the
 * message text.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public final class LogMessage {

	static public final int MSG_FATAL_ERROR = 6;

	static public final int MSG_IGNORE = 0;

	static public final int MSG_INFORMATION = 2;

	static public final int MSG_TRACE = 1;

	static public final int MSG_VALIDATION_ERROR = 5;

	static public final int MSG_VALIDATION_WARNING = 4;

	static public final int MSG_WARNING = 3;

	int meSeverity;

	private final List<LogMessageData> mItems = new ArrayList<LogMessageData>();

	private String msLocale;

	private String getLocale() {
		if (StringUtils.isEmpty(msLocale))
			msLocale = LcLocale.getLocale();
		
		LcLocale oLocale = new LcLocale(msLocale);
		
		if (! oLocale.isValid())
			msLocale = LcLocale.DEFAULT_LOCALE;
		else
			msLocale = oLocale.getIsoName();
		return msLocale;
	}

	/**
	 * Default Constructor
	 */
	public LogMessage() {

	}

	/**
	 * Construct a log message from a jfException object. <br>
	 * <br>
	 * oSource may contain more than one error message therefore this message
	 * will be a collection of LogMessageData objects. All messages will have
	 * the same eSeverity and date-time stamp.
	 * 
	 * @param oSource -
	 *            contains message Id and text
	 * @param eSeverity -
	 *            the message severity
	 */
	public LogMessage(ExFull oSource, int eSeverity /* = MSG_FATAL_ERROR */) {
		meSeverity = LogMessage.MSG_INFORMATION;
		msLocale = "";
		
		insertMessage(oSource, eSeverity, "");
	}

	/**
	 * Construct a log message given a message Id and a string representing the
	 * message text.
	 * 
	 * @param nId -
	 *            the message Id
	 * @param sText -
	 *            the message text
	 * @param eSeverity -
	 *            the message severity
	 */
	public LogMessage(int nId, String sText, int eSeverity) {
		meSeverity = LogMessage.MSG_INFORMATION;
		msLocale = "";
		
		insertMessage(nId, sText, eSeverity, "");
	}

	/**
	 * Construct a log message from a jfFormatPos object.
	 * 
	 * @param oSource -
	 *            contains message Id and text
	 * @param eSeverity -
	 *            the message severity
	 */
	public LogMessage(MsgFormatPos oSource, int eSeverity) {
		meSeverity = eSeverity;
		msLocale = "";
		
		insertMessage(oSource, eSeverity, "");
	}

	private void addMessage(
			int nId, 
			String sText, 
			int eSeverity,
			ISODate pDate, 
			ISOTime pTime, 
			String sSOMExpression/* ="" */) {
		
		if (pDate != null && pTime != null) {
			// Most callers specify the date and time. Avoid constructing the oDate
			// and oTime objects in this case, as they're expensive.
			LogMessageData oMsgData = new LogMessageData(nId, sText, eSeverity, pDate, pTime, sSOMExpression);
			mItems.add(oMsgData);
			return;
		}
		
		if (pDate == null)
			pDate = new ISODate(getLocale(), LcDate.DEFAULT_CENTURY_SPLIT);
		if (pTime == null)
			pTime = new ISOTime(getLocale(), LcDate.DEFAULT_CENTURY_SPLIT);
		
		LogMessageData oMsgData = new LogMessageData(nId, sText, eSeverity, pDate, pTime, sSOMExpression);
		mItems.add(oMsgData);
	}

	/**
	 * Return a count of the number of LogMessageData objects in this object's
	 * collection.
	 * 
	 * @return the count.
	 */
	public int count() {
		return mItems.size();
	}

	/**
	 * Reference the <i>nIndex</i>'th LogMessageData object in this object's
	 * collection.
	 * 
	 * @param index -
	 *            the index of LogMessageData object to return. Must be in the
	 *            range 0 to Count() - 1.
	 * @return the <i>nIndex</i>'th LogMessageData object.
	 */
	public LogMessageData get(int index) {
		return mItems.get(index);
	}


	/**
	 * Get the severity for this message. <br>
	 * <br>
	 * See enum LogMessage::LogMessageSeverity for possible return values.
	 * 
	 * @return an LogMessageSeverity, representing the severity for this
	 *         message.
	 */
	public int getSeverity() {
		return meSeverity;
	}

	/**
	 * Insert a new message from a jfException object. Create the date and time
	 * from a desired locale.
	 * 
	 * @param oSource -
	 *            contains message Id and text
	 * @param eSeverity -
	 *            the message severity
	 * @param sLocale -
	 *            the desired locale to create the date and time from <br>
	 *            <br>
	 *            oSource may contain more than one error message therefore this
	 *            message will be a collection of LogMessageData objects. All
	 *            messages will have the same eSeverity and date-time stamp.
	 */
	public void insertMessage(ExFull oSource, int eSeverity /* = LogMessage.MSG_FATAL_ERROR*/, String sLocale /* = "" */) {
		ISODate oDate = new ISODate(sLocale, LcDate.DEFAULT_CENTURY_SPLIT);
		ISOTime oTime = new ISOTime(sLocale, 0);

		String sText = "";
		int nId;
		int nSize = oSource.count();
		//
		// Resolve resource Ids
		//
		oSource.resolve();

		for (int i = 0; i < nSize; i++) {
			ExErrItem oExErrItem = oSource.item(i);

			nId = oExErrItem.resId();
			sText = oExErrItem.text();
			StringBuilder sTextBuf = new StringBuilder(sText);
			StringUtils.trim(sTextBuf);
			sText = sTextBuf.toString();

			addMessage(nId, sText, eSeverity, oDate, oTime, "");
		}
		updateSeverity(eSeverity);
	}

	/**
	 * Insert a new message given a message Id and a string representing the
	 * text of the message. Create the date and time from a desired locale.
	 * 
	 * @param nId -
	 *            the message Id
	 * @param sText -
	 *            the message text
	 * @param eSeverity -
	 *            the message severity
	 * @param sLocale -
	 *            the desired locale to create the date and time from
	 */
	public void insertMessage(int nId, String sText, int eSeverity /* = LogMessage.MSG_FATAL_ERROR*/, String sLocale /* = "" */) {
		ISODate	oDate = new ISODate(sLocale, LcDate.DEFAULT_CENTURY_SPLIT);
		ISOTime	oTime = new ISOTime(sLocale, 0);

		String str = sText;
		str = str.trim();

		addMessage(nId, str, eSeverity, oDate, oTime, "");
		updateSeverity(eSeverity);
	}

	/**
	 * Insert a new message from a jfFormatPos object. Create the date and time
	 * from a desired locale.
	 * 
	 * @param oSource -
	 *            contains message Id and text
	 * @param eSeverity -
	 *            the message severity
	 * @param sLocale -
	 *            the desired locale to create the date and time from
	 */
	public void insertMessage(MsgFormatPos oSource, int eSeverity /* = LogMessage.MSG_FATAL_ERROR*/, String sLocale /* = "" */) {
		ISODate	oDate = new ISODate(sLocale, LcDate.DEFAULT_CENTURY_SPLIT);
		ISOTime	oTime = new ISOTime(sLocale, 0);

		int nId = oSource.resId();

		String sText = oSource.toString();
		sText = sText.trim();

		addMessage(nId, sText, eSeverity, oDate, oTime, "");
		updateSeverity(eSeverity);
	}

	/**
	 * Insert a new message from a jfFormatPos object. Create the date and time
	 * from a desired locale.
	 * 
	 * @param sValidationMsg
	 *            the validation error text
	 * @param sLocale
	 *            the desired locale to create the date and time from
	 * @param sSOMExpression
	 *            the SOM expression for the node that caused the validation
	 *            error
	 */
	public void insertValidationError(String sValidationMsg, String sSOMExpression, String sLocale) {
		ISODate oDate = new ISODate(sLocale, LcDate.DEFAULT_CENTURY_SPLIT);
		ISOTime oTime = new ISOTime(sLocale, 0);

		addMessage(0, sValidationMsg, LogMessage.MSG_VALIDATION_ERROR, oDate, oTime, sSOMExpression);
	}

	/**
	 * Insert a new message from a jfFormatPos object. Create the date and time
	 * from a desired locale.
	 * 
	 * @param sValidationMsg
	 *            the validation warning text
	 * @param sLocale
	 *            the desired locale to create the date and time from
	 * @param sSOMExpression
	 *            the SOM expression for the node that caused the validation
	 *            warning
	 */
	public void insertValidationWarning(String sValidationMsg, String sSOMExpression, String sLocale) {
		ISODate oDate = new ISODate(sLocale, LcDate.DEFAULT_CENTURY_SPLIT);
		ISOTime oTime = new ISOTime(sLocale, 0);

		addMessage(0, sValidationMsg, LogMessage.MSG_VALIDATION_WARNING, oDate, oTime, sSOMExpression);
	}

	/**
	 * Set the severity for this message. <br>
	 * <br>
	 * See enum LogMessage::LogMessageSeverity for possible values.
	 * 
	 * @param eSeverity
	 *            the severity of this message
	 */
	public void setSeverity(int eSeverity) {
		meSeverity = eSeverity;
	}

	private void updateSeverity(int eSeverity) {
		// Update the severity
		if (eSeverity > getSeverity())
			setSeverity(eSeverity);
	}
}