/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.configuration;


import org.xml.sax.Attributes;

import com.adobe.xfa.Arg;
import com.adobe.xfa.Element;
import com.adobe.xfa.GenericAttribute;
import com.adobe.xfa.Node;
import com.adobe.xfa.Obj;
import com.adobe.xfa.Schema;
import com.adobe.xfa.ScriptDynamicPropObj;
import com.adobe.xfa.ScriptTable;
import com.adobe.xfa.XFA;
import com.adobe.xfa.ut.Assertions;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.StringUtils;


/**
 * A base class for ConfigurationKey and ConfigurationValue.
 *
 * @author Ian Benedict Carreon
 * @author Mike Tardif (ported to Java)
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public class ConfigurationElement extends Element {

    /**
     * Constructor.
     */
	public ConfigurationElement(Element parent, Node prevSibling, String uri,
								String lname, String qname, Attributes attr,
								int classTag, String className) {
		super(parent, prevSibling, uri, lname, qname, attr, classTag, className);
	}

	/**
	 * @see Node#canCreateChild(boolean, String)
	 */
	@FindBugsSuppress(code="ES")
	protected boolean canCreateChild(boolean bIsLeaf, String aName) {
		assert(aName != null);
		if (Assertions.isEnabled) assert(aName == aName.intern());

		int eClassTag = XFA.getTag(aName);
		if (eClassTag >=0 && isValidElement(eClassTag, false))
			return true;
		
		else if (bIsLeaf && isValidElement(XFA.CONFIGURATIONVALUETAG, false))
			return true;
		
		else if (!bIsLeaf && isValidElement(XFA.CONFIGURATIONKEYTAG, false))
			return true;
		
		return false;
	}
	/**
	 * @see Node#createChild(boolean, String)
	 */
	@FindBugsSuppress(code="ES")
	protected Node createChild(boolean bIsLeaf, String aName) {
		if (Assertions.isEnabled) assert(aName == aName.intern());
		
		Node oNode;
		int eClassTag = XFA.getTag(aName);
		if (eClassTag >= 0 && isValidElement(eClassTag, false))
			oNode = getModel().createElement(this, null, null, aName, aName, null, 0, null);
		else if (bIsLeaf)
			oNode = getModel().createElement(XFA.CONFIGURATIONVALUETAG, aName);
		else
			oNode = getModel().createElement(XFA.CONFIGURATIONKEYTAG, aName);
		return oNode;
	}

	/**
	 * Set the class name for this configuration element
	 * to the given tag, given it's parent.
	 *
	 * If the given tag is a valid child of the given parent,
	 * then re-classify this configuration node.
	 *
	 * @param parent the parent of the node to reclassify.
	 * @param eTag the XFA tag.
	 */
	protected void setClass(Element parent, int eTag) {
		if (parent != null && parent.isValidChild(eTag, 0, false, false))
			super.setClass(XFA.getAtom(eTag), eTag);
	}

	/**
	 * Get the name of this configuration element.
	 *
	 * @return the name of this element.
	 */
	public String getName() {
		String sName = super.getName();
		if (StringUtils.isEmpty(sName))
			sName = super.getLocalName();
		return sName;
	}

	/**
	 * Determine if a specified attribute tag is valid for this node.
	 * @param eTag 
	 *            the XFA tag to check
	 * @return true if valid.
	 *
	 * @exclude from published api.
	 */
	public boolean isValidAttr(int eTag, boolean bReport /* = false */, String value /* = null */) {
		if (super.isValidAttr(eTag, bReport, value))
			return true;

		if (getClassTag() == XFA.CONFIGURATIONKEYTAG ||
			getClassTag() == XFA.CONFIGURATIONVALUETAG)
			return true;

		return false;
	}

	protected ScriptDynamicPropObj getDynamicScriptProp(String sPropertyName,
								boolean bPropertyOverride, boolean bPeek) {
		// no schema defined,
		if (getClassTag() == XFA.CONFIGURATIONKEYTAG || 
			getClassTag() == XFA.CONFIGURATIONVALUETAG) {
			
			// only check children so use tree method
			ScriptDynamicPropObj scriptDynamicPropObj = super.getDynamicScriptProp(sPropertyName, bPropertyOverride, bPeek);
			if (scriptDynamicPropObj != null)
				return scriptDynamicPropObj;

			int nAttrIdx = findAttr(null, sPropertyName.intern());
			if (nAttrIdx >= 0) {
				return attributeFuncScriptObj;
			}
		}
		else {
			// use schema
			return super.getDynamicScriptProp(sPropertyName, bPropertyOverride, bPeek);
		}
		
		return null;
	}
	
	private final static ScriptDynamicPropObj attributeFuncScriptObj = 
		new ScriptDynamicPropObj(Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_ALL) {
		
		public boolean invokeGetProp(Obj scriptThis, Arg retValue, String sPropertyName) {
			return ConfigurationElement.getAttributeFunc(scriptThis, retValue, sPropertyName);
		}
		
		public boolean invokeSetProp(Obj scriptThis, Arg retValue, String sPropertyName) {
			return ConfigurationElement.setAttributeFunc(scriptThis, retValue, sPropertyName);
		}
		
		public boolean hasSetter() {
			return true;
		}
	};
	
	@FindBugsSuppress(code="ES")
	private static boolean getAttributeFunc(Obj scriptThis, Arg retVal, String sProp) {
		if (StringUtils.isEmpty(sProp))
			return false;
		
		ConfigurationElement element = (ConfigurationElement)scriptThis;
		int index = element.findAttr(null, sProp.intern());
		
		if (index == -1)
			return false;
		
		retVal.setString(element.getAttrVal(index));
		return true;
	}
	
	@FindBugsSuppress(code="ES")
	private static boolean setAttributeFunc(Obj scriptThis, Arg value, String sProp) {
		if (StringUtils.isEmpty(sProp))
			return false;
		
		ConfigurationElement element = (ConfigurationElement)scriptThis;
		
		element.updateAttribute(new GenericAttribute(null, sProp, sProp, value.toString()));
		
		return true;
	}

	/**
	 * @exclude from published api.
	 */
	public ScriptTable getScriptTable() {
		return ConfigurationElementScript.getScriptTable();
	}
}