package com.adobe.xfa.text;

import com.adobe.xfa.font.FontInfo;
import com.adobe.xfa.font.FontInstance;
import com.adobe.xfa.font.FontService;

import com.adobe.xfa.gfx.GFXTextAttr;

import com.adobe.xfa.ut.LcLocale;
import com.adobe.xfa.ut.LcData;
import com.adobe.xfa.ut.StringUtils;
import com.adobe.xfa.ut.UnitSpan;

/**
 * <p>
 * The text attribute object can represent any possible combination of
 * attributes that apply to text (e.g., typeface, size, left margin,
 * tabs).  Applications use this object to set and query rich text
 * attributes.
 * </p>
 * <p>
 * Note that, for architectural reasons, this class extends class
 * TextGfxAttr.  Class TextGfxAttr provides sparse attribute
 * coverage for so-called "graphic" text attributes (e.g., colour,
 * underline).	Class TextAttr extends that with all other attributes
 * that apply to text.	For more information on graphic text attributes,
 * please see class TextGfxAttr.
 * </p>
 * <p>
 * Each text attribute can be set or queried independently.
 * Additionally, each attribute can be flagged as enabled or disabled.
 * When enabled, the corresponding attribute value has meaning.  When
 * disabled, the value can be thought of as unknown or irrelevant.	To
 * set a single text attribute over a range of text, the application
 * creates a TextAttr with just that attribute enabled, and applies it
 * to the range.  Similarly, the application can apply a subset of
 * attributes by enabling just those values.  The application can also
 * query attributes over a range.  Those attributes that are constant
 * over the range will appear as enabled values in the resulting
 * TextAttr object.  Those that vary over the range are disabled.
 * </p>
 * <p>
 * This object has a large number of accessors, which appear in groups
 * of four, one such group for each attribute.	For example, consider
 * the typeface attribute.	There are two overloads each of Typeface()
 * and TypefaceEnable().  In each overload pair, one sets the value and
 * one retrieves it.  In other words, one Typeface() overload returns
 * the current typeface value and one sets it, and one TypefaceEnable()
 * overload returns the enabled state and one sets it.	Enabled flags
 * are all Boolean, with TRUE indicating the value is enabled.	Setting
 * an attribute value in a TextAttr object (e.g., by second Typeface()
 * overload) automatically enables it.	Enabling a value (e.g., by
 * second TypefaceEnable() overload) that has never been set usually
 * sets it to a default value.	In rare cases, the uninitialized value
 * may be left disabled.
 * </p>
 * <p>
 * Some attributes apply at the paragraph level while other can apply to
 * smaller runs of text.  When the application sets paragraph attributes
 * over a text range, the range is implicitly rounded up to complete
 * paragraphs for the paragraph attributes.
 * </p>
 * <p>
 * Font information may be set by individual components (e.g., typeface,
 * size) or by font object pointer.
 * </p>
 *
 * @exclude from published api.
 */

public class TextAttr extends TextGfxAttr {
	static class FontDesc {
		public static final int VALUE_ABSENT = 0;
		public static final int VALUE_PRESENT = 1;
		public static final int VALUE_VALID = 2;

		private String msTypeface;
		private String msEncoding;
		private UnitSpan moSize;
		private int mnWeight;
		private double mnHorizontalScale;
		private double mnVerticalScale;
		private boolean mbItalic;

		private boolean mbTypefacePresent;
		private boolean mbEncodingPresent;
		private boolean mbSizePresent;
		private boolean mbWeightPresent;
		private boolean mbItalicPresent;
		private boolean mbHorizontalScalePresent;
		private boolean mbVerticalScalePresent;

		public FontDesc () {
			initialize();
		}

		public FontDesc (TextAttr oAttr) {
			initialize();
			setAttr (oAttr);
		}

		public FontDesc (String sTypeface) {
			initialize();
			setTypeface (sTypeface);
		}

		public FontDesc (UnitSpan oSize) {
			initialize();
			setSize (oSize);
		}

		public FontDesc (int nWeight) {
			initialize();
			setWeight (nWeight);
		}

		public void setAttr (TextAttr oAttr) {
			setTypeface (oAttr.typeface());
			setEncoding (oAttr.encoding());
			setSize (oAttr.size());
			setWeight (oAttr.weight());
			setItalic (oAttr.italic(), oAttr.italicEnable());
			setHorizontalScale (oAttr.horizontalScale(), oAttr.horizontalScaleEnable());
			setVerticalScale (oAttr.verticalScale(), oAttr.verticalScaleEnable());
		}

		public int hasTypeface () {
			if (! mbTypefacePresent) {
				return VALUE_ABSENT;
			}
			if ((msTypeface == null) || (msTypeface.length() == 0)) {
				return VALUE_PRESENT;
			}
			return VALUE_VALID;
		}

		public String getTypeface () {
			return msTypeface;
		}

		public void setTypeface (String sTypeface) {
			msTypeface = sTypeface;
			mbTypefacePresent = true;
		}

		public int hasEncoding () {
			if (! mbEncodingPresent) {
				return VALUE_ABSENT;
			}
			if ((msEncoding == null) || (msEncoding.length() == 0)) {
				return VALUE_PRESENT;
			}
			return VALUE_VALID;
		}

		public String getEncoding () {
			return msEncoding;
		}

		public void setEncoding (String sEncoding) {
			msEncoding = sEncoding;
			mbEncodingPresent = true;
		}

		public int hasSize () {
			if (! mbSizePresent) {
				return VALUE_ABSENT;
			}
			if (moSize.value() < 0) {
				return VALUE_PRESENT;
			}
			return VALUE_VALID;
		}

		public UnitSpan getSize () {
			return moSize;
		}

		public void setSize (UnitSpan oSize) {
			moSize = oSize;
			mbSizePresent = true;
		}

		public int hasWeight () {
			if (! mbWeightPresent) {
				return VALUE_ABSENT;
			}
			if (mnWeight == FontInfo.WEIGHT_UNKNOWN) {
				return VALUE_PRESENT;
			}
			return VALUE_VALID;
		}

		public int getWeight () {
			return mnWeight;
		}

		public void setWeight (int nWeight) {
			mnWeight = (nWeight == FontInfo.WEIGHT_UNKNOWN) ? nWeight : coerceWeight (nWeight);
			mbWeightPresent = true;
		}

		public static int coerceWeight (int nWeight) {
			return (nWeight >= FontInfo.WEIGHT_BOLD) ? FontInfo.WEIGHT_BOLD : FontInfo.WEIGHT_NORMAL;
		}

		public int hasItalic () {
			if (! mbItalicPresent) {
				return VALUE_ABSENT;
			}
			return VALUE_VALID;
		}

		public boolean getItalic () {
			return mbItalic;
		}

		public void setItalic (boolean bItalic, boolean bEnable) {
			mbItalic = bItalic;
			mbItalicPresent = bEnable;
		}
		public void setItalic (boolean bItalic) {
			setItalic (bItalic, true);
		}

		public int hasHorizontalScale () {
			if (! mbHorizontalScalePresent) {
				return VALUE_ABSENT;
			}
			if (mnHorizontalScale <= 0.0) {
				return VALUE_PRESENT;
			}
			return VALUE_VALID;
		}

		public double getHorizontalScale () {
			return mnHorizontalScale;
		}

		public void setHorizontalScale (double nHorizontalScale, boolean bEnable) {
			mnHorizontalScale = nHorizontalScale;
			mbHorizontalScalePresent = bEnable;
		}
		public void setHorizontalScale (double nHorizontalScale) {
			setHorizontalScale (nHorizontalScale, true);
		}

		public int hasVerticalScale () {
			if (! mbVerticalScalePresent) {
				return VALUE_ABSENT;
			}
			if (mnVerticalScale <= 0.0) {
				return VALUE_PRESENT;
			}
			return VALUE_VALID;
		}

		public double getVerticalScale () {
			return mnVerticalScale;
		}

		public void setVerticalScale (double nVerticalScale, boolean bEnable) {
			mnVerticalScale = nVerticalScale;
			mbVerticalScalePresent = bEnable;
		}
		public void setVerticalScale (double nVerticalScale) {
			setVerticalScale (nVerticalScale, true);
		}

		public boolean isEmpty () {
			return (hasTypeface() == VALUE_ABSENT)
				&& (hasEncoding() == VALUE_ABSENT)
				&& (hasSize() == VALUE_ABSENT)
				&& (hasWeight() == VALUE_ABSENT)
				&& (hasItalic() == VALUE_ABSENT)
				&& (hasHorizontalScale() == VALUE_ABSENT)
				&& (hasVerticalScale() == VALUE_ABSENT);
		}

		public boolean isValid () {
// These are the minimum four settings required to establish a font.
// Encoding, horizontal scale and vertical scale were added after and
// have defaults.
			return (hasTypeface() == VALUE_VALID)
				&& (hasSize() == VALUE_VALID)
				&& (hasWeight() == VALUE_VALID)
				&& (hasItalic() == VALUE_VALID);
		}

		private void initialize () {
			moSize = new UnitSpan (UnitSpan.POINTS_1K, -1);
			mnWeight = FontInfo.WEIGHT_UNKNOWN;
			mbItalic = false;
			mnHorizontalScale = -1.0;
			mnVerticalScale = -1.0;

			mbTypefacePresent = false;
			mbEncodingPresent = false;
			mbSizePresent = false;
			mbWeightPresent = false;
			mbItalicPresent = false;
			mbHorizontalScalePresent = false;
			mbVerticalScalePresent = false;
		}
	}

/**
 * Don't know why this is needed; it's a private class.
 * @exclude from published api.
 */
	private static class AttrExtra {
		TextMeasurement[] moUnit;
		int[] moInt;
		boolean[] moBool;	// TODO: reconcile with TextAttr.mbBoolValue
		double[] moFloat;
		TextBaselineShift moShift;
		TextAttr mpoOriginalAttr;
		TextStream mpoLeaderContent;

		AttrExtra () {
			initialize();
		}

		void initialize () {
			moUnit = new TextMeasurement [MAX_UNIT_INDEX];		// TODO: allocate these only if necessary
			moInt = new int [MAX_INT_INDEX];
			moFloat = new double [MAX_FLOAT_INDEX];
			moBool = new boolean [MAX_BOOL_INDEX];

			int i;

			for (i = 0; i < MAX_UNIT_INDEX; i++) {
				moUnit[i] = TextMeasurement.zero();
			}
			for (i = 0; i < MAX_INT_INDEX; i++) {
				moInt[i] = TextAttr.getDefaultIntValue (i);
			}
			for (i = 0; i < MAX_FLOAT_INDEX; i++) {
				moFloat[i] = 1;
			}
			for (i = 0; i < MAX_BOOL_INDEX; i++) {
				moBool[i] = false;
			}
		}

		public void setLeaderContent (TextStream oLeaderContent, FontService poFontService) {
			clearLeaderContent();
			mpoLeaderContent = new TextStream (oLeaderContent);
			mpoLeaderContent.fontService (poFontService);
		}

		public void clearLeaderContent () {
			mpoLeaderContent = null;
		}

		void copyFrom (AttrExtra oSource) {
			int i;
			for (i = 0; i < MAX_UNIT_INDEX; i++) {
				moUnit[i] = oSource.moUnit[i];
			}
			for (i = 0; i < MAX_INT_INDEX; i++) {
				moInt[i] = oSource.moInt[i];
			}
			for (i = 0; i < MAX_FLOAT_INDEX; i++) {
				moFloat[i] = oSource.moFloat[i];
			}
			for (i = 0; i < MAX_BOOL_INDEX; i++) {
				moBool[i] = oSource.moBool[i];
			}
			moShift = oSource.moShift;
			mpoOriginalAttr = oSource.mpoOriginalAttr;
			if (oSource.mpoLeaderContent != null) {
				setLeaderContent (oSource.mpoLeaderContent, oSource.mpoLeaderContent.fontService());
			} else {
				clearLeaderContent();
			}
		}
	}

	private static class ReconcileInfo {
		final FontInstance mFontInstance;
		final String mOriginalTypeface;

		ReconcileInfo (FontInstance fontInstance, String originalTypeface) {
			mFontInstance = fontInstance;
			mOriginalTypeface = originalTypeface;
		}
	}

	public static final int JUST_V_UNKNOWN = 0;
	public static final int JUST_V_TOP = 1;
	public static final int JUST_V_MIDDLE = 2;
	public static final int JUST_V_BOTTOM = 3;
	public static final int JUST_H_UNKNOWN = 4;
	public static final int JUST_H_LEFT = 5;
	public static final int JUST_H_CENTRE = 6;
	public static final int JUST_H_RIGHT = 7;
	public static final int JUST_H_SPREAD = 8;
	public static final int JUST_H_SPREAD_ALL = 9;
	public static final int JUST_H_RADIX = 10;
	public static final int JUST_H_COMB_LEFT = 11;
	public static final int JUST_H_COMB_CENTRE = 12;
	public static final int JUST_H_COMB_RIGHT = 13;

	public static final int DIGITS_LOCALE = 0;
	public static final int DIGITS_ARABIC = 1;
	public static final int DIGITS_INDIC = 2;

	public static final int DIRECTION_NEUTRAL = 0;
	public static final int DIRECTION_LTR = 1;
	public static final int DIRECTION_RTL = 2;

	public static final int ORIENTATION_HORIZONTAL = 0;
	public static final int ORIENTATION_VERTICAL_LTR = 1;
	public static final int ORIENTATION_VERTICAL_RTL = 2;

	public static final int LIGATURE_MINIMUM = 0;
	public static final int LIGATURE_COMMON = 1;

	public static final int HYPHEN_OFF = 0;
	public static final int HYPHEN_PREFERRED = 1;
	public static final int HYPHEN_NORMAL = 2;
	public static final int HYPHEN_ALL = 3;

	public static final int LEADER_PATTERN_SPACE = 0;
	public static final int LEADER_PATTERN_RULE = 1;
	public static final int LEADER_PATTERN_DOTS = 2;
	public static final int LEADER_PATTERN_USE_CONTENT = 3;

	public static final int LEADER_ALIGN_NONE = 0;
	public static final int LEADER_ALIGN_PAGE = 1;

	public static final int RULE_STYLE_NONE = 0;
	public static final int RULE_STYLE_SOLID = 1;
	public static final int RULE_STYLE_DOTTED = 2;
	public static final int RULE_STYLE_DASHED = 3;

	public static final int DEFAULT_HYPH_LEVEL = HYPHEN_OFF;
	public static final int DEFAULT_HYPH_MIN_WORD = 7;
	public static final int DEFAULT_HYPH_MIN_PREFIX = 3;
	public static final int DEFAULT_HYPH_MIN_SUFFIX = 3;
	public static final int DEFAULT_HYPH_MAX_LINES = 2;
	public static final boolean DEFAULT_HYPH_SUPPRESS_NAMES = false;
	public static final boolean DEFAULT_HYPH_SUPPRESS_ACRONYMS = false;
	public static final int DEFAULT_LEADER_PATTERN = LEADER_PATTERN_SPACE;
	public static final int DEFAULT_LEADER_ALIGN = LEADER_ALIGN_NONE;
	public static final int DEFAULT_RULE_STYLE = RULE_STYLE_SOLID;

	private static final int UNIT_SPACING = 0;
	private static final int UNIT_MARGIN_L = 1;
	private static final int UNIT_MARGIN_R = 2;
	private static final int UNIT_RADIX_OFFSET = 3;
	private static final int UNIT_SPECIAL = 4;
	private static final int UNIT_SPACE_BEFORE = 5;
	private static final int UNIT_SPACE_AFTER = 6;
	private static final int UNIT_CHAR_SPACING = 7;
	private static final int UNIT_WORD_SPACING = 8;
	private static final int UNIT_LEADER_PATTERN_WIDTH = 9;
	private static final int UNIT_RULE_THICKNESS = 10;
	private static final int MAX_UNIT_INDEX = 11;

	private static final int INT_RADIX_POS = 0;
	private static final int INT_INVIS_CHAR = 1;
	private static final int INT_DIGITS = 2;
	private static final int INT_DIRECTION = 3;
	private static final int INT_PARA_DIRECTION = 4;
	private static final int INT_LAYOUT_ORIENTATION = 5;
	private static final int INT_LIGATURE = 6;
	private static final int INT_HYPH_LEVEL = 7;
	private static final int INT_HYPH_MIN_WORD = 8;
	private static final int INT_HYPH_MIN_PREFIX = 9;
	private static final int INT_HYPH_MIN_SUFFIX = 10;
	private static final int INT_HYPH_MAX_LINES = 11;
	private static final int INT_LEADER_PATTERN = 12;
	private static final int INT_LEADER_ALIGN = 13;
	private static final int INT_RULE_STYLE = 14;
	private static final int MAX_INT_INDEX = 15;

	private static final int BOOL_ITALIC = 0;
	private static final int BOOL_TRANSPARENT = 1;
	private static final int BOOL_INVISIBLE = 2;
	private static final int BOOL_KERNING = 3;
	private static final int BOOL_HYPH_SUPPRESS_NAMES = 4;
	private static final int BOOL_HYPH_SUPPRESS_ACRONYMS = 5;
	private static final int MAX_BOOL_INDEX = 6;

	private static final int FLOAT_HORIZONTAL_SCALE = 0;
	private static final int FLOAT_VERTICAL_SCALE = 1;
	private static final int MAX_FLOAT_INDEX = 2;
	
	private TextGfxSource moGfxSource;
	private AttrExtra mpoExtra;
	private FontInstance moFontInstance;
	private String msTypeface = "";
	private String msOriginalTypeface = ""; // Original typeface name in the case of a font substitution.
	private String msEncoding = ""; // PA passthru attribute
	private UnitSpan moSize = goDefaultUnitNeg;
	private int miWeight;
	private int meJustifyV = JUST_V_TOP;
	private int meJustifyH = JUST_H_LEFT;
	private TextTabList mpoTabs;
	private String msLocale = "";

	private int mbUnitEnable;
	private int mbIntEnable;
	private int mbFloatEnable;
	private int mbBoolEnable;
	private int mbBoolValue;

	private boolean mbJustifyVEnable;
	private boolean mbJustifyHEnable;
	private boolean mbLocaleEnable;
	private boolean mbBaselineShiftEnable;
	private boolean mbLeaderContentEnable;

	private final static UnitSpan goDefaultUnitNeg = new UnitSpan (UnitSpan.POINTS_1K, -1);
	private final static TextMeasurement goDefaultMeasureNeg = new TextMeasurement (goDefaultUnitNeg);
	private static final String gmsDefaultTypeface = "Courier Std";	// TODO: figure out how these are used
	private static final String gsDefaultEncoding = "iso8859-1";	// TODO: figure out how these are used
	
	public static final TextAttr defaultFull = new TextAttr (true);
	public static final TextAttr defaultEmpty = new TextAttr (false);

/**
 * Default constructor.
 * <p>
 * Populates the text attribute object with all attributes disabled.
 */
	public TextAttr () {
		initialize (false);
	}

/**
 * Constructor with control over defaulting technique.
 * <p>
 * This constructor can behave like the default constructor (all
 * attributes disabled) or it can create the object with attributes
 * enabled and set to default values.
 * @param bDefault - FALSE if attributes are to be disabled; TRUE if
 * they are to be enabled with default values.
 */
	public TextAttr (boolean bDefault) {
		initialize (bDefault);
	}

/**
 * Copy constructor.
 * <p>
 * Copies all attributes and their enabled/disabled status.  This also
 * copies the graphic source connection of the source attribute object.
 * @param oSource - Source attribute object to copy.
 */
	public TextAttr (TextAttr oSource) {
		initialize (false); // set all parms to default before copying
		moGfxSource = oSource.moGfxSource; // note: copy pool and env pointers
		copyFrom (oSource); // optimized copy
	}

/**
 * Copy constructor with graphic source.
 * <p>
 * Copies all attributes and their enabled/disabled status, but
 * reestablishes them in the context of a different graphic source.
 * @param oSource - Source attribute object to copy.
 * @param oGfxSource - Graphic source to use for the copy.
 */
	public TextAttr (TextAttr oSource, TextGfxSource oGfxSource) {
		initialize (false); // set all parms to default before copying
		moGfxSource = oGfxSource;
		if (oSource.moGfxSource != moGfxSource) {
			AttrCopy copier = new AttrCopy (oSource, AttrCopy.COPY_ALL);
			copier.copy (this);
		} else {
			copyFrom (oSource); // optimized copy
		}
	}

/**
 * Obtain graphic source information used by the stream.
 * <p>
 * AXTE collects the various sources and pools of graphic information
 * (text and Gfx attribute pools, and font mapper) into a text graphic
 * source object (TextGfxSource).  This method returns the current
 * graphic source of the stream.
 * @return Constant reference to the text stream's graphic source
 */
	public TextGfxSource gfxSource () {
		return moGfxSource;
	}

/**
 * Change the graphic source used by the stream.
 * <p>
 * AXTE collects the various sources and pools of graphic information
 * (text and Gfx attribute pools, and font mapper) into a text graphic
 * source object (TextGfxSource).  This method allows the caller to
 * change the object's graphic source.
 * @param oNewGfxSource - New graphic source to use for the object.
 */
	public void gfxSource (TextGfxSource oNewGfxSource) {
// Before doing anything, reconcile our font in the new graphic source.
		FontInstance oNewFontInstance = null;
		String sOriginalTypeface = originalTypeface();

		if (moFontInstance != null) {
			oNewFontInstance = moFontInstance;
		} else if ((oNewGfxSource.getFontService() != null) && typefaceEnable() && sizeEnable() && weightEnable() && italicEnable()) {
			FontDesc oFontDesc = new FontDesc (this);
			ReconcileInfo reconcile = reconcileFont (oNewGfxSource.getFontService(), oFontDesc);
			oNewFontInstance = reconcile.mFontInstance;
			sOriginalTypeface = reconcile.mOriginalTypeface;
		}

//// Now we can safely disconnect from the existing pool (using old gfx
//// source).
//		CleanupAllGfx();
//
// Now we can install the new source.
		moGfxSource = oNewGfxSource;
//
//		if ((AttrPool() != null) && GfxTextAttrEnable()) {
//// Reconcile the saved copy in the new attribute pool.
//			UpdateGfxTextAttr (AttrPool().AddAttr (GetGfxTextAttr()));
//		}

// Record any font information.
		if (oNewFontInstance != null) {
			updateFont (oNewFontInstance, sOriginalTypeface);
		}

		if ((mpoExtra != null) && (mpoExtra.mpoLeaderContent != null)) {
			mpoExtra.mpoLeaderContent.fontService (oNewGfxSource.getFontService());
		}
	}

/**
 * Get a pointer to the FontService object used by this stream.
 * <p>
 * The font service object used CoolType font handling to create and
 * look up fonts.
 * A text stream may use a font service instead of a text attribute
 * pool; it may use the font service that comes from the text attribute
 * pool; or it may not use a font service at all, though this may cause
 * limitations with certain drivers.  This method returns a pointer to
 * the font service currently in use.
 * @return Pointer to the font service currently in use.  May be NULL.
 */
	public FontService fontService () {
		return (moGfxSource == null) ? null : moGfxSource.getFontService();
	}

/**
 * Set the FontService object used by this stream.
 * <p>
 * The font service object used CoolType font handling to create and
 * look up fonts.
 * A text stream may use a font service instead of a text attribute
 * pool; it may use the font service that comes from the text attribute
 * pool; or it may not use a font service at all, though this may cause
 * limitations with certain drivers.  This method establishes a new font
 * service for the stream.
 * @param poNewFontService Pointer to new font service to use.	May be
 * NULL.
 * @param bSuppressReconcile - (optional) TRUE to prevent immediate
 * reconcilation of the attribute's font info in the new service; FALSE
 * (default) to have the reconcile occur.
 */
	public void fontService (FontService poNewFontService, boolean bSuppressReconcile) {
		if (poNewFontService == fontService()) {
			return;
		}

		if (bSuppressReconcile) {
			moGfxSource = new TextGfxSource (poNewFontService);
		} else {
//			TextGfxSource oSource (poNewFontService);	// TODO:
//			oSource.Pool (AttrPool());
			gfxSource (new TextGfxSource (poNewFontService));
		}

		if ((mpoExtra != null) && (mpoExtra.mpoLeaderContent != null)) {
			mpoExtra.mpoLeaderContent.fontService (poNewFontService);
		}
	}
	public void fontService (FontService poNewFontService) {
		fontService (poNewFontService, false);
	}

/**
 * Get a pointer to the Gfx attribute pool used by the stream.
 * <p>
 * Besides a Gfx font mapper, the stream may carry a pointer to a Gfx
 * attribute pool object in order to reduce memory consumption for
 * common attributes.  A Gfx attribute pool is a distinct object from a
 * text attribute pool.  This method returns a pointer to the Gfx
 * attribute pool currently in use.
 * @return Pointer to the attribute pool currently in use.	May be NULL.
 */
/*
	public GfxAttrPool AttrPool () {
		return moGfxSource.Pool();
	}
*/

/**
 * Set the Gfx attribute pool to be used by the stream.
 * <p>
 * Besides a Gfx font mapper, the stream may carry a pointer to a Gfx
 * attribute pool object in order to reduce memory consumption for
 * common attributes.  A Gfx attribute pool is a distinct object from a
 * text attribute pool.  This method sets the Gfx attribute pool to be
 * used by the stream.
 * @param poNewPool - Pointer to the Gfx attribute pool to use.  May be
 * NULL.
 */
/*
	public void AttrPool (GfxAttrPool poNewPool) {
		if (poNewAttrPool == AttrPool()) {
			return;
		}

		TextGfxSource oSource (poNewAttrPool);
		oSource.FontService (FontService());
		GfxSource (oSource);
	}
*/

/**
 * Return the font instance value if enabled
 * @return FontInstance value.	NULL if not enabled.
 */
	public FontInstance fontInstance () {
		return moFontInstance;
	}

/**
 * Set and enable the font instance value
 * @param oNewFont - New font instance value.	Note: setting the font
 * instance value to a NULL pointer disables it.
 * @param sOriginalTypeface - Original typeface value for substituted fonts.
 */
	public void fontInstance (FontInstance oNewFont, String sOriginalTypeface) {
		if (oNewFont != moFontInstance) {
			updateFont (oNewFont, sOriginalTypeface);
		}
	}
	public void fontInstance (FontInstance oNewFont) {
		fontInstance (oNewFont, "");
	}

/**
 * Return if the font instance is a substitute font
 */
	public boolean substituteFont () {
		return !StringUtils.isEmpty(msOriginalTypeface);
	}

/**
 * Query whether the font value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean fontEnable () {
		return moFontInstance != null;
	}

/**
 * Enable/disable the font value
 * @param bNewEnable - TRUE if the font value is to be enabled; FALSE if
 * it is to be disabled.
 */
	public void fontEnable (boolean bNewEnable) {
		if (! bNewEnable) {
			updateFont (null, "");
		}
	}

/**
 * Query whether any of the font attributes is enabled
 * @return TRUE if any of typeface, size, weight or italic is enabled;
 * FALSE if all disabled.
 */
	public boolean anyFontEnable () {
		return typefaceEnable() || sizeEnable() || weightEnable() || italicEnable() || horizontalScaleEnable() || verticalScaleEnable();
	}

/**
 * Return the Typeface value if enabled
 * @return Typeface value.
 */
	public String typeface () {
		return msTypeface;
	}

/**
 * Set and enable the typeface value
 * @param sNewTypeface - New typeface value.
 * @param nWeight - bold attribute.
 * @param bItalic - Italicize.
 */
	public void typeface (String sNewTypeface, int nWeight, boolean bItalic, String sEncoding) {
		msTypeface = sNewTypeface; // TBD: this is odd
		FontDesc oFontDesc = new FontDesc();
		oFontDesc.setTypeface (sNewTypeface);
		oFontDesc.setWeight (nWeight);
		oFontDesc.setItalic (bItalic);
		oFontDesc.setEncoding (sEncoding);
		updateFont (oFontDesc);
	}

/**
 * Set and enable the typeface value
 * @param sNewTypeface - New typeface value.
 */
	public void typeface (String sNewTypeface) {
		FontDesc oFontDesc = new FontDesc (sNewTypeface);
		updateFont (oFontDesc);
	}

/**
 * Set and enable one or more of typeface, size, weight and italic.
 * NOTE: This method will not set the corresponding attributes if the parameters are empty or zero.
 * @param sNewTypeface - New typeface value.
 * @param oSize - New size value.
 * @param nWeight - New weight value.
 * @param nItalic - New italic value.
 */
	public void typeface (String sNewTypeface, UnitSpan oSize, int nWeight, int nItalic) {
		FontDesc oFontDesc = new FontDesc();

		if ((sNewTypeface != null) && (sNewTypeface.length() > 0)) {
			oFontDesc.setTypeface (sNewTypeface);
		}
		if ((oSize != null) && (oSize.value() > 0)) {
			oFontDesc.setSize (oSize);
		}
		if (nWeight != FontInfo.WEIGHT_UNKNOWN) {
			oFontDesc.setWeight (nWeight);
		}
		if (nItalic != FontInfo.STYLE_UNKNOWN) {
			oFontDesc.setItalic (nItalic == FontInfo.STYLE_ITALIC);
		}

		updateFont (oFontDesc);
	}

/**
 * Query whether the typeface value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean typefaceEnable () {
		return msTypeface.length() > 0;
	}

/**
 * Enable/disable the typeface value
 * @param bNewEnable - TRUE if the typeface value is to be enabled;
 * FALSE if it is to be disabled.
 */

	public void typefaceEnable (boolean bNewEnable) {
		if (! bNewEnable) {
			typeface ("");
		}
	}

/**
 * Return the OriginalTypeface value
 * @return OriginalTypeface value.
 */
	public String originalTypeface () {
		return msOriginalTypeface;
	}

/**
 * Return the encoding value if enabled
 * @return Encoding value.
 */
	public String encoding () {
		return msEncoding;
	}

/**
 * Set and enable the encoding value
 * @param sNewEncoding - New encoding value.
 */
	public void encoding (String sNewEncoding) {
		FontDesc oFontDesc = new FontDesc();
		oFontDesc.setEncoding (sNewEncoding);
		updateFont (oFontDesc);
	}

/**
 * Query whether the encoding value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean encodingEnable () {
		return msEncoding.length() > 0;
	}

/**
 * Enable/disable the encoding value
 * @param bNewEnable - TRUE if the encoding value is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void encodingEnable (boolean bNewEnable) {
		if (! bNewEnable) {
			encoding ("");
		}
	}

/**
 * Return the size value if enabled
 * @return Size value.
 */
	public UnitSpan size () {
		return moSize;
	}

/**
 * Set and enable the size value
 * @param oNewSize - New size value.
 */
	public void size (UnitSpan oNewSize) {
		FontDesc oFontDesc = new FontDesc (oNewSize);
		updateFont (oFontDesc);
	}

/**
 * Query whether the size value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean sizeEnable () {
		return moSize.value() >= 0;
	}

/**
 * Enable/disable the size value
 * @param bNewEnable - TRUE if the size value is to be enabled; FALSE if
 * it is to be disabled.
 */
	public void sizeEnable (boolean bNewEnable) {
		size (new UnitSpan (UnitSpan.defaultUnits(), -1));	// TODO: shouldn't there be a static single instance?
	}

/**
 * Return the weight value if enabled
 * @return Weight value.
 */
	public int weight () {
		return miWeight;
	}

/**
 * Set and enable the weight value
 * @param iNewWeight - New weight value.
 */
	public void weight (int iNewWeight) {
		FontDesc oFontDesc = new FontDesc (iNewWeight);
		updateFont (oFontDesc);
	}

/**
 * Query whether the weight value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean weightEnable () {
		return miWeight != FontInfo.WEIGHT_UNKNOWN;
	}

/**
 * Enable/disable the weight value
 * @param bNewEnable - TRUE if the weight value is to be enabled; FALSE
 * if it is to be disabled.
 */
	public void weightEnable (boolean bNewEnable) {
		weight (FontInfo.WEIGHT_UNKNOWN);
	}

/**
 * Return the italic value if enabled
 * @return Italic value.
 */
	public boolean italic () {
		return getBoolValue (BOOL_ITALIC);
	}

/**
 * Set and enable the italic value
 * @param bNewItalic - New italic value.
 */
	public void italic (boolean bNewItalic) {
		FontDesc oFontDesc = new FontDesc();
		oFontDesc.setItalic (bNewItalic);
		updateFont (oFontDesc);
	}

/**
 * Query whether the italic value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean italicEnable () {
		return isBoolEnabled (BOOL_ITALIC);
	}

/**
 * Enable/disable the italic value
 * @param bNewEnable - TRUE if the italic value is to be enabled; FALSE
 * if it is to be disabled.
 */
	public void italicEnable (boolean bNewEnable) {
		enableBool (BOOL_ITALIC, bNewEnable);
		updateFont();
	}

/**
 * Query the collected graphic text attributes
 * @return - The collected graphic text attributes (GfxTextAttr)
 * represented in this object.
 */
	public GFXTextAttr gfxTextAttr () {
		return getGfxTextAttr();
	}

/**
 * Return the transparent flag if enabled
 * @return Transparent flag.
 */
	public boolean transparent () {
		return getBoolValue (BOOL_TRANSPARENT);
	}

/**
 * Set and enable the transparent flag
 * @param bNewTransparent - New transparent flag.
 */
	public void transparent (boolean bNewTransparent) {
		setBoolValue (BOOL_TRANSPARENT, bNewTransparent);
	}

/**
 * Query whether the transparent flag is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean transparentEnable () {
		return isBoolEnabled (BOOL_TRANSPARENT);
	}

/**
 * Enable/disable the transparent flag
 * @param bNewEnable - TRUE if the transparent flag is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void transparentEnable (boolean bNewEnable) {
		enableBool (BOOL_TRANSPARENT, bNewEnable);
	}

/**
 * Return the line spacing if enabled
 * @return Line spacing.
 */
	public TextMeasurement spacing () {
		return getUnitValue (UNIT_SPACING);
	}

/**
 * Set and enable the line spacing
 * @param oNewSpacing - New line spacing.
 */
	public void spacing (TextMeasurement oNewSpacing) {
		setUnitValue (UNIT_SPACING, oNewSpacing);
	}

/**
 * Query whether the line spacing is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean spacingEnable () {
		return isUnitEnabled (UNIT_SPACING);
	}

/**
 * Enable/disable the line spacing
 * @param bNewEnable - TRUE if the line spacing is to be enabled; FALSE
 * if it is to be disabled.
 */
	public void spacingEnable (boolean bNewEnable) {
		enableUnit (UNIT_SPACING, bNewEnable);
	}

/**
 * Return the left margin if enabled
 * @return Left margin.
 */
	public TextMeasurement marginL () {
		return getUnitValue (UNIT_MARGIN_L);
	}

/**
 * Set and enable the left margin
 * @param oNewMarginL - New left margin.
 */
	public void marginL (TextMeasurement oNewMarginL) {
		setUnitValue (UNIT_MARGIN_L, oNewMarginL);
	}

/**
 * Query whether the left margin is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean marginLEnable () {
		return isUnitEnabled (UNIT_MARGIN_L);
	}

/**
 * Enable/disable the left margin
 * @param bNewEnable - TRUE if the left margin is to be enabled; FALSE
 * if it is to be disabled.
 */
	public void marginLEnable (boolean bNewEnable) {
		enableUnit (UNIT_MARGIN_L, bNewEnable);
	}

/**
 * Return the right margin if enabled
 * @return Right margin.
 */
	public TextMeasurement marginR () {
		return getUnitValue (UNIT_MARGIN_R);
	}

/**
 * Set and enable the right margin
 * @param oNewMarginR - New right margin.
 */
	public void marginR (TextMeasurement oNewMarginR) {
		setUnitValue (UNIT_MARGIN_R, oNewMarginR);
	}

/**
 * Query whether the right margin is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean marginREnable () {
		return isUnitEnabled (UNIT_MARGIN_R);
	}

/**
 * Enable/disable the right margin
 * @param bNewEnable - TRUE if the right margin is to be enabled; FALSE
 * if it is to be disabled.
 */
	public void marginREnable (boolean bNewEnable) {
		enableUnit (UNIT_MARGIN_R, bNewEnable);
	}

/**
 * Return the vertical justification if enabled
 * @return vertical justification.
 */
	public int justifyV () {
		return meJustifyV;
	}

/**
 * Set and enable the vertical justification
 * @param eNewJustifyV - New vertical justification.
 */
	public void justifyV (int eNewJustifyV) {
		meJustifyV = eNewJustifyV;
		mbJustifyVEnable = (meJustifyV != JUST_V_UNKNOWN);
	}

/**
 * Query whether the vertical justification is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean justifyVEnable () {
		return mbJustifyVEnable;
	}

/**
 * Enable/disable the vertical justification
 * @param bNewEnable - TRUE if the vertical justification is to be
 * enabled; FALSE if it is to be disabled.
 */
	public void justifyVEnable (boolean bNewEnable) {
		mbJustifyVEnable = bNewEnable;
		if (! mbJustifyVEnable) {
			meJustifyV = JUST_V_UNKNOWN;
		}
	}

/**
 * Return the horizontal justification if enabled
 * @return Horizontal justification.
 */
	public int justifyH () {
		return meJustifyH;
	}

/**
 * Set and enable the horizontal justification
 * @param eNewJustifyH - New horizontal justification.
 */
	public void justifyH (int eNewJustifyH) {
		meJustifyH = eNewJustifyH;
		mbJustifyHEnable = (meJustifyH != JUST_H_UNKNOWN);
	}

/**
 * Query whether the horizontal justification is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean justifyHEnable () {
		return mbJustifyHEnable;
	}

/**
 * Enable/disable the horizontal justification
 * @param bNewEnable - TRUE if the horizontal justification is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void justifyHEnable (boolean bNewEnable) {
		mbJustifyHEnable = bNewEnable;
		if (! mbJustifyHEnable) {
			meJustifyH = JUST_H_UNKNOWN;
		}
	}

/**
 * Return the radix offset value if enabled
 * @return Radix offset value.
 */
	public TextMeasurement radixOffset () {
		return getUnitValue (UNIT_RADIX_OFFSET);
	}

/**
 * Set and enable the radix offset value
 * @param oNewRadixOffset - New radix offset value.
 */
	public void radixOffset (TextMeasurement oNewRadixOffset) {
		setUnitValue (UNIT_RADIX_OFFSET, oNewRadixOffset);
	}

/**
 * Query whether the radix offset value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean radixOffsetEnable () {
		return isUnitEnabled (UNIT_RADIX_OFFSET);
	}

/**
 * Enable/disable the radix offset value
 * @param bNewEnable - TRUE if the radix offset value is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void radixOffsetEnable (boolean bNewEnable) {
		enableUnit (UNIT_RADIX_OFFSET, bNewEnable);
	}

/**
 * Get the radix position value
 * @return the radix position value.
 */
	public int radixPos () {
		return (int) getIntValue (INT_RADIX_POS);
	}

/**
 * Set the radix position value
 * @param nNewRadixPos - New radix position value.
 */
	public void radixPos (int nNewRadixPos) {
		setIntValue (INT_RADIX_POS, nNewRadixPos);
	}

/**
 * Query whether the radix position value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean radixPosEnable () {
		return isIntEnabled (INT_RADIX_POS);
	}

/**
 * Enable/disable the radix position value
 * @param bNewEnable - TRUE if the radix offset value is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void radixPosEnable (boolean bNewEnable) {
		enableInt (INT_RADIX_POS, bNewEnable);
	}

/**
 * Return the tab list if enabled
 * @return tab list.
 */
	public TextTabList tabs () {
		return (mpoTabs == null) ? TextTabList.DEFAULT_TAB_LIST : mpoTabs;
	}

/**
 * Set and enable the tab list
 * @param oNewTabs - New tab list.
 */
	public void tabs (TextTabList oNewTabs) {
		if (oNewTabs == TextTabList.DEFAULT_TAB_LIST) {
			mpoTabs = TextTabList.DEFAULT_TAB_LIST;
		} else {
			mpoTabs = new TextTabList (oNewTabs);
		}
	}

/**
 * Query whether the tab list is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean tabsEnable () {
		return mpoTabs != null;
	}

/**
 * Enable/disable the tab list
 * @param bNewEnable - TRUE if the tab list is to be enabled; FALSE if
 * it is to be disabled.
 */
	public void tabsEnable (boolean bNewEnable) {
		if (bNewEnable) {
			if (mpoTabs == null) {
				mpoTabs = TextTabList.DEFAULT_TAB_LIST;
			}
		} else {
			mpoTabs = null;
		}
	}

/**
 * Return the special first line handling value if enabled
 * @return Special first line handling value.
 */
	public TextMeasurement special () {
		return getUnitValue (UNIT_SPECIAL);
	}

/**
 * Set and enable the special first line handling value
 * @param oNewSpecial - New special first line handling value.
 */
	public void special (TextMeasurement oNewSpecial) {
		setUnitValue (UNIT_SPECIAL, oNewSpecial);
	}

/**
 * Query whether the special first line handling value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean specialEnable () {
		return isUnitEnabled (UNIT_SPECIAL);
	}

/**
 * Enable/disable the special first line handling value
 * @param bNewEnable - TRUE if the special first line handling value is
 * to be enabled; FALSE if it is to be disabled.
 */
	public void specialEnable (boolean bNewEnable) {
		enableUnit (UNIT_SPECIAL, bNewEnable);
	}

/**
 * Return the vertical space before paragraph value if enabled
 * @return Vertical space before paragraph.
 */
	public TextMeasurement spaceBefore () {
		return getUnitValue (UNIT_SPACE_BEFORE);
	}

/**
 * Set and enable the vertical space before paragraph value
 * @param oNewSpaceBefore - New vertical space before paragraph value.
 */
	public void spaceBefore (TextMeasurement oNewSpaceBefore) {
		setUnitValue (UNIT_SPACE_BEFORE, oNewSpaceBefore);
	}

/**
 * Query whether the vertical space before paragraph value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean spaceBeforeEnable () {
		return isUnitEnabled (UNIT_SPACE_BEFORE);
	}

/**
 * Enable/disable the vertical space before paragraph value
 * @param bNewEnable - TRUE if the vertical space before paragraph value
 * is to be enabled; FALSE if it is to be disabled.
 */
	public void spaceBeforeEnable (boolean bNewEnable) {
		enableUnit (UNIT_SPACE_BEFORE, bNewEnable);
	}

/**
 * Return the vertical space after paragraph value if enabled
 * @return Vertical space after paragraph value.
 */
	public TextMeasurement spaceAfter () {
		return getUnitValue (UNIT_SPACE_AFTER);
	}

/**
 * Set and enable the vertical space after paragraph value
 * @param oNewSpaceAfter - New vertical space after paragraph value.
 */
	public void spaceAfter (TextMeasurement oNewSpaceAfter) {
		setUnitValue (UNIT_SPACE_AFTER, oNewSpaceAfter);
	}

/**
 * Query whether the vertical space after paragraph value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean spaceAfterEnable () {
		return isUnitEnabled (UNIT_SPACE_AFTER);
	}

/**
 * Enable/disable the vertical space after paragraph value
 * @param bNewEnable - TRUE if the vertical space after paragraph value
 * is to be enabled; FALSE if it is to be disabled.
 */
	public void spaceAfterEnable (boolean bNewEnable) {
		enableUnit (UNIT_SPACE_AFTER, bNewEnable);
	}

/**
 * Return the invisible flag if enabled
 * @return Invisible flag.
 */
	public boolean invisible () {
		return getBoolValue (BOOL_INVISIBLE);
	}

/**
 * Set and enable the invisible flag
 * @param bNewInvisible - New invisible flag.
 */
	public void invisible (boolean bNewInvisible) {
		setBoolValue (BOOL_INVISIBLE, bNewInvisible);
	}

/**
 * Query whether the invisible flag is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean invisibleEnable () {
		return isBoolEnabled (BOOL_INVISIBLE);
	}

/**
 * Enable/disable the invisible flag
 * @param bNewEnable - TRUE if the invisible flag is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void invisibleEnable (boolean bNewEnable) {
		enableBool (BOOL_INVISIBLE, bNewEnable);
	}

/**
 * Return the invisible character if enabled
 * @return Invisible character.
 */
	public char invisChar () {
		return (char) getIntValue (INT_INVIS_CHAR);
	}

/**
 * Set and enable the invisible character
 * @param cNewInvisChar - New invisible character.
 */
	public void invisChar (char cNewInvisChar) {
		setIntValue (INT_INVIS_CHAR, (int) cNewInvisChar);
	}

/**
 * Query whether the invisible character is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean invisCharEnable () {
		return isIntEnabled (INT_INVIS_CHAR);
	}

/**
 * Enable/disable the invisible character
 * @param bNewEnable - TRUE if the invisible character is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void invisCharEnable (boolean bNewEnable) {
		enableInt (INT_INVIS_CHAR, bNewEnable);
	}

/**
 * Return the baseline shift if enabled
 * @return Baseline shift.
 */
	public TextBaselineShift baselineShift () {
		return ((mpoExtra == null) || (mpoExtra.moShift == null))
			? TextBaselineShift.DEFAULT_SHIFT
			: mpoExtra.moShift;
	}

/**
 * Set and enable the baseline shift
 * @param oNewShift - New baseline shift.
 */
	public void baselineShift (TextBaselineShift oNewShift) {
		if (oNewShift != baselineShift()) {
			needExtra();
			mpoExtra.moShift = oNewShift;
		}
		mbBaselineShiftEnable = oNewShift != null;
	}

/**
 * Query whether the baseline shift is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean baselineShiftEnable () {
		return mbBaselineShiftEnable;
	}

/**
 * Enable/disable the baseline shift
 * @param bNewEnable - TRUE if the baseline shift is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void baselineShiftEnable (boolean bNewEnable) {
		mbBaselineShiftEnable = bNewEnable;
	}

/**
 * Flatten a measurement in the context of this attribute.
 * <p>
 * Given a text measurement, this method flattens it to an absolute unit
 * span.  If the given measurement repsents a length, its length value
 * is returned unchanged.  Otherwise, its relative value is flattened in
 * the context of font instance if not null, or the font size otherwise.
 * @param oMeasurement - Measurement to be flattened.
 * @return The flattened value.
 */
	public UnitSpan flattenMeasurement (TextMeasurement oMeasurement) {
		if (fontEnable() && (moFontInstance != null)) {
			return oMeasurement.flatten (moFontInstance);
		} else {
			return oMeasurement.flatten (moSize);
		}
	}

/**
 * Flatten relative measurements to absolute values.
 * <p>
 * This method flattens all relative values in the attribute.  Potential
 * relative values are those returned as TextMeasurement, as well as
 * subscript and superscript baseline shifts.  If the potentially
 * relative values are all absolute, the method does nothing.
 * Flattening a subscript or superscript baseline shift yields an
 * absolute displacement, with a corresponding adjustment to the font
 * size.
 * @return - True if any change took place; false if this object is
 * unaltered.
 */
	public boolean flatten () {
		if (! sizeEnable()) {
			return false;
		}
		UnitSpan oFontSize = size();
		if (oFontSize.value() == 0) {
			return false;
		}

		boolean bFlattened = false;

		if (canFlattenBaselineShift()) {
			TextBaselineShift oOldShift = baselineShift();
			UnitSpan oOldSize = oFontSize;
			UnitSpan[] flat = oOldShift.flatten (UnitSpan.ZERO, oOldSize, oFontSize);
			size (flat[1]);
			baselineShift (new TextBaselineShift (flat[0]));
			bFlattened = true;
		}

		FontInstance oFontInstance = null;
		if (fontEnable()) {
			oFontInstance = fontInstance();
		}

		if (mpoExtra != null) {
			for (int i = 0; i < MAX_UNIT_INDEX; i++) {
				TextMeasurement oMeasurement = mpoExtra.moUnit[i];
				if (oMeasurement.getType() != TextMeasurement.TYPE_LENGTH) {
					UnitSpan oFlat;
					if (oFontInstance == null) {
						oFlat = oMeasurement.flatten (oFontSize);
					} else {
						oFlat = oMeasurement.flatten (oFontInstance);
					}
					mpoExtra.moUnit[i] = new TextMeasurement (oFlat);
				}
			}
		}

		return bFlattened;
	}

/**
 * Perform flattening without affecting this object.
 * <p>
 * This method returns a new, flattened text attribute object.	If
 * flattening would have no effect, the method simply returns NULL.
 * Otherwise it clones this object and flattens it.  The font size must
 * also be enabled and be non-zero for this method to do anything.
 * </p>
 * <p>
 * Note that the resulting attribute object caches a pointer to this
 * object and assumes that it is reference counted.  Therefore, the
 * caller must employ reference counting on this object to ensure that
 * it isn't deleted too soon.
 * </p>
 * @return NULL if no flattening took place, or a new flattened text
 * attribute object. with flattened baseline shift.
 */
	public TextAttr conditionalFlatten () {
		if ((! sizeEnable()) || (size().value() == 0)) {
			return null;
		}

		boolean bFlatten = false;
		if (canFlattenBaselineShift()) {
			bFlatten = true;
		} else if (mpoExtra != null) {
			for (int i = 0; i < MAX_UNIT_INDEX; i++) {
				if (mpoExtra.moUnit[i].getType() != TextMeasurement.TYPE_LENGTH) {
					bFlatten = true;
					break;
				}
			}
		}
		if (! bFlatten) {
			return null;
		}

		TextAttr poNewAttr = new TextAttr (this);
		poNewAttr.flatten();
		poNewAttr.needExtra();
		poNewAttr.mpoExtra.mpoOriginalAttr = this;

		return poNewAttr;
	}

/**
 * Return the original attribute object from one that has had its
 * baseline shift flattened.
 * <p>
 * If a client is processing layout, it will see flattened baseline
 * shifts.	If it wishes to work with the more abstract Subscript and
 * Superscript designations, it can track the original attribute object
 * through the return value of this method.
 * @return Pointer to original attribute object if this object is the
 * result of flattening a baseline shift; NULL otherwise.
 */
	public TextAttr getOriginalAttr () {
		return (mpoExtra == null) ? null : mpoExtra.mpoOriginalAttr;
	}

/**
 * Return the locale value if enabled
 * @return Locale value.
 */
	public String locale () {
		return msLocale;
	}

/**
 * Set and enable the locale value
 * @param sNewLocale - New locale value.
 */
	public void locale (String sNewLocale) {
		msLocale = sNewLocale;
		mbLocaleEnable = true;
	}

/**
 * Query whether the locale value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean localeEnable () {
		return mbLocaleEnable;
	}

/**
 * Enable/disable the locale value
 * @param bNewEnable - TRUE if the locale value is to be enabled; FALSE
 * if it is to be disabled.
 */
	public void localeEnable (boolean bNewEnable) {
		mbLocaleEnable = bNewEnable;
	}

/**
 * Resolve the locale for the text attribute object.
 * <p>
 * The text attribute object may or may not actually specify a locale.
 * This method returns a locale name that the caller can use.  If the
 * text attribute object specifies a locale (enabled and non-empty
 * string), that value is returned.  If locale is enabled but empty, the
 * ambient locale is returned.	If locale is disabled, the locale
 * "en_US" is returned.
 * @return Actual locale string.
 */
	public String actualLocale () {
		String sLocale;

		if (! mbLocaleEnable) {
			sLocale = "en_US";
		} else if (msLocale.length() == 0) {
			sLocale = LcLocale.getLocale();
		} else {
			sLocale = msLocale;
		}

		return sLocale;
	}

/**
 * Return the radix text string, resolved by locale.
 * <p>
 * Returns the radix text string (e.g., period in en_US) resolved by
 * locale.	Uses the value returned by ActualLocale() to obtain the
 * locale that determines the radix text string.
 * @return Radix text string.
 */
	public String radixText () {
		String sRadixText;
		LcLocale oLocale = new LcLocale (actualLocale());			// TODO: is new necessary?

		if (oLocale.isValid()) {
			LcData oLocaleData = new LcData (oLocale.getName());	// TODO: is new necessary?
			sRadixText = oLocaleData.getRadixSymbol();
		} else {
			sRadixText = ".";
		}

		return sRadixText;
	}

/**
 * Return the radix character, resolved by locale.
 * <p>
 * AXTE currently supports radix alignment on a single character.  This
 * method simply returns the first character of the string returned by
 * the RadixText() method.
 * @return Radix character.
 */
	public char radixChar () { // TBD: need to determine whether any locale has multi-char radix symbol
		String sRadixText = radixText();
		char c = sRadixText.charAt (0);		// TODO: use UniCharIterator?
		return (c == '\0') ? '.' : c;
	}

	public int digits () {
		return getIntValue (INT_DIGITS);
	}

	public void digits (int eNewDigits) {
		setIntValue (INT_DIGITS, eNewDigits);
	}

	public boolean digitsEnable () {
		return isIntEnabled (INT_DIGITS);
	}

	public void digitsEnable (boolean bNewEnable) {
		enableInt (INT_DIGITS, bNewEnable);
	}

/**
 * Return the direction for neutral characters.
 * <p>
 * Many characters (e.g., some punctuation, spaces) are directionally
 * neutral, that is, their direction (LTR or RTL) is determined by
 * context at render time.	Some times this is inappropriate, for
 * example, part numbers that include punctuation.	This attribute value
 * can force directionality to LTR or RTL for neutral characters.
 * @return Current neutral character directionality setting.
 */
	public int direction () {
		return getIntValue (INT_DIRECTION);
	}

/**
 * Set the direction for neutral characters.
 * <p>
 * Please see the description of the first overload for the purpose of
 * this attribute.	This method allows the caller to change the neutral
 * character directionality.
 * @param eNewDirection - New directionality setting.
 */
	public void direction (int eNewDirection) {
		setIntValue (INT_DIRECTION, eNewDirection);
	}

/**
 * Query whether the neutral character direction value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean directionEnable () {
		return isIntEnabled (INT_DIRECTION);
	}

/**
 * Enable/disable the neutral character direction value
 * @param bNewEnable - TRUE if the neutral character direction value is
 * to be enabled; FALSE if it is to be disabled.
 */
	public void directionEnable (boolean bNewEnable) {
		enableInt (INT_DIRECTION, bNewEnable);
	}

/**
 * Return the paragraph direction.
 * <p>
 * The paragraph direction determines the higher-level ordering of mixed
 * direction runs.	If neutral, it is gleaned from the locale.	The
 * paragraph direction is also used to set vertical writing.
 * @return Current paragraph direction in effect.
 */
	public int paraDirection () {
		return getIntValue (INT_PARA_DIRECTION);
	}

/**
 * Change the paragraph direction.
 * @param eNewDirection - New paragraph direction for this attribute
 * object.
 */
	public void paraDirection (int eNewDirection) {
		setIntValue (INT_PARA_DIRECTION, eNewDirection);
	}

/**
 * Query whether the paragraph character direction value is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean paraDirectionEnable () {
		return isIntEnabled (INT_PARA_DIRECTION);
	}

/**
 * Enable/disable the paragraph character direction value.
 * @param bNewEnable - TRUE if the paragraph character direction value
 * is to be enabled; FALSE if it is to be disabled.
 */
	public void paraDirectionEnable (boolean bNewEnable) {
		enableInt (INT_PARA_DIRECTION, bNewEnable);
	}

/**
 * Return the layout orientation in effect.
 * <p>
 * The layout orientation specifies horizontal or vertical writing.
 * This is a paragraph-level attribute.  Because orientation cannot
 * change mid-frame, changes to orientation in rich text may not be
 * honoured.  Indeed, in the initial implementation, the orientation in
 * effect at the start of the stream affects all frames for that stream.
 * @return Layout orientation.	Default is horizontal.
 */
	public int layoutOrientation () {
		return getIntValue (INT_LAYOUT_ORIENTATION);
	}

/**
 * Set the layout orientation.
 * <p>
 * For more information on layout orientation, see the first overload of
 * this method name.
 * @param eNewLayoutOrientation - New orientation for this attribute object.
 */
	public void layoutOrientation (int eNewLayoutOrientation) {
		setIntValue (INT_LAYOUT_ORIENTATION, eNewLayoutOrientation);
	}

/**
 * Query whether the layout orientation is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean layoutOrientationEnable () {
		return isIntEnabled (INT_LAYOUT_ORIENTATION);
	}

/**
 * Enable the layout orientation value.
 * @param bNewEnable - TRUE if the layout orientation value is to be
 * enabled; FALSE if it is to be disabled.
 */
	public void layoutOrientationEnable (boolean bNewEnable) {
		enableInt (INT_LAYOUT_ORIENTATION, bNewEnable);
	}

/**
 * Return the ligature level.
 * <p>
 * For compatibility with other products and versions, there is some
 * control over the application of ligatures.  Currently, there are two
 * levels: minimal for repeatable forms processing, and common for text
 * documents with improved appearance.
 * @return Current ligature level.
 */
	public int ligature () {
		return getIntValue (INT_LIGATURE);
	}

/**
 * Set the ligature level.
 * <p>
 * Please see the description of the first overload for the purpose of
 * this attribute.	This method allows the caller to change the ligature
 * level.
 * @param eNewLigature - New ligature setting.
 */
	public void ligature (int eNewLigature) {
		setIntValue (INT_LIGATURE, eNewLigature);
	}

/**
 * Query whether the ligature level is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean ligatureEnable () {
		return isIntEnabled (INT_LIGATURE);
	}

/**
 * Enable/disable the ligature level.
 * @param bNewEnable - TRUE if the ligature level is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void ligatureEnable (boolean bNewEnable) {
		enableInt (INT_LIGATURE, bNewEnable);
	}

/**
 * Returns the character spacing setting for this text attribute object.
 * <p>
 * The character spacing is added to each character's default spacing
 * when laying out text.
 * @return Character spacing value currently in effect for this
 * attribute object.  The default value is zero.
 */
	public TextMeasurement charSpacing () {
		return getUnitValue (UNIT_CHAR_SPACING);
	}

/**
 * Set a new character spacing value for this attribute object.
 * <p>
 * Note that setting a value with this method implicitly enables the
 * character spacing setting.  For more information, see method
 * CharSpacingEnable().
 * @param nNewCharSpacing - New character spacing value.  Negative
 * values are allowed.
 */
	public void charSpacing (TextMeasurement nNewCharSpacing) {
		setUnitValue (UNIT_CHAR_SPACING, nNewCharSpacing);
	}

/**
 * Query whether the character spacing setting is enabled in this text
 * attribute object.
 * <p>
 * If enabled, the value returned by CharSpacing() is valid.  If not
 * enabled, the character spacing is effectively unknown.
 * @return True if character spacing is currently enabled; false if not.
 */
	public boolean charSpacingEnable () {
		return isUnitEnabled (UNIT_CHAR_SPACING);
	}

/**
 * Enable or disable the character spacing setting in this attribute
 * object.
 * @param bNewEnable - True if the character spacing setting is to be
 * enabled; false if it is to be disabled.	Enabling a previously
 * disabled setting causes it to have a default value (one).
 */
	public void charSpacingEnable (boolean bNewEnable) {
		enableUnit (UNIT_CHAR_SPACING, bNewEnable);
	}

/**
 * Returns the word spacing setting for this text attribute object.
 * <p>
 * The word spacing is added to each internal space in a line when
 * laying out text.
 * @return Word spacing value currently in effect for this attribute
 * object.	The default value is zero.
 */
	public TextMeasurement wordSpacing () {
		return getUnitValue (UNIT_WORD_SPACING);
	}

/**
 * Set a new word spacing value for this attribute object.
 * <p>
 * Note that setting a value with this method implicitly enables the
 * word spacing setting.  For more information, see method
 * WordSpacingEnable().
 * @param nNewWordSpacing - New word spacing value.  Negative values are
 * invalid.
 */
	public void wordSpacing (TextMeasurement nNewWordSpacing) {
		setUnitValue (UNIT_WORD_SPACING, nNewWordSpacing);
	}

/**
 * Query whether the word spacing setting is enabled in this text
 * attribute object.
 * <p>
 * If enabled, the value returned by WordSpacing() is valid.  If not
 * enabled, the word spacing is effectively unknown.
 * @return True if word spacing is currently enabled; false if not.
 */
	public boolean wordSpacingEnable () {
		return isUnitEnabled (UNIT_WORD_SPACING);
	}

/**
 * Enable or disable the word spacing setting in this attribute object.
 * @param bNewEnable - True if the word spacing setting is to be
 * enabled; false if it is to be disabled.	Enabling a previously
 * disabled setting causes it to have a default value (one).
 */
	public void wordSpacingEnable (boolean bNewEnable) {
		enableUnit (UNIT_WORD_SPACING, bNewEnable);
	}

/**
 * Returns the kerning setting for this text attribute object.
 * @return Indicates whether kerning is currently in effect for this
 * attribute object.  The default value is false.
 */
	public boolean kerning () {
		return getBoolValue (BOOL_KERNING);
	}

/**
 * Set a new kerning value for this attribute object.
 * <p>
 * Note that setting a value with this method implicitly enables the
 * kerning setting.  For more information, see method KerningEnable().
 * @param bNewKerning - New kerning value.	True to enable kerning;
 * false to disable it.
 */
	public void kerning (boolean bNewKerning) {
		setBoolValue (BOOL_KERNING, bNewKerning);
	}

/**
 * Query whether the kerning setting is enabled in this text attribute
 * object.
 * <p>
 * If enabled, the value returned by Kerning() is valid.  If not
 * enabled, the kerning is effectively unknown.
 * @return True if kerning is currently enabled; false if not.
 */
	public boolean kerningEnable () {
		return isBoolEnabled (BOOL_KERNING);
	}

/**
 * Enable or disable the kerning setting in this attribute object.
 * @param bNewEnable - True if the kerning setting is to be enabled;
 * false if it is to be disabled.  Enabling a previously disabled
 * setting causes it to have a default value (false).
 */
	public void kerningEnable (boolean bNewEnable) {
		enableBool (BOOL_KERNING, bNewEnable);
	}

/**
 * Return the hyphenation level if enabled
 * @return Hyphenation level.
 */
	public int hyphLevel () {
		return getIntValue (INT_HYPH_LEVEL);
	}

/**
 * Set and enable the hyphenation level
 * @param eNewHyphLevel - New hyphenation level.
 */
	public void hyphLevel (int eNewHyphLevel) {
		setIntValue (INT_HYPH_LEVEL, eNewHyphLevel);
	}

/**
 * Query whether the hyphenation level is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean hyphLevelEnable () {
		return isIntEnabled (INT_HYPH_LEVEL);
	}

/**
 * Enable/disable the hyphenation level
 * @param bNewEnable - TRUE if the hyphenation level is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void hyphLevelEnable (boolean bNewEnable) {
		enableInt (INT_HYPH_LEVEL, bNewEnable);
	}

/**
 * Return the hyphenation minimum word size if enabled
 * @return Hyphenation minimum word size.
 */
	public int hyphMinWord () {
		return getIntValue (INT_HYPH_MIN_WORD);
	}

/**
 * Set and enable the hyphenation minimum word size
 * @param nNewHyphMinWord - New hyphenation minimum word size.
 */
	public void hyphMinWord (int nNewHyphMinWord) {
		setIntValue (INT_HYPH_MIN_WORD, nNewHyphMinWord);
	}

/**
 * Query whether the hyphenation minimum word size is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean hyphMinWordEnable () {
		return isIntEnabled (INT_HYPH_MIN_WORD);
	}

/**
 * Enable/disable the hyphenation minimum word size
 * @param bNewEnable - TRUE if the hyphenation minimum word size is to
 * be enabled; FALSE if it is to be disabled.
 */
	public void hyphMinWordEnable (boolean bNewEnable) {
		enableInt (INT_HYPH_MIN_WORD, bNewEnable);
	}

/**
 * Return the hyphenation minimum prefix size if enabled
 * @return Hyphenation minimum prefix size.
 */
	public int hyphMinPrefix () {
		return getIntValue (INT_HYPH_MIN_PREFIX);
	}

/**
 * Set and enable the hyphenation minimum prefix size
 * @param nNewHyphMinPrefix - New hyphenation minimum prefix size.
 */
	public void hyphMinPrefix (int nNewHyphMinPrefix) {
		setIntValue (INT_HYPH_MIN_PREFIX, nNewHyphMinPrefix);
	}

/**
 * Query whether the hyphenation minimum prefix size is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean hyphMinPrefixEnable () {
		return isIntEnabled (INT_HYPH_MIN_PREFIX);
	}

/**
 * Enable/disable the hyphenation minimum prefix size
 * @param bNewEnable - TRUE if the hyphenation minimum prefix size is to
 * be enabled; FALSE if it is to be disabled.
 */
	public void hyphMinPrefixEnable (boolean bNewEnable) {
		enableInt (INT_HYPH_MIN_PREFIX, bNewEnable);
	}

/**
 * Return the hyphenation minimum suffix size if enabled
 * @return Hyphenation minimum suffix size.
 */
	public int hyphMinSuffix () {
		return getIntValue (INT_HYPH_MIN_SUFFIX);
	}

/**
 * Set and enable the hyphenation minimum suffix size
 * @param nNewHyphMinSuffix - New hyphenation minimum suffix size.
 */
	public void hyphMinSuffix (int nNewHyphMinSuffix) {
		setIntValue (INT_HYPH_MIN_SUFFIX, nNewHyphMinSuffix);
	}

/**
 * Query whether the hyphenation minimum suffix size is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean hyphMinSuffixEnable () {
		return isIntEnabled (INT_HYPH_MIN_SUFFIX);
	}

/**
 * Enable/disable the hyphenation minimum suffix size
 * @param bNewEnable - TRUE if the hyphenation minimum suffix size is to
 * be enabled; FALSE if it is to be disabled.
 */
	public void hyphMinSuffixEnable (boolean bNewEnable) {
		enableInt (INT_HYPH_MIN_SUFFIX, bNewEnable);
	}

/**
 * Return the maximum consecutive hyphenated lines if enabled
 * @return Maximum consecutive hyphenated lines.
 */
	public int hyphMaxLines () {
		return getIntValue (INT_HYPH_MAX_LINES);
	}

/**
 * Set and enable the hyphenation maximum consecutive hyphenated lines
 * @param nNewHyphMaxLines - New maximum consecutive hyphenated lines.
 */
	public void hyphMaxLines (int nNewHyphMaxLines) {
		setIntValue (INT_HYPH_MAX_LINES, nNewHyphMaxLines);
	}

/**
 * Query whether the maximum consecutive hyphenated lines is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean hyphMaxLinesEnable () {
		return isIntEnabled (INT_HYPH_MAX_LINES);
	}

/**
 * Enable/disable the hyphenation maximum consecutive hyphenated lines
 * @param bNewEnable - TRUE if the maximum consecutive hyphenated lines
 * is to be enabled; FALSE if it is to be disabled.
 */
	public void hyphMaxLinesEnable (boolean bNewEnable) {
		enableInt (INT_HYPH_MAX_LINES, bNewEnable);
	}

/**
 * Return the hyphenation suppress formal names if enabled
 * @return Hyphenation suppress formal names.
 */
	public boolean hyphSuppressNames () {
		return getBoolValue (BOOL_HYPH_SUPPRESS_NAMES);
	}

/**
 * Set and enable the hyphenation suppress formal names
 * @param bNewHyphSuppressNames - New hyphenation suppress formal names.
 */
	public void hyphSuppressNames (boolean bNewHyphSuppressNames) {
		setBoolValue (BOOL_HYPH_SUPPRESS_NAMES, bNewHyphSuppressNames);
	}

/**
 * Query whether the hyphenation suppress formal names is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean hyphSuppressNamesEnable () {
		return isBoolEnabled (BOOL_HYPH_SUPPRESS_NAMES);
	}

/**
 * Enable/disable the hyphenation suppress formal names
 * @param bNewEnable - TRUE if the hyphenation suppress formal names is
 * to be enabled; FALSE if it is to be disabled.
 */
	public void hyphSuppressNamesEnable (boolean bNewEnable) {
		enableBool (BOOL_HYPH_SUPPRESS_NAMES, bNewEnable);
	}

/**
 * Return the hyphenation suppress acronyms if enabled
 * @return Hyphenation suppress acronyms.
 */
	public boolean hyphSuppressAcronyms () {
		return getBoolValue (BOOL_HYPH_SUPPRESS_ACRONYMS);
	}

/**
 * Set and enable the hyphenation suppress acronyms
 * @param bNewHyphSuppressAcronyms - New hyphenation suppress acronyms.
 */
	public void hyphSuppressAcronyms (boolean bNewHyphSuppressAcronyms) {
		setBoolValue (BOOL_HYPH_SUPPRESS_ACRONYMS, bNewHyphSuppressAcronyms);
	}

/**
 * Query whether the hyphenation suppress acronyms is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean hyphSuppressAcronymsEnable () {
		return isBoolEnabled (BOOL_HYPH_SUPPRESS_ACRONYMS);
	}

/**
 * Enable/disable the hyphenation suppress acronyms
 * @param bNewEnable - TRUE if the hyphenation suppress acronyms is to
 * be enabled; FALSE if it is to be disabled.
 */
	public void hyphSuppressAcronymsEnable (boolean bNewEnable) {
		enableBool (BOOL_HYPH_SUPPRESS_ACRONYMS, bNewEnable);
	}

/**
 * Retrieve the leader pattern currently in effect.
 * @return Leader pattern code describing how to render tab leaders.
 */
	public int leaderPattern () {
		return getIntValue (INT_LEADER_PATTERN);
	}

/**
 * Change the leader pattern property in this attribute object.
 * @param eNewLeaderPattern - New leader pattern value to use.
 */
	public void leaderPattern (int eNewLeaderPattern) {
		setIntValue (INT_LEADER_PATTERN, eNewLeaderPattern);
	}

/**
 * Query whether the leader pattern is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean leaderPatternEnable () {
		return isIntEnabled (INT_LEADER_PATTERN);
	}

/**
 * Enable/disable the leader pattern.
 * @param bNewEnable - TRUE if the leader pattern is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void leaderPatternEnable (boolean bNewEnable) {
		enableInt (INT_LEADER_PATTERN, bNewEnable);
	}

/**
 * Retrieve the leader pattern width currently in effect.
 * @return Leader pattern width code describing how to render tab
 * leaders.
 */
	public TextMeasurement leaderPatternWidth () {
		return getUnitValue (UNIT_LEADER_PATTERN_WIDTH);
	}

/**
 * Change the leader pattern width property in this attribute object.
 * @param oNewLeaderPatternWidth - New leader pattern width value to
 * use.
 */
	public void leaderPatternWidth (TextMeasurement oNewLeaderPatternWidth) {
		setUnitValue (UNIT_LEADER_PATTERN_WIDTH, oNewLeaderPatternWidth);
	}

/**
 * Query whether the leader pattern width is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean leaderPatternWidthEnable () {
		return isUnitEnabled (UNIT_LEADER_PATTERN_WIDTH);
	}

/**
 * Enable/disable the leader pattern width.
 * @param bNewEnable - TRUE if the leader pattern width is to be
 * enabled; FALSE if it is to be disabled.
 */
	public void leaderPatternWidthEnable (boolean bNewEnable) {
		enableUnit (UNIT_LEADER_PATTERN_WIDTH, bNewEnable);
	}

/**
 * Retrieve the leader alignment currently in effect.
 * @return Leader alignment code describing how to render tab leaders.
 */
	public int leaderAlign () {
		return getIntValue (INT_LEADER_ALIGN);
	}

/**
 * Change the leader alignment property in this attribute object.
 * @param eNewLeaderAlign - New leader alignment value to use.
 */
	public void leaderAlign (int eNewLeaderAlign) {
		setIntValue (INT_LEADER_ALIGN, eNewLeaderAlign);
	}

/**
 * Query whether the leader alignment is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean leaderAlignEnable () {
		return isIntEnabled (INT_LEADER_ALIGN);
	}

/**
 * Enable/disable the leader alignment.
 * @param bNewEnable - TRUE if the leader alignment is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void leaderAlignEnable (boolean bNewEnable) {
		enableInt (INT_LEADER_ALIGN, bNewEnable);
	}

/**
 * Retrieve the leader content currently in effect.
 * @return Leader content code describing how to render tab leaders.
 */
	public TextStream leaderContent () {
		if ((! mbLeaderContentEnable) || (mpoExtra == null) || (mpoExtra.mpoLeaderContent == null)) {
			return TextStream.defaultStream();
		} else {
			return mpoExtra.mpoLeaderContent;
		}
	}

/**
 * Change the leader content property in this attribute object.
 * @param oNewLeaderContent - New leader content value to use.
 */
	public void leaderContent (TextStream oNewLeaderContent) {
		if (oNewLeaderContent.posnCount() == 0) {
			if (mpoExtra != null) {
				mpoExtra.clearLeaderContent();
			}
		} else {
			needExtra();
			mpoExtra.setLeaderContent (oNewLeaderContent, fontService());
		}
		mbLeaderContentEnable = true;
	}

/**
 * Query whether the leader content is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean leaderContentEnable () {
		return mbLeaderContentEnable;
	}

/**
 * Enable/disable the leader content.
 * @param bNewEnable - TRUE if the leader content is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void leaderContentEnable (boolean bNewEnable) {
		mbLeaderContentEnable = bNewEnable;
	}

/**
 * Retrieve the rule style currently in effect.
 * @return Rule style code describing how to render tab leaders.
 */
	public int ruleStyle () {
		return getIntValue (INT_RULE_STYLE);
	}

/**
 * Change the rule style property in this attribute object.
 * @param eNewRuleStyle - New rule style value to use.
 */
	public void ruleStyle (int eNewRuleStyle) {
		setIntValue (INT_RULE_STYLE, eNewRuleStyle);
	}

/**
 * Query whether the rule style is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean ruleStyleEnable () {
		return isIntEnabled (INT_RULE_STYLE);
	}

/**
 * Enable/disable the rule style.
 * @param bNewEnable - TRUE if the rule style is to be enabled; FALSE if
 * it is to be disabled.
 */
	public void ruleStyleEnable (boolean bNewEnable) {
		enableInt (INT_RULE_STYLE, bNewEnable);
	}

/**
 * Retrieve the rule thickness currently in effect.
 * @return Rule thickness code describing how to render tab leaders.
 */
	public TextMeasurement ruleThickness () {
		return getUnitValue (UNIT_RULE_THICKNESS);
	}

/**
 * Change the rule thickness property in this attribute object.
 * @param oNewRuleThickness - New rule thickness value to use.
 */
	public void ruleThickness (TextMeasurement oNewRuleThickness) {
		setUnitValue (UNIT_RULE_THICKNESS, oNewRuleThickness);
	}

/**
 * Query whether the rule thickness is enabled.
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean ruleThicknessEnable () {
		return isUnitEnabled (UNIT_RULE_THICKNESS);
	}

/**
 * Enable/disable the rule thickness.
 * @param bNewEnable - TRUE if the rule thickness is to be enabled;
 * FALSE if it is to be disabled.
 */
	public void ruleThicknessEnable (boolean bNewEnable) {
		enableUnit (UNIT_RULE_THICKNESS, bNewEnable);
	}
/**
 * Query the horizontal scale.
 * <p>
 * The horizontal scale is applied to the glyph X axis of the font,
 * allowing glyphs to be stretched or shrunk in X independent of Y
 * scaling.
 * @return Horizontal scale factor.  Default is 1.0.
 */
	public double horizontalScale () {
		return getFloatValue (FLOAT_HORIZONTAL_SCALE);
	}

/**
 * Set the horizontal scale factor.
 * @param nNewHorizontalScale - New Horizontal scale factor to apply to
 * font.
 */
	public void horizontalScale (double nNewHorizontalScale) {
		setFloatValue (FLOAT_HORIZONTAL_SCALE, nNewHorizontalScale);
		FontDesc oFontDesc = new FontDesc();
		oFontDesc.setHorizontalScale (nNewHorizontalScale);
		updateFont (oFontDesc);
	}

/**
 * Query whether the horizontal scale value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean horizontalScaleEnable () {
		return isFloatEnabled (FLOAT_HORIZONTAL_SCALE);
	}

/**
 * Enable/disable the horizontal scale value
 * @param bNewEnable - TRUE if the horizontal scale value is to be
 * enabled; FALSE if it is to be disabled.
 */
	public void horizontalScaleEnable (boolean bNewEnable) {
		enableFloat (FLOAT_HORIZONTAL_SCALE, bNewEnable);
		updateFont();
	}

/**
 * Query the Vertical scale.
 * <p>
 * The vertical scale is applied to the glyph Y axis of the font,
 * allowing glyphs to be stretched or shrunk in Y independent of X
 * scaling.
 * @return Vertical scale factor.  Default is 1.0.
 */
	public double verticalScale () {
		return getFloatValue (FLOAT_VERTICAL_SCALE);
	}

/**
 * Set the vertical scale factor.
 * @param nNewVerticalScale - New vertical scale factor to apply to font.
 */
	public void verticalScale (double nNewVerticalScale) {
		setFloatValue (FLOAT_VERTICAL_SCALE, nNewVerticalScale);
		FontDesc oFontDesc = new FontDesc();
		oFontDesc.setVerticalScale (nNewVerticalScale);
		updateFont (oFontDesc);
	}

/**
 * Query whether the vertical scale value is enabled
 * @return TRUE if enabled; FALSE if not.
 */
	public boolean verticalScaleEnable () {
		return isFloatEnabled (FLOAT_VERTICAL_SCALE);
	}

/**
 * Enable/disable the vertical scale value
 * @param bNewEnable - TRUE if the vertical scale value is to be
 * enabled; FALSE if it is to be disabled.
 */
	public void verticalScaleEnable (boolean bNewEnable) {
		enableFloat (FLOAT_VERTICAL_SCALE, bNewEnable);
		updateFont();
	}

/**
 * Fill the text attribute holder with default values.
 * <p>
 * This method behaves like a Clear() or Reset() method might.	It
 * replaces all values in the text attribute object with defaults.
 * @param bDefault - FALSE if all attributes are to be disabled; TRUE if
 * they are to be enabled with default values.
 */
	public void setDefault (boolean bDefault) {
// update the font instance when change attributes
		if (bDefault) {
			UnitSpan o12pt = new UnitSpan (UnitSpan.POINTS_1K, 12000);
			int iWeight = FontInfo.WEIGHT_NORMAL;
			boolean bItalic = false;

			moFontInstance = null;

// We will need the default typeface a lot so make the call once
// (or twice or so) and keep it around using a GlobalString.	Ensure
// we lose the global status when passing it to UpdateFont().
//			GlobalString msDefaultTypeface = getMsDefaultTypeface();
//			if (msDefaultTypeface.Length() == 0) {
//				msDefaultTypeface.LoadResource (JF_TEXT_DEFAULT_TYPEFACE);
//			}
// This forces a copy to a non-global string for performance.
			String sDefaultTypeface = gmsDefaultTypeface;

			FontDesc oFontDesc = new FontDesc();
			oFontDesc.setTypeface (sDefaultTypeface);
			oFontDesc.setSize (o12pt);
			oFontDesc.setWeight (iWeight);
			oFontDesc.setItalic (bItalic);
			oFontDesc.setEncoding (gsDefaultEncoding);
			oFontDesc.setHorizontalScale (1.0);
			oFontDesc.setVerticalScale (1.0);
			updateFont (oFontDesc);

			setGfxTextAttrDefault (true);

			transparent (false);

			justifyV (JUST_V_TOP);
			justifyH (JUST_H_LEFT);

			tabs (TextTabList.DEFAULT_TAB_LIST);

			invisible (false);

			localeEnable (true);

			if (mpoExtra != null) {
				mpoExtra.initialize();
			}
			spacingEnable (true);
			marginLEnable (true);
			marginREnable (true);
			radixOffsetEnable (true);
			radixPosEnable (true);
			specialEnable (true);
			spaceBeforeEnable (true);
			spaceAfterEnable (true);
			invisCharEnable (true);
			baselineShiftEnable (true);
			digitsEnable (true);
			directionEnable (true);
			paraDirectionEnable (true);
			layoutOrientationEnable (true);
			ligatureEnable (true);
			charSpacingEnable (true);
			wordSpacingEnable (true);
			kerning (false);
			hyphLevelEnable (true);
			hyphMinWordEnable (true);
			hyphMinPrefixEnable (true);
			hyphMinSuffixEnable (true);
			hyphMaxLinesEnable (true);
			hyphSuppressNames (false);
			hyphSuppressAcronyms (false);
			leaderPatternEnable (true);
			leaderPatternWidthEnable (true);
			leaderAlignEnable (true);
			leaderContentEnable (true);
			ruleStyleEnable (true);
			ruleThicknessEnable (true);
		} else {
			fontEnable (false);
			typefaceEnable (false);
			encodingEnable (false);
			sizeEnable (false);
			weightEnable (false);
			italicEnable (false);
			horizontalScaleEnable (false);
			verticalScaleEnable (false);
			setGfxTextAttrDefault (false);
			transparent (false);
			transparentEnable (false);
			spacingEnable (false);
			marginLEnable (false);
			marginREnable (false);
			justifyVEnable (false);
			justifyHEnable (false);
			radixOffsetEnable (false);
			radixPosEnable (false);
			tabsEnable (false);
			specialEnable (false);
			spaceBeforeEnable (false);
			spaceAfterEnable (false);
			invisible (false);
			invisibleEnable (false);
			invisCharEnable (false);
			baselineShiftEnable (false);
			localeEnable (false);
			digitsEnable (false);
			directionEnable (false);
			paraDirectionEnable (false);
			layoutOrientationEnable (false);
			ligatureEnable (false);
			charSpacingEnable (false);
			wordSpacingEnable (false);
			kerning (false);
			kerningEnable (false);
			hyphLevelEnable (false);
			hyphMinWordEnable (false);
			hyphMinPrefixEnable (false);
			hyphMinSuffixEnable (false);
			hyphMaxLinesEnable (false);
			hyphSuppressNamesEnable (false);
			hyphSuppressAcronymsEnable (false);
			leaderPatternEnable (false);
			leaderPatternWidthEnable (false);
			leaderAlignEnable (false);
			leaderContentEnable (false);
			ruleStyleEnable (false);
			ruleThicknessEnable (false);
		}
	}

/**
 * Partial copy, filling in disabled attributes.
 * <p>
 * Copy candidate attributes from the source attribute object.	An
 * attribute is a candidate for copying if it is disabled in this object
 * and enabled in the source.
 * @param oSource - Source atrributes for copy.
 */
	public void addDisabled (TextAttr oSource) {
// this does all the work
		AttrCopy copier = new AttrCopy (oSource, AttrCopy.ADD_DISABLED);
		copier.copy (this);
	}

/**
 * Partial copy, selecting enabled attributes in the source.
 * <p>
 * Copy candidate attributes from the source attribute object.	An
 * attribute is a candidate for copying if it is enabled in the source.
 * In other words, this method overrides attributes in this object, but
 * will not override an enabled attribute with a disabled one.
 * @param oOverride - Source attributes for copy.
 * @param poDiffs - (optional) Pointer to an attribute to receive just
 * the true differences.  If NULL, the differences are not recorded
 * separately.
 */
	public void override (TextAttr oOverride, TextAttr poDiffs) {
// this does all the work
		AttrCopy copier = new AttrCopy (oOverride, AttrCopy.OVERRIDE_ONLY);
		copier.copy (this, poDiffs);
	}
	public void override (TextAttr oOverride) {
		override (oOverride, null);
	}

/**
 * Disable attributes where different.
 * <p>
 * Compare this object against another on an attribute by attribute
 * basis.  Where attribute values differ between the two objects,
 * disable them in this object.
 * @param oCompare - Attribute object to compare against.
 */
	public void dropDiff (TextAttr oCompare) {
		if (oCompare.fontEnable()) {
			if (! FontInstance.match (moFontInstance, oCompare.moFontInstance)) {
				fontEnable (false);
			}
		}

		if (typefaceEnable() && oCompare.typefaceEnable()) {
			if (! typeface().equals (oCompare.typeface())) {
				typefaceEnable (false);
			}
		}

		if (encodingEnable() && oCompare.encodingEnable()) {
			if (encoding() != oCompare.encoding()) {
				encodingEnable (false);
			}
		}

		if (sizeEnable() && oCompare.sizeEnable()) {
			if (size() != oCompare.size()) {
				sizeEnable (false);
			}
		}

		if (weightEnable() && oCompare.weightEnable()) {
			if (weight() != oCompare.weight()) {
				weightEnable (false);
			}
		}

		if (italicEnable() && oCompare.italicEnable()) {
			if (italic() != oCompare.italic()) {
				italicEnable (false);
			}
		}

		if (horizontalScaleEnable() && oCompare.horizontalScaleEnable()) {
			if (horizontalScale() != oCompare.horizontalScale()) {
				horizontalScaleEnable (false);
			}
		}

		if (verticalScaleEnable() && oCompare.verticalScaleEnable()) {
			if (verticalScale() != oCompare.verticalScale()) {
				verticalScaleEnable (false);
			}
		}

		if (underlineEnable() && oCompare.underlineEnable()) {
			if (underline() != oCompare.underline()) {
				underlineEnable (false);
			}
		}

		if (overlineEnable() && oCompare.overlineEnable()) {
			if (overline() != oCompare.overline()) {
				overlineEnable (false);
			}
		}

		if (strikeoutEnable() && oCompare.strikeoutEnable()) {
			if (strikeout() != oCompare.strikeout()) {
				strikeoutEnable (false);
			}
		}

		if (overlineEnable() && oCompare.overlineEnable()) {
			if (overline() != oCompare.overline()) {
				overlineEnable (false);
			}
		}

		if (colourEnable() && oCompare.colourEnable()) {
			if (colour() != oCompare.colour()) {
				colourEnable (false);
			}
		}

		if (colourBgEnable() && oCompare.colourBgEnable()) {
			if (colourBg() != oCompare.colourBg()) {
				colourBgEnable (false);
			}
		}

		if (styleEnable() && oCompare.styleEnable()) {
			if (style() != oCompare.style()) {
				styleEnable (false);
			}
		}

		if (shadeEnable() && oCompare.shadeEnable()) {
			if (shade() != oCompare.shade()) {
				shadeEnable (false);
			}
		}

		if (shadeScaleEnable() && oCompare.shadeScaleEnable()) {
			if (shadeScale() != oCompare.shadeScale()) {
				shadeScaleEnable (false);
			}
		}

		if (graphicContextEnable() && oCompare.graphicContextEnable()) {
			if (graphicContext() != oCompare.graphicContext()) {
				graphicContextEnable (false);
			}
		}

		if (justifyVEnable() && oCompare.justifyVEnable()) {
			if (justifyV() != oCompare.justifyV()) {
				justifyVEnable (false);
			}
		}

		if (justifyHEnable() && oCompare.justifyHEnable()) {
			if (justifyH() != oCompare.justifyH()) {
				justifyHEnable (false);
			}
		}

		if (tabsEnable() && oCompare.tabsEnable()) {
			if (tabs() != oCompare.tabs()) {
				tabsEnable (false);
			}
		}

		if (baselineShiftEnable() && oCompare.baselineShiftEnable()) {
			if (baselineShift() != oCompare.baselineShift()) {
				baselineShiftEnable (false);
			}
		}

		if (localeEnable() && oCompare.localeEnable()) {
			if (locale() != oCompare.locale()) {
				localeEnable (false);
			}
		}

		if (leaderContentEnable() && oCompare.leaderContentEnable()) {
			if (leaderContent() != oCompare.leaderContent()) {
				leaderContentEnable (false);
			}
		}

		int i;

		for (i = 0; i < MAX_BOOL_INDEX; i++) {
			if (i != BOOL_ITALIC) { // italic must be explicit call because it may change font
				if (isBoolEnabled (i) && oCompare.isBoolEnabled (i)) {
					if (getBoolValue (i) != oCompare.getBoolValue (i)) {
						enableBool (i, false);
					}
				}
			}
		}

		for (i = 0; i < MAX_UNIT_INDEX; i++) {
			if (isUnitEnabled (i) && oCompare.isUnitEnabled (i)) {
				if (getUnitValue (i) != oCompare.getUnitValue (i)) {
					enableUnit (i, false);
				}
			}
		}

		for (i = 0; i < MAX_INT_INDEX; i++) {
			if (isIntEnabled (i) && oCompare.isIntEnabled (i)) {
				if (getIntValue (i) != oCompare.getIntValue (i)) {
					enableInt (i, false);
				}
			}
		}

		for (i = 0; i < MAX_FLOAT_INDEX; i++) {
			if ((i != FLOAT_HORIZONTAL_SCALE) && (i != FLOAT_VERTICAL_SCALE)) { // must be explicit call because it may change font
				if (isFloatEnabled (i) && oCompare.isFloatEnabled (i)) {
					if (getFloatValue (i) != oCompare.getFloatValue (i)) {
						enableFloat (i, false);
					}
				}
			}
		}
	}

/**
 * Disable attributes where same.
 * <p>
 * Compare this object against another on an attribute by attribute
 * basis.  Where attribute values are the same between the two objects,
 * disable them in this object.
 * @param oCompare - Attribute object to compare against.
 * @param bPreserveGfx - (optional) If TRUE, don't drop the graphic text
 * attributes even if they are the same.  If FALSE (default) don't give
 * graphic text attributes any special treatment.
 */
	public void dropSame (TextAttr oCompare, boolean bPreserveGfx) {
		boolean bPreserving = bPreserveGfx /* && (AttrPool() != null) */;

		if (fontEnable() && oCompare.fontEnable()) {
			if (FontInstance.match (moFontInstance, oCompare.moFontInstance)) {
				fontEnable (false);
			}
		}

// note: disabling font (pointer) above doesn't disable individual
// components, so have to do them all anyway
		if (typefaceEnable() && oCompare.typefaceEnable()) {
			if (typeface().equals (oCompare.typeface())) {
				typefaceEnable (false);
			}
		}

		if (sizeEnable() && oCompare.sizeEnable()) {
			if (UnitSpan.match (size(), oCompare.size())) {
				sizeEnable (false);
			}
		}

		if (weightEnable() && oCompare.weightEnable()) {
			if (weight() == oCompare.weight()) {
				weightEnable (false);
			}
		}

		if (italicEnable() && oCompare.italicEnable()) {
			if (italic() == oCompare.italic()) {
				italicEnable (false);
			}
		}

		if (horizontalScaleEnable() && oCompare.horizontalScaleEnable()) {
			if (horizontalScale() == oCompare.horizontalScale()) {
				horizontalScaleEnable (false);
			}
		}

		if (verticalScaleEnable() && oCompare.verticalScaleEnable()) {
			if (verticalScale() == oCompare.verticalScale()) {
				verticalScaleEnable (false);
			}
		}

		if (encodingEnable() && oCompare.encodingEnable()) {
			if (encoding() == oCompare.encoding()) {
				encodingEnable (false);
			}
		}

		if ((! typefaceEnable())
		 || (! sizeEnable())
		 || (! weightEnable())
		 || (! italicEnable())
		 || (! horizontalScaleEnable())
		 || (! verticalScaleEnable())) {
			fontEnable (false);
		}

		boolean bSkipGfx = false;
		if (bPreserving) {
//			if (mpoPooledAttr == oCompare.mpoPooledAttr) {
//				UnderlineEnable (false); // shortcut -- avoid testing each
//				OverlineEnable (false); // shortcut -- avoid testing each
//				ColourEnable (false);
//				ColourBgEnable (false);
//				StyleEnable (false);
//				ShadeEnable (false);
//				ShadeScaleEnable (false);
//				GraphicContextEnable (false);
//				bSkipGfx = true;
//			}
		}

		if (! bSkipGfx) {
			if (underlineEnable() && oCompare.underlineEnable()) {
				if (underline() == oCompare.underline()) {
					underlineEnable (false);
				}
			}

			if (overlineEnable() && oCompare.overlineEnable()) {
				if (overline() == oCompare.overline()) {
					overlineEnable (false);
				}
			}

			if (strikeoutEnable() && oCompare.strikeoutEnable()) {
				if (strikeout() == oCompare.strikeout()) {
					strikeoutEnable (false);
				}
			}

			if (overlineEnable() && oCompare.overlineEnable()) {
				if (overline() == oCompare.overline()) {
					overlineEnable (false);
				}
			}

			if (colourEnable() && oCompare.colourEnable()) {
				if (colour() == oCompare.colour()) {
					colourEnable (false);
				}
			}

			if (colourBgEnable() && oCompare.colourBgEnable()) {
				if (colourBg() == oCompare.colourBg()) {
					colourBgEnable (false);
				}
			}

			if (styleEnable() && oCompare.styleEnable()) {
				if (style() == oCompare.style()) {
					styleEnable (false);
				}
			}

			if (shadeEnable() && oCompare.shadeEnable()) {
				if (shade() == oCompare.shade()) {
					shadeEnable (false);
				}
			}

			if (shadeScaleEnable() && oCompare.shadeScaleEnable()) {
				if (shadeScale() == oCompare.shadeScale()) {
					shadeScaleEnable (false);
				}
			}

			if (graphicContextEnable() && oCompare.graphicContextEnable()) {
				if (graphicContext() == oCompare.graphicContext()) {
					graphicContextEnable (false);
				}
			}
		}

		if (justifyVEnable() && oCompare.justifyVEnable()) {
			if (justifyV() == oCompare.justifyV()) {
				justifyVEnable (false);
			}
		}

		if (justifyHEnable() && oCompare.justifyHEnable()) {
			if (justifyH() == oCompare.justifyH()) {
				justifyHEnable (false);
			}
		}

		if (tabsEnable() && oCompare.tabsEnable()) {
			if (tabs() == oCompare.tabs()) {
				tabsEnable (false);
			}
		}

		if (baselineShiftEnable() && oCompare.baselineShiftEnable()) {
			if (baselineShift() == oCompare.baselineShift()) {
				baselineShiftEnable (false);
			}
		}

		if (localeEnable() && oCompare.localeEnable()) {
			if (locale() == oCompare.locale()) {
				localeEnable (false);
			}
		}

		if (leaderContentEnable() && oCompare.leaderContentEnable()) {
			if (leaderContent() == oCompare.leaderContent()) {
				leaderContentEnable (false);
			}
		}

		int i;

		for (i = 0; i < MAX_BOOL_INDEX; i++) {
			if (i != BOOL_ITALIC) { // italic must be explicit call because it may change font
				if (isBoolEnabled (i) && oCompare.isBoolEnabled (i)) {
					if (getBoolValue (i) == oCompare.getBoolValue (i)) {
						enableBool (i, false);
					}
				}
			}
		}

		for (i = 0; i < MAX_UNIT_INDEX; i++) {
			if (isUnitEnabled (i) && oCompare.isUnitEnabled (i)) {
				if (getUnitValue (i) == oCompare.getUnitValue (i)) {
					enableUnit (i, false);
				}
			}
		}

		for (i = 0; i < MAX_INT_INDEX; i++) {
			if (isIntEnabled (i) && oCompare.isIntEnabled (i)) {
				if (getIntValue (i) == oCompare.getIntValue (i)) {
					enableInt (i, false);
				}
			}
		}

		for (i = 0; i < MAX_FLOAT_INDEX; i++) {
			if ((i != FLOAT_HORIZONTAL_SCALE) && (i != FLOAT_VERTICAL_SCALE)) { // must be explicit call because it may change font
				if (isFloatEnabled (i) && oCompare.isFloatEnabled (i)) {
					if (getFloatValue (i) == oCompare.getFloatValue (i)) {
						enableFloat (i, false);
					}
				}
			}
		}
	}

	public void dropSame (TextAttr oCompare) {
		dropSame (oCompare, false);
	}

/**
 * Isolate paragraph attributes.
 * <p>
 * Isolate the paragraph attributes in this object either by disabling
 * everything else or by disabling the paragraph attributes.
 * @param bKeep - (optional) If TRUE (default) paragraph attributes are
 * preserved and all others are disabled.  If FALSE paragraph attributes
 * are disabled and all others are preserved.
 */
	public void isolatePara (boolean bKeep, boolean bLegacyPositioning) {
		if (bKeep) {
			fontEnable (false);
			typefaceEnable (false);
			encodingEnable (false);
			sizeEnable (false);
			weightEnable (false);
			italicEnable (false);
			horizontalScaleEnable (false);
			verticalScaleEnable (false);
			underlineEnable (false);
			overlineEnable (false);
			strikeoutEnable (false);
			overlineEnable (false);
			colourEnable (false);
			colourBgEnable (false);
			styleEnable (false);
			shadeEnable (false);
			shadeScaleEnable (false);
			graphicContextEnable (false);
			transparentEnable (false);
			invisibleEnable (false);
			invisCharEnable (false);
			baselineShiftEnable (false);
			localeEnable (false);
			digitsEnable (false);
			directionEnable (false);
			ligatureEnable (false);
			charSpacingEnable (false);
			wordSpacingEnable (false);
			kerningEnable (false);
			leaderPatternEnable (false);
			leaderPatternWidthEnable (false);
			leaderAlignEnable (false);
			leaderContentEnable (false);
			ruleStyleEnable (false);
			ruleThicknessEnable (false);
		} else {
			marginLEnable (false);
			marginREnable (false);
			justifyVEnable (false);
			justifyHEnable (false);
			radixOffsetEnable (false);
			radixPosEnable (false);
			tabsEnable (false);
			specialEnable (false);
			spaceBeforeEnable (false);
			spaceAfterEnable (false);
			paraDirectionEnable (false);
			layoutOrientationEnable (false);
			hyphLevelEnable (false);
			hyphMinWordEnable (false);
			hyphMinPrefixEnable (false);
			hyphMinSuffixEnable (false);
			hyphMaxLinesEnable (false);
			hyphSuppressNamesEnable (false);
			hyphSuppressAcronymsEnable (false);
		}

		if (bKeep == bLegacyPositioning) {
			spacingEnable (false);
		}
	}
	public void isolatePara (boolean bKeep) {
		isolatePara (bKeep, false);
	}

/**
 * Determine whether all attributes are enabled
 * @return TRUE if all attributes are enabled; FALSE if any attribute is
 * disabled.
 */
	public boolean isComplete () {
		return typefaceEnable()
			&& encodingEnable()
			&& sizeEnable()
			&& weightEnable()
			&& italicEnable()
			&& horizontalScaleEnable()
			&& verticalScaleEnable()
			&& gfxTextAttrEnable()
			&& transparentEnable()
			&& spacingEnable()
			&& marginLEnable()
			&& marginREnable()
			&& justifyVEnable()
			&& justifyHEnable()
			&& radixOffsetEnable()
			&& radixPosEnable()
			&& tabsEnable()
			&& specialEnable()
			&& spaceBeforeEnable()
			&& spaceAfterEnable()
			&& invisibleEnable()
			&& invisCharEnable()
			&& baselineShiftEnable()
			&& localeEnable()
			&& digitsEnable()
			&& directionEnable()
			&& paraDirectionEnable()
			&& layoutOrientationEnable()
			&& ligatureEnable()
			&& charSpacingEnable()
			&& wordSpacingEnable()
			&& kerningEnable()
			&& hyphLevelEnable()
			&& hyphMinWordEnable()
			&& hyphMinPrefixEnable()
			&& hyphMinSuffixEnable()
			&& hyphMaxLinesEnable()
			&& hyphSuppressNamesEnable()
			&& hyphSuppressAcronymsEnable()
			&& leaderPatternEnable()
			&& leaderPatternWidthEnable()
			&& leaderAlignEnable()
			&& leaderContentEnable()
			&& ruleStyleEnable()
			&& ruleThicknessEnable();
	}

/**
 * Determine whether all attributes are disabled
 * @return TRUE if all attributes are disabled; FALSE if any attribute
 * is enabled.
 */
	public boolean isEmpty () {
		return ! (typefaceEnable()
			   || encodingEnable()
			   || sizeEnable()
			   || weightEnable()
			   || italicEnable()
			   || horizontalScaleEnable()
			   || verticalScaleEnable()
			   || underlineEnable()
			   || overlineEnable()
			   || strikeoutEnable()
			   || overlineEnable()
			   || colourEnable()
			   || colourBgEnable()
			   || styleEnable()
			   || shadeEnable()
			   || shadeScaleEnable()
			   || graphicContextEnable()
			   || transparentEnable()
			   || spacingEnable()
			   || marginLEnable()
			   || marginREnable()
			   || justifyVEnable()
			   || justifyHEnable()
			   || radixOffsetEnable()
			   || radixPosEnable()
			   || tabsEnable()
			   || specialEnable()
			   || spaceBeforeEnable()
			   || spaceAfterEnable()
			   || invisibleEnable()
			   || invisCharEnable()
			   || baselineShiftEnable()
			   || localeEnable()
			   || digitsEnable()
			   || directionEnable()
			   || paraDirectionEnable()
			   || layoutOrientationEnable()
			   || ligatureEnable()
			   || charSpacingEnable()
			   || wordSpacingEnable()
			   || kerningEnable()
			   || hyphLevelEnable()
			   || hyphMinWordEnable()
			   || hyphMinPrefixEnable()
			   || hyphMinSuffixEnable()
			   || hyphMaxLinesEnable()
			   || hyphSuppressNamesEnable()
			   || hyphSuppressAcronymsEnable()
			   || leaderPatternEnable()
			   || leaderPatternWidthEnable()
			   || leaderAlignEnable()
			   || leaderContentEnable()
			   || ruleStyleEnable()
			   || ruleThicknessEnable());
	}

/**
 * Determine whether any paragraph attribute is enabled
 * @return TRUE if any paragraph attribute is enabled; FALSE if all
 * paragraph attributes are disabled.
 */
	public boolean hasParaAttrs () {
		return marginLEnable()
			|| marginREnable()
			|| justifyVEnable()
			|| justifyHEnable()
			|| radixOffsetEnable()
			|| radixPosEnable()
			|| tabsEnable()
			|| specialEnable()
			|| spacingEnable()
			|| spaceBeforeEnable()
			|| spaceAfterEnable()
			|| paraDirectionEnable()
			|| layoutOrientationEnable()
			|| hyphLevelEnable()
			|| hyphMinWordEnable()
			|| hyphMinPrefixEnable()
			|| hyphMinSuffixEnable()
			|| hyphMaxLinesEnable()
			|| hyphSuppressNamesEnable()
			|| hyphSuppressAcronymsEnable();
	}

/**
 * Populate a graphic font selection object.
 * <p>
 * A graphic font selection object represents font information that may
 * span several fonts, in a manner similar to the way this object can
 * selectively enable or disable font attributes.  This method populates
 * the font selection object based on the font attributes and their
 * enabled status.
 * @param oSelect - Graphic font selection object to populate.
 * @param bReconcile - (optional) If TRUE (default), the selection
 * object is asked to reconcile the font as well.  If FALSE, the method
 * simply populates the selection object.
 */
/*
	public void ApplyToFontSelect (GfxFontSelect oSelect, boolean bReconcile) {
		if (TypefaceEnable()) {
			oSelect.Typeface (msTypeface);
		} else {
			oSelect.TypefaceVariousSet();
		}

		if (SizeEnable()) {
			oSelect.Size (moSize);
		} else {
			oSelect.SizeVariousSet();
		}

		if (WeightEnable()) {
			oSelect.Bold (miWeight >= GfxTypeface.WEIGHT_BOLD);
		} else {
			oSelect.BoldVariousSet();
		}

		if (ItalicEnable()) {
			oSelect.Italic (Italic());
		} else {
			oSelect.ItalicVariousSet();
		}

		if (bReconcile) {
			oSelect.Reconcile();
		}
	}
*/

/**
 * Update font attributes from a graphic font selection object.
 * <p>
 * A graphic font selection object represents font information that may
 * span several fonts, in a manner similar to the way this object can
 * selectively enable or disable font attributes.  This method updates
 * attributes from the font selection object.  Note that a selection
 * value of various does <b>not</b> disable corresponding text attribute
 * values.	The purpose of this method is only to accumulate new "real"
 * values.
 * @param oSelect - Graphic font selection object to examine.
 */
/*
	public void SetFromFontSelect (GfxFontSelect oSelect) {
		if (! oSelect.TypefaceVarious()) {
			Typeface (oSelect.Typeface());
		}

		if (! oSelect.SizeVarious()) {
			Size (oSelect.Size());
		}

		if (! oSelect.BoldVarious()) {
			Weight (oSelect.Bold() ? GfxTypeface.WEIGHT_BOLD : GfxTypeface.WEIGHT_NORMAL);
		}

		if (! oSelect.ItalicVarious()) {
			Italic (oSelect.Italic());
		}
	}
*/

/**
 * Format a relative value as a percentage.
 * <p>
 * Given a relative value, this method returns a string representing
 * that relative value formatted as a percentage.  For example, if the
 * relative value is 1.0, the result is "100%".  Similarly, a relative
 * value of 0.75 yields a result of "75%".
 * @param dRelative - Relative value to be formatted.
 * @return String result, as described above.
 */
	public static String formatPercent (double dRelative) {
		return Units.doubleToString (dRelative, 3) + '%';
	}

/**
 * Parse a persentage string to get a relative value.
 * <p>
 * Given a string, this method parses it as a percentage and returns a
 * relative value.	For example, if the string is "120%", the result is
 * 1.2.  The percent sign is optional; if omitted, it is assumed.  In
 * other words, an input of "1.2" is assumed to mean 1.2% and yields a
 * result of 0.012.
 * @param sText - Text string to parse.
 * @param bAllowNegative - (optional) True to allow negative
 * percentages; false (default) to treat them as errors.
 * @return True if the parse succeeded; false if an error occurred.
 */
	public static double parsePercent (String sText, boolean bAllowNegative) {
		double dValue = 0;

		StringBuilder sContent = new StringBuilder (sText);
		int last = sContent.length() - 1;
		while ((last > 0) && (sContent.charAt (last) == ' ')) {
			last--;
		}
		if ((last >= 0) && (sContent.charAt (last) == '%')) {
			last--;
		}
		sContent.delete (last+1, sContent.length());

		try {
			dValue = Double.parseDouble (sContent.toString());
		} catch (NumberFormatException e) {
			return Double.NaN;
		}
		if ((! bAllowNegative) && (dValue < 0)) {
			return Double.NaN;
		}

		return dValue / 100;
	}

/**
 * Replace all attributes with those from the source object.
 * <p>
 * The standard assignment copies everything, including enabled and
 * disabled status.  Graphic source information is also copied.
 * @param oSource - Source attribute object to copy.
 */
	public void copyFrom (TextAttr oSource) {
//		if (oSource.mpoPooledAttr != mpoPooledAttr) {
//			CleanupGfxTextAttr(); // before switching graphic source
//		}
		moGfxSource = oSource.moGfxSource; // Note: copy source's pointers
		moFontInstance = oSource.moFontInstance;

		msTypeface = oSource.msTypeface;
		msOriginalTypeface = oSource.msOriginalTypeface;
		assert (msOriginalTypeface != null);
		msEncoding = oSource.msEncoding;
		moSize = oSource.moSize;
		miWeight = oSource.miWeight;

//		if ((oSource.mpoPooledAttr != mpoPooledAttr) || (mpoPooledAttr == null)) {
//			if (AttrPool() == null) {
//				TextGfxAttr.copyFrom (oSource);
//			}
//			else {
//				if (oSource.mpoPooledAttr != null) {
//					AttrPool().AddPointer (oSource.mpoPooledAttr);
//					UpdateGfxTextAttr (oSource.mpoPooledAttr);
//				}
//				else if (oSource.GfxTextAttrEnable()) {
//					UpdateGfxTextAttr (AttrPool().AddAttr (oSource.GetGfxTextAttr()));
//				}
//				else {
					super.copyFrom (oSource);
//				}
//			}
//		}

		if (oSource.mpoExtra == null) {
			if (mpoExtra != null) {
				mpoExtra.initialize();
			}
		} else {
			needExtra();
			mpoExtra.copyFrom (oSource.mpoExtra);
		}

		meJustifyV = oSource.meJustifyV;
		meJustifyH = oSource.meJustifyH;

		if (oSource.mpoTabs == null) {
			tabsEnable (false);
		} else {
			mpoTabs = oSource.mpoTabs;	// can simply copy ref; TODO: assumes tabs used immutably
		}

		msLocale = oSource.msLocale;

		mbBoolValue = oSource.mbBoolValue;
		mbBoolEnable = oSource.mbBoolEnable;
		mbUnitEnable = oSource.mbUnitEnable;
		mbIntEnable = oSource.mbIntEnable;
		mbFloatEnable = oSource.mbFloatEnable;

		mbBaselineShiftEnable = oSource.mbBaselineShiftEnable;
		mbJustifyVEnable = oSource.mbJustifyVEnable;
		mbJustifyHEnable = oSource.mbJustifyHEnable;
		mbLocaleEnable = oSource.mbLocaleEnable;
		mbLeaderContentEnable = oSource.mbLeaderContentEnable;
	}

/**
 * Equality comparison operator.
 * <p>
 * Compares on an attribute by attribute basis.  Two attributes are
 * considered equal if they are both disabled, or they are both enabled
 * and their values compare for equality.
 * @param object Text attribute object to compare against.
 * @return TRUE if the text attribute objects are considered equal;
 * FALSE otherwise.
 */
	public boolean equals (Object object) {
		
		if (this == object)
			return true;
		
		if (!super.equals(object))
			return false;
		
		TextAttr oCompare = (TextAttr) object;
		
		if (fontEnable() != oCompare.fontEnable()) {
			return false;
		}
		if (typefaceEnable() != oCompare.typefaceEnable()) {
			return false;
		}
		if (encodingEnable() != oCompare.encodingEnable()) {
			return false;
		}
		if (sizeEnable() != oCompare.sizeEnable()) {
			return false;
		}
		if (weightEnable() != oCompare.weightEnable()) {
			return false;
		}
		if (justifyVEnable() != oCompare.justifyVEnable()) {
			return false;
		}
		if (justifyHEnable() != oCompare.justifyHEnable()) {
			return false;
		}
		if (tabsEnable() != oCompare.tabsEnable()) {
			return false;
		}
		if (baselineShiftEnable() != oCompare.baselineShiftEnable()) {
			return false;
		}
		if (localeEnable() != oCompare.localeEnable()) {
			return false;
		}
		if (leaderContentEnable() != oCompare.leaderContentEnable()) {
			return false;
		}

		if (mbUnitEnable != oCompare.mbUnitEnable) {
			return false;
		}
		if (mbIntEnable != oCompare.mbIntEnable) {
			return false;
		}
		if (mbBoolEnable != oCompare.mbBoolEnable) {
			return false;
		}
		if (mbFloatEnable != oCompare.mbFloatEnable) {
			return false;
		}

		if (fontEnable()) {
			if (! FontInstance.match (fontInstance(), oCompare.fontInstance())) {
				return false;
			}
		}
		if (typefaceEnable()) {
			if (typeface() != oCompare.typeface()) {
				return false;
			}
		}
		if (encodingEnable()) {
			if (encoding() != oCompare.encoding()) {
				return false;
			}
		}
		if (sizeEnable()) {
			if (size() != oCompare.size()) {
				return false;
			}
		}
		if (weightEnable()) {
			if (weight() != oCompare.weight()) {
				return false;
			}
		}
		if (justifyVEnable()) {
			if (justifyV() != oCompare.justifyV()) {
				return false;
			}
		}
		if (justifyHEnable()) {
			if (justifyH() != oCompare.justifyH()) {
				return false;
			}
		}
		if (tabsEnable()) {
			if (tabs() != oCompare.tabs()) {
				return false;
			}
		}
		if (baselineShiftEnable()) {
			if (baselineShift() != oCompare.baselineShift()) {
				return false;
			}
		}
		if (localeEnable()) {
			if (locale() != oCompare.locale()) {
				return false;
			}
		}

		int i;

		for (i = 0; i < MAX_UNIT_INDEX; i++) {
			if (isUnitEnabled (i)) {
				if (! TextMeasurement.match (getUnitValue(i), oCompare.getUnitValue(i))) {
					return false;
				}
			}
		}

		for (i = 0; i < MAX_INT_INDEX; i++) {
			if (isIntEnabled (i)) {
				if (getIntValue (i) != oCompare.getIntValue (i)) {
					return false;
				}
			}
		}

		for (i = 0; i < MAX_BOOL_INDEX; i++) {
			if (isBoolEnabled (i)) {
				if (getBoolValue (i) != oCompare.getBoolValue (i)) {
					return false;
				}
			}
		}

		for (i = 0; i < MAX_FLOAT_INDEX; i++) {
			if (isFloatEnabled (i)) {
				if (getFloatValue (i) != oCompare.getFloatValue (i)) {
					return false;
				}
			}
		}

		return true;
	}

/**
 * Returns a hash code value for the object. This method is unsupported.
 * @exclude from published api.
 */
	public int hashCode() {
		int hashCode = 13;
		hashCode = (hashCode * 31) ^ Boolean.valueOf(fontEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(typefaceEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(encodingEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(sizeEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(weightEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(justifyVEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(justifyHEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(tabsEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(baselineShiftEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(localeEnable()).hashCode();
		hashCode = (hashCode * 31) ^ Boolean.valueOf(leaderContentEnable()).hashCode();
		hashCode = (hashCode * 31) ^ mbUnitEnable;
		hashCode = (hashCode * 31) ^ mbIntEnable;
		hashCode = (hashCode * 31) ^ mbBoolEnable;
		hashCode = (hashCode * 31) ^ mbFloatEnable;
		
		if (fontEnable())
			hashCode = (hashCode * 31) ^ fontInstance().hashCode();
		
		if (typefaceEnable())			
			hashCode = (hashCode * 31) ^ typeface().hashCode();
		
		if (encodingEnable())
			hashCode = (hashCode * 31) ^ encoding().hashCode();
		
		if (sizeEnable())
			hashCode = (hashCode * 31) ^ size().hashCode();
		
		if (weightEnable())
			hashCode = (hashCode * 31) ^ weight();
		
		if (justifyVEnable())
			hashCode = (hashCode * 31) ^ justifyV();
		
		if (justifyHEnable())
			hashCode = (hashCode * 31) ^ justifyH();
		
		if (tabsEnable())
			hashCode = (hashCode * 31) ^ tabs().hashCode();
		
		if (baselineShiftEnable())
			hashCode = (hashCode * 31) ^ baselineShift().hashCode();
		
		if (localeEnable())
			hashCode = (hashCode * 31) ^ locale().hashCode();
		
		for (int i = 0; i < MAX_UNIT_INDEX; i++)
			if (isUnitEnabled(i))
				hashCode = (hashCode * 31) ^ getUnitValue(i).hashCode();

		for (int i = 0; i < MAX_INT_INDEX; i++)
			if (isIntEnabled(i))
				hashCode = (hashCode * 31) ^ getIntValue(i);

		for (int i = 0; i < MAX_BOOL_INDEX; i++)
			if (isBoolEnabled(i))
				hashCode = (hashCode * 31) ^ Boolean.valueOf(getBoolValue(i)).hashCode();

		for (int i = 0; i < MAX_FLOAT_INDEX; i++)
			if (isFloatEnabled(i)) {
				long bits = Double.doubleToLongBits(getFloatValue(i));
				hashCode = (hashCode * 31) ^ (int)(bits ^ (bits >>> 32));
			}
		
		return hashCode;
	}
	
/**
 * Inequality comparison operator.
 * <p>
 * Compares on an attribute by attribute basis.  Two attributes are
 * considered unequal if their enabled/disabled settings don't match, or
 * if they are both enabled and their values are unequal.
 * @param oCompare - Text attribute object to compare against.
 * @return TRUE if the text attribute objects are considered not equal;
 * FALSE otherwise.
 */
	public boolean notEqual (TextAttr oCompare) {
		return ! equals (oCompare);
	}

	public static boolean match (TextAttr poAttr1, TextAttr poAttr2) {
		if (poAttr1 == poAttr2) {
			return true;
		}

		if ((poAttr1 == null) || (poAttr2 == null)) {
			return false;
		}

		return poAttr1.equals (poAttr2);
	}

/**
 * Return a reference to a constant default text attribute object.
 * <p>
 * Returns a reference to a pre-constructed text attribute object.	The
 * agttributes may all be disabled or they may all be set to default
 * values.
 * @param bDefault - FALSE if attributes are to be disabled; TRUE if
 * they are to be enabled with default values.
 * @return Reference to the default text attribute object.
 */
	public static TextAttr defaultAttr (boolean bDefault) {
		return bDefault ? defaultFull : defaultEmpty;
	}

	public void updateFont () {
		updateFont ((FontDesc) null);
	}

	public void updateFont (FontDesc oNewFontDesc) {
		FontDesc oFontDesc = new FontDesc (this);

// Override with any new settings
		if (oNewFontDesc != null) {
			if (oNewFontDesc.hasTypeface() != FontDesc.VALUE_ABSENT) {
				msTypeface = oNewFontDesc.getTypeface();
				oFontDesc.setTypeface (msTypeface);
			}
			if (oNewFontDesc.hasEncoding() != FontDesc.VALUE_ABSENT) {
				msEncoding = oNewFontDesc.getEncoding();
				oFontDesc.setEncoding (msEncoding);
			}
			if (oNewFontDesc.hasSize() != FontDesc.VALUE_ABSENT) {
				moSize = oNewFontDesc.getSize();
				oFontDesc.setSize (moSize);
			}
			if (oNewFontDesc.hasWeight() != FontDesc.VALUE_ABSENT) {
				miWeight = oNewFontDesc.getWeight();
				oFontDesc.setWeight (miWeight);
			}
			if (oNewFontDesc.hasItalic() != FontDesc.VALUE_ABSENT) {
				setBoolValue (BOOL_ITALIC, oNewFontDesc.getItalic());
				oFontDesc.setItalic (italic());
			}
			if (oNewFontDesc.hasHorizontalScale() != FontDesc.VALUE_ABSENT) {
				setFloatValue (FLOAT_HORIZONTAL_SCALE, oNewFontDesc.getHorizontalScale());
				oFontDesc.setHorizontalScale (horizontalScale());
			}
			if (oNewFontDesc.hasVerticalScale() != FontDesc.VALUE_ABSENT) {
				setFloatValue (FLOAT_VERTICAL_SCALE, oNewFontDesc.getVerticalScale());
				oFontDesc.setVerticalScale (verticalScale());
			}
		}

// Something not defined: drop the current font, otherwise map the new one
		if ((! oFontDesc.isValid()) || (fontService() == null)) {
			cleanupFont();
			return;
		}

		ReconcileInfo reconcile = reconcileFont (fontService(), oFontDesc);
		if (reconcile.mFontInstance != null) {
			updateFont (reconcile.mFontInstance, reconcile.mOriginalTypeface);
		}
	}

	public void updateFont (FontInstance oNewFont, String sOriginalTypeface, boolean bUpdateOptional) {
		if ((oNewFont == null) && (moFontInstance == null)) {
			return;
		}

		cleanupFont();

		if (oNewFont == null) {
			return;
		}

		moFontInstance = oNewFont;

// These settings always get enabled, either implicitly or explicitly.
		msTypeface = moFontInstance.getTypeface();
//		msEncoding = moFontInstance.getTextEncoding();	// TODO:
		moSize = moFontInstance.getSize();
		miWeight = FontDesc.coerceWeight (moFontInstance.getWeight());
		setBoolValue (BOOL_ITALIC, moFontInstance.getItalic());

// Horizontal and vertical scale were added later and get enabled only if
// the caller requests it or the value is not the default.  This is to
// prevent these settings appearing in certain markup outputs when they
// are defaulted.
		double dHorizontalScale = moFontInstance.getHorizontalScale();
		setFloatValue (FLOAT_HORIZONTAL_SCALE, dHorizontalScale, bUpdateOptional || (dHorizontalScale != 1.0));
		double dVerticalScale = moFontInstance.getVerticalScale();
		setFloatValue (FLOAT_VERTICAL_SCALE, dVerticalScale, bUpdateOptional || (dVerticalScale != 1.0));

		msOriginalTypeface = (sOriginalTypeface == null) ? "" : sOriginalTypeface;
	}

	public void updateFont (FontInstance oNewFont, String sOriginalTypeface) {
		updateFont (oNewFont, sOriginalTypeface, false);
	}

	public void updateFont (TextAttr oSource) {
		moFontInstance = oSource.fontInstance();
		msTypeface = oSource.typeface();
		msOriginalTypeface = oSource.originalTypeface();
		assert (msOriginalTypeface != null);
		msEncoding = oSource.encoding();
		moSize = oSource.size();
		miWeight = oSource.weight();
		setBoolValue (BOOL_ITALIC, oSource.italic());
		enableBool (BOOL_ITALIC, oSource.italicEnable());
		setFloatValue (FLOAT_HORIZONTAL_SCALE, oSource.horizontalScale(), oSource.horizontalScaleEnable());
		setFloatValue (FLOAT_VERTICAL_SCALE, oSource.verticalScale(), oSource.verticalScaleEnable());
	}

	void updateGfxTextAttr (TextGfxAttr oNewAttr) {
		cleanupGfxTextAttr();
		super.copyFrom (oNewAttr);
//		if ((AttrPool() != null) && GfxTextAttrEnable()) {
//			UpdateGfxTextAttr (AttrPool().AddAttr (GetGfxTextAttr()));
//		}
	}

//	void UpdateGfxTextAttr (GfxTextAttr poNewAttr) {
//		CleanupGfxTextAttr();
//		mpoPooledAttr = poNewAttr;
//		if (mpoPooledAttr == null) {
//			SetGfxTextAttrDefault (false);
//		}
//		else {
//			SetGfxTextAttr (mpoPooledAttr);
//		}
//	}

	static int getDefaultIntValue (int nIndex) {
		switch (nIndex) {
			case INT_RADIX_POS:
				return ~0;
			case INT_INVIS_CHAR:
				return ' ';
			case INT_DIGITS:
				return DIGITS_LOCALE;
			case INT_DIRECTION:
				return DIRECTION_NEUTRAL;
			case INT_PARA_DIRECTION:
				return DIRECTION_NEUTRAL;
			case INT_LAYOUT_ORIENTATION:
				return ORIENTATION_HORIZONTAL;
			case INT_LIGATURE:
				return LIGATURE_MINIMUM;
			case INT_HYPH_LEVEL:
				return HYPHEN_OFF;
			case INT_HYPH_MIN_WORD:
				return DEFAULT_HYPH_MIN_WORD;
			case INT_HYPH_MIN_PREFIX:
				return DEFAULT_HYPH_MIN_PREFIX;
			case INT_HYPH_MIN_SUFFIX:
				return DEFAULT_HYPH_MIN_SUFFIX;
			case INT_HYPH_MAX_LINES:
				return DEFAULT_HYPH_MAX_LINES;
			case INT_LEADER_PATTERN:
				return DEFAULT_LEADER_PATTERN;
			case INT_LEADER_ALIGN:
				return DEFAULT_LEADER_ALIGN;
			case INT_RULE_STYLE:
				return DEFAULT_RULE_STYLE;
			default:
				return 0;
		}
	}

	protected void onUpdateGfxTextAttr () {
		updateGfxTextAttr (this);
	}

	private void initialize (boolean bDefault) {
		mpoExtra = null;
//		mpoPooledAttr = null;
		mpoTabs = null;
		mbBoolValue = 0;
		mbBoolEnable = 0;
		mbUnitEnable = 0;
		mbIntEnable = 0;
		mbFloatEnable = 0;

		setDefault (bDefault);
	}

	private void needExtra () {
		if (mpoExtra == null) {
			mpoExtra = new AttrExtra();
			if (! spacingEnable()) {
				mpoExtra.moUnit[UNIT_SPACING] = goDefaultMeasureNeg;
			}
			if (! marginLEnable()) {
				mpoExtra.moUnit[UNIT_MARGIN_L] = goDefaultMeasureNeg;
			}
			if (! marginREnable()) {
				mpoExtra.moUnit[UNIT_MARGIN_R] = goDefaultMeasureNeg;
			}
			if (! spaceBeforeEnable()) {
				mpoExtra.moUnit[UNIT_SPACE_BEFORE] = goDefaultMeasureNeg;
			}
			if (! spaceAfterEnable()) {
				mpoExtra.moUnit[UNIT_SPACE_AFTER] = goDefaultMeasureNeg;
			}
			if (! leaderPatternWidthEnable()) {
				mpoExtra.moUnit[UNIT_LEADER_PATTERN_WIDTH] = goDefaultMeasureNeg;
			}
			if (! ruleThicknessEnable()) {
				mpoExtra.moUnit[UNIT_RULE_THICKNESS] = goDefaultMeasureNeg;
			}
		}
	}

// For debugging.
	public void debug () {
		debug (0);
	}

	void debug (int indent) {
		String prefix = Pkg.doIndent (indent+1);
		System.out.println (prefix + "Text attribute");
		prefix += ' ';
		if (typefaceEnable()) {
			System.out.println (prefix + "Typeface:      " + typeface());
		}
		if (encodingEnable()) {
			System.out.println (prefix + "Encoding:      " + encoding());
		}
		if (sizeEnable()) {
			System.out.println (prefix + "Size:          " + size());
		}
		if (weightEnable()) {
			System.out.println (prefix + "Weight:        " + weight());
		}
		if (italicEnable()) {
			System.out.println (prefix + "Italic:        " + italic());
		}
		if (underlineEnable()) {
			System.out.println (prefix + "Underline:     " + underline());
		}
		if (overlineEnable()) {
			System.out.println (prefix + "Overline:      " + overline());
		}
		if (strikeoutEnable()) {
			System.out.println (prefix + "Strikeout:     " + strikeout());
		}
		if (colourEnable()) {
			System.out.println (prefix + "Colour:        " + colour().r() + " " + colour().g() + " " + colour().b());
		}
		if (colourBgEnable()) {
			System.out.println (prefix + "ColourBg:      " + colourBg().r() + " " + colourBg().g() + " " + colourBg().b());
		}
		if (styleEnable()) {
			System.out.println (prefix + "Style:         " + style());
		}
		if (shadeEnable()) {
			System.out.println (prefix + "Shade:         " + shade());
		}
		if (shadeScaleEnable()) {
			System.out.println (prefix + "ShadeScale:    " + shadeScale());
		}
		if (transparentEnable()) {
			System.out.println (prefix + "Transparent:   " + transparent());
		}
		if (spacingEnable()) {
			System.out.println (prefix + "Spacing:       " + spacing());
		}
		if (marginLEnable()) {
			System.out.println (prefix + "MarginL:       " + marginL());
		}
		if (marginREnable()) {
			System.out.println (prefix + "MarginR:       " + marginR());
		}
		if (justifyVEnable()) {
			System.out.println (prefix + "JustifyV:      " + justifyV());
		}
		if (justifyHEnable()) {
			System.out.println (prefix + "JustifyH:      " + justifyH());
		}
		if (radixOffsetEnable()) {
			System.out.println (prefix + "RadixOffset:   " + radixOffset());
		}
		if (radixPosEnable()) {
			System.out.println (prefix + "RadixPos:      " + radixPos());
		}
		if (specialEnable()) {
			System.out.println (prefix + "Special:       " + special());
		}
		if (spaceBeforeEnable()) {
			System.out.println (prefix + "SpaceBefore:   " + spaceBefore());
		}
		if (spaceAfterEnable()) {
			System.out.println (prefix + "SpaceAfter:    " + spaceAfter());
		}
		if (invisibleEnable()) {
			System.out.println (prefix + "Invisible:     " + invisible());
		}
		if (invisCharEnable()) {
			System.out.println (prefix + "InvisChar:     " + invisChar());
		}
		if (baselineShiftEnable()) {
			System.out.println (prefix + "BaselineShift: " + baselineShift().getString (false));
		}
		if (localeEnable()) {
			System.out.println (prefix + "Locale:        " + locale());
		}
		if (digitsEnable()) {
			System.out.println (prefix + "Digits:        " + digits());
		}
		if (directionEnable()) {
			System.out.println (prefix + "Direction:     " + direction());
		}
		if (ligatureEnable()) {
			System.out.println (prefix + "Ligature:      " + ligature());
		}
		if (tabsEnable()) {
			tabs().debug (indent+1);
		}
// TODO: add in new attributes
	}

	private boolean bitTest (int nIndex, int nMask) {
		return (nMask & (1 << nIndex)) != 0;
	}

	private int bitSet (int nIndex, int nMask, boolean bValue) {
		if (bValue) {
			return nMask | (1 << nIndex);
		} else {
			return nMask & ~(1 << nIndex);
		}
	}

	private boolean getBoolValue (int nIndex) {
		return bitTest (nIndex, mbBoolValue);
	}

	private void setBoolValue (int nIndex, boolean bNewValue) {
		mbBoolValue = bitSet (nIndex, mbBoolValue, bNewValue);
		enableBool (nIndex, true);
	}

	private boolean isBoolEnabled (int nIndex) {
		return bitTest (nIndex, mbBoolEnable);
	}

	private void enableBool (int nIndex, boolean bEnable) {
		mbBoolEnable = bitSet (nIndex, mbBoolEnable, bEnable);
	}

	private TextMeasurement getUnitValue (int nIndex) {
		if (mpoExtra == null) {
			if (! isUnitEnabled (nIndex)) {
				switch (nIndex) {
					case UNIT_SPACING:
					case UNIT_MARGIN_L:
					case UNIT_MARGIN_R:
					case UNIT_SPACE_AFTER:
					case UNIT_LEADER_PATTERN_WIDTH:
					case UNIT_RULE_THICKNESS:
						return goDefaultMeasureNeg;
				}
			}
			return TextMeasurement.zero();
		} else {
			return mpoExtra.moUnit[nIndex];
		}
	}

	private void setUnitValue (int nIndex, TextMeasurement oNewValue) {
		boolean bEnable = true;
		boolean bSet = false;

		if ((nIndex == UNIT_SPACING)
		 || (nIndex == UNIT_MARGIN_L)
		 || (nIndex == UNIT_MARGIN_R)
		 || (nIndex == UNIT_SPACE_AFTER)
		 || (nIndex == UNIT_LEADER_PATTERN_WIDTH)
		 || (nIndex == UNIT_RULE_THICKNESS)) {
			int nValue = oNewValue.getLength().value();		// TODO: does this logic work for relative values?
			if (nValue > 0) {				// a "real" value ...
				bSet = true;				// ... must be set
			} else {						// a "fake" value (0=enabled, -ve=disabled)
				if (mpoExtra != null) {		// if we're already storing values ...
					bSet = true;			// ... we'll have to set this one
				}
				if (nValue < 0) {
					bEnable = false;		// check special disabled value
				}
			}
		} else if (oNewValue != getUnitValue (nIndex)) {	// normal behaviour ...
			bSet = true;
		}

		if (bSet) {
			needExtra();
			mpoExtra.moUnit[nIndex] = oNewValue;
		}

		enableUnit (nIndex, bEnable);
	}

	private boolean isUnitEnabled (int nIndex) {
		return bitTest (nIndex, mbUnitEnable);
	}

	private void enableUnit (int nIndex, boolean bEnable) {
		mbUnitEnable = bitSet (nIndex, mbUnitEnable, bEnable);
	}

	private int getIntValue (int nIndex) {
		if (mpoExtra == null) {
			return getDefaultIntValue (nIndex);
		} else {
			return mpoExtra.moInt[nIndex];
		}
	}

	private void setIntValue (int nIndex, int oNewValue) {
		if (oNewValue != getIntValue (nIndex)) {
			needExtra();
			mpoExtra.moInt[nIndex] = oNewValue;
		}
		enableInt (nIndex, true);
	}

	public double getFloatValue (int nIndex) {
		if (! isFloatEnabled (nIndex)) {
			return -1.0;
		} else if (mpoExtra == null) {
			return 1.0;
		} else {
			return mpoExtra.moFloat[nIndex];
		}
	}

	public void setFloatValue (int nIndex, double nNewValue, boolean bEnable) {
		if (nNewValue < 0) {
			bEnable = false;
		}
		if (nNewValue != getFloatValue (nIndex)) {
			needExtra();
			mpoExtra.moFloat[nIndex] = nNewValue;
		}
		if (bEnable) {
			enableFloat (nIndex, true);
		}
	}

	public void setFloatValue (int nIndex, double nNewValue) {
		setFloatValue (nIndex, nNewValue, true);
	}

	public boolean isFloatEnabled (int nIndex) {
		return bitTest (nIndex, mbFloatEnable);
	}

	public void enableFloat (int nIndex, boolean bEnable) {
		mbFloatEnable = bitSet (nIndex, mbFloatEnable, bEnable);
	}

	private boolean isIntEnabled (int nIndex) {
		return bitTest (nIndex, mbIntEnable);
	}

	private void enableInt (int nIndex, boolean bEnable) {
		mbIntEnable = bitSet (nIndex, mbIntEnable, bEnable);
	}

	private ReconcileInfo reconcileFont (FontService poFontService, FontDesc oFontDesc) {
		FontInfo oFontInfo = new FontInfo (oFontDesc.getTypeface(), oFontDesc.getWeight(), oFontDesc.getItalic());
		double dHorizontalScale = oFontDesc.getHorizontalScale();
		if (dHorizontalScale < 0.0) {
			dHorizontalScale = 1.0;
		}
		double dVerticalScale = oFontDesc.getVerticalScale();
		if (dVerticalScale < 0.0) {
			dVerticalScale = 1.0;
		}
		FontInstance oFontInstance = poFontService.reconcile (oFontInfo, size(), dHorizontalScale, dVerticalScale);
		assert (oFontInstance != null);								// TODO:
		String sOriginalTypeface = (oFontInfo.equals (oFontInstance.getFontItem())) ? "" : oFontDesc.getTypeface();
		return new ReconcileInfo (oFontInstance, sOriginalTypeface);
	}

	private void cleanupGfxTextAttr () {
//		if (mpoPooledAttr != null) {
//			AttrPool().RemoveAttr (mpoPooledAttr);
//			mpoPooledAttr = null;
//		}
	}

	private void cleanupFont () {
		if (moFontInstance != null) {
			msOriginalTypeface = "";
			moFontInstance = null;
		}
	}

//	private void cleanupAllGfx () {
//		cleanupFont();
//		cleanupGfxTextAttr();
//	}

	private boolean canFlattenBaselineShift () {
		if (! baselineShiftEnable()) {
			return false;
		}

		TextBaselineShift oOldShift = baselineShift();
		if (oOldShift.isNeutral()) {
			return false;
		}

		if ((oOldShift.getType() != TextBaselineShift.SUBSCRIPT)
		 && (oOldShift.getType() != TextBaselineShift.SUPERSCRIPT)) {
			return false;
		}

		return true;
	}
}
