/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.api;

import org.apache.jackrabbit.api.security.user.User;
import org.apache.sling.api.resource.Resource;

import aQute.bnd.annotation.ProviderType;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Collection;

import javax.jcr.Binary;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

/**
 * The <code>AssetManager</code> provides utility methods for assets.
 * <p>
 * The asset manager can be retrieved as follows:
 * <blockquote><pre>
 *   ...
 *   AssetManager manager = resourceResolver.adaptTo(AssetManager.class);
 *   ...
 * </pre></blockquote>
 */

@ProviderType
public interface AssetManager {

    /**
     * Restores an {@code Asset}.
     *
     * @param revisionId revision id
     *
     * @return the restored {@link Asset}
     *
     * @throws Exception Upon encountering an error while restoring an asset.
     */
    Asset restore(String revisionId) throws Exception;

    /**
     * Lists all available {@link Revision}s.
     *
     * @param path path of asset
     * @param cal  starting date to search revisions or {@code null} in order to fetch all
     *
     * @return all available {@link Revision}s
     *
     * @throws Exception Upon encountering an error while getting the revisions for an asset.
     */
    Collection<Revision> getRevisions(String path, Calendar cal) throws Exception;

    /**
     * This method creates the complete asset structure in <i>/content/dam</i>. The following structure is created:
     * &lt;xmp&gt; + file.jpg (dam:Asset) + jcr:content )(dam:AssetContent) + renditions (sling:OrderedFolder) + metadata
     * (nt:unstructured)
     * <p>
     * &lt;/xmp&gt;
     * <p>
     * The given <code>binaryPath</code> is translated into the final asset location (see {@link
     * com.day.cq.dam.commons.util.DamUtil#binaryToAssetPath(String)}).
     * <p>
     *
     * @param binaryPath The path of the asset's binary in <i>/var/dam</i>.
     * @param doSave     Whether the repository changes are saved or not.
     *
     * @return The newly created asset or {@code null} if the binaryPath cannot be handled
     *
     * @see <a href="https://github.com/adobe/aem-upload">aem-upload</a>
     * @deprecated In AEM as a Cloud Service, assets should instead take advantage of direct binary access.  A
     * discussion of this change as well as an SDK to ease implementation of this new pattern can be found at
     * <a href="https://github.com/adobe/aem-upload">aem-upload</a>.
     */
    @Deprecated
    Asset createAssetForBinary(String binaryPath, boolean doSave);

    /**
     * Returns the {@link Resource} of the {@link Asset} corresponding to the binary given with the <code>path</code>
     * parameter.
     * <blockquote><pre>
     *   DamUtil.getAssetNode("/var/dam/myfolder/test.jpg", session)    =  Resource("/content/dam/myfolder/test.jpg")
     * </pre></blockquote>
     *
     * @param binaryPath The path of the binary.
     *
     * @return The resource representing the binary's asset, or <code>null</code> if it couldn't be found.
     *
     * @see <a href="https://github.com/adobe/aem-upload">aem-upload</a>
     * @deprecated This method was used by an older paradigm of asset upload, where a binary would be uploaded to /var
     * with an asset separately created under /content.  In AEM as a Cloud Service, assets should instead take advantage
     * of direct binary access.  As a result, this method should no longer be needed.  A discussion of this change as
     * well as an SDK to ease implementation of this new pattern can be found at
     * <a href="https://github.com/adobe/aem-upload">aem-upload</a>.
     */
    @Deprecated
    Asset getAssetForBinary(String binaryPath);

    /**
     * Removes the {@link javax.jcr.Node} of the {@link Asset} corresponding to the binary given with the
     * <code>path</code> parameter.
     * <blockquote><pre>
     *   DamUtil.removeAssetNode("/var/dam/myfolder/test.jpg", session) =i&gt; Delete Node("/content/dam/myfolder/test.jpg")
     * </pre></blockquote>
     *
     * @param binaryPath The path of the binary.
     *
     * @return <code>true</code> if the asset was successfully removed.
     *
     * @see <a href="https://github.com/adobe/aem-upload">aem-upload</a>
     * @deprecated This method was used by an older paradigm of asset upload, where a binary would be uploaded to /var
     * with an asset separately created under /content.  In AEM as a Cloud Service, assets should instead take advantage
     * of direct binary access.  As a result, this method should no longer be needed.  A discussion of this change as
     * well as an SDK to ease implementation of this new pattern can be found at
     * <a href="https://github.com/adobe/aem-upload">aem-upload</a>.
     */
    @Deprecated
    boolean removeAssetForBinary(String binaryPath);

    /**
     * Creates a new {@link Asset} at the given <code>path</code>. If an asset
     * already exists at the given <code>path</code>, its original rendition is
     * updated instead of creating a new asset. If inputStream is null new
     * {@link Asset} is created without original rendition. If an asset already
     * exists at given path and inputstream is null, original rendition is not
     * updated.
     *
     * @param path The path of the asset to be created.
     * @param is The input stream of the new asset's original binary.
     * @param mimeType The mime type of the new asset's original binary.
     * @param doSave Whether the repository changes are saved or not.
     * @return The newly created asset or {@code null} if something exists at the given path already
     *         or there was an unexpected error
     * @see <a href="https://github.com/adobe/aem-upload">aem-upload</a>
     * @deprecated In AEM as a Cloud Service, assets should instead take advantage of direct binary access.  A
     * discussion of this change as well as an SDK to ease implementation of this new pattern can be found at
     * <a href="https://github.com/adobe/aem-upload">aem-upload</a>.
     */
    @Deprecated
    Asset createAsset(String path, InputStream is, String mimeType, boolean doSave);

    /**
     * Creates a new {@link Asset} at the given <code>path</code>. If an asset already exists at the given
     * <code>path</code>, its original rendition is updated instead of creating a new asset. If binary is null new
     * {@link Asset} is created without original rendition. If an asset already exists at given path, original rendition
     * is not updated.
     *
     * @param path     The path of the asset to be created.
     * @param binary   The binary for new asset's original
     * @param mimeType The mime type of the new asset's original binary.
     * @param doSave   Whether the repository changes are saved or not.
     * @return The newly created asset or {@code null} if something exists at the given path already or there was an
     * unexpected error
     */
    Asset createOrUpdateAsset(final String path, final Binary binary, final String mimeType, final boolean doSave);

    /**
     * Creates a new {@link Asset} at the given <code>path</code>. If an asset already exists at the given
     * <code>path</code>, its original rendition is updated instead of creating a new asset. If binary is null new
     * {@link Asset} is created without original rendition. If an asset already exists at given path, original rendition
     * is not updated. If asset already exists and create revision is true, the given revision label and revision comment
     * will be used to create a new revision.
     *
     * @param path     The path of the asset to be created.
     * @param binary   The binary for new asset's original
     * @param mimeType The mime type of the new asset's original binary.
     * @param doSave   Whether the repository changes are saved or not.
     * @param createRevision  Whether or not a new revision will be created.
     * @param revisionLabel   The label to associate with the newly created revision.
     * @param revisionComment The comment to associate with the newly created revision.
     * @return The newly created asset or {@code null} if something exists at the given path already or there was an
     * unexpected error
     */
    Asset createOrUpdateAsset(
        final String path, 
        final Binary binary, 
        final String mimeType, 
        final boolean doSave, 
        final boolean createRevision,
        final String revisionLabel,
        final String revisionComment);

    /**
     * Creates a new {@link Asset} at the given <code>path</code>. If an asset already exists at the given
     * <code>path</code>, it will be removed and re-created. If binary is null new {@link Asset} is created without original 
     * rendition.
     *
     * @param path     The path of the asset to be created.
     * @param binary   The binary for new asset's original
     * @param mimeType The mime type of the new asset's original binary.
     * @param doSave   Whether the repository changes are saved or not.
     * @return The newly created asset or {@code null} there was an unexpected error.
     */
    Asset createOrReplaceAsset(final String path, final Binary binary, final String mimeType, final boolean doSave);

    /**
     * Create a new {@link Revision} of the asset. The revision will be created as a standard JCR version of the
     * underlying asset node. Owner of the session through which {@link Asset} was created is added as
     * {@link DamConstants.PN_VERSION_CREATOR} of the asset.
     *
     * @param asset   The asset to version
     * @param label   version label
     * @param comment version comment
     *
     * @return The created revision
     *
     * @throws Exception Thrown when an error during version creation occurred.
     */
    Revision createRevision(Asset asset, String label, String comment) throws Exception;

    /**
     * Create a new {@link Revision} of the asset. The revision will be created as a standard JCR version of the
     * underlying asset node. This API allows specifying principal of an AEM User to be added a
     * {@link DamConstants.PN_VERSION_CREATOR} of the asset being versioned.
     *
     * @param asset   The asset to version
     * @param label   version label
     * @param comment version comment
     * @param versionCreator version creator. If null, this API is equivalent to
     *                  {@link createRevision(Asset asset, String label, String comment)}
     *
     * @return The created revision
     *
     * @throws Exception Thrown when an error during version creation occurred.
     */
    Revision createRevision(Asset asset, String label, String comment, User versionCreator) throws Exception;

    /**
     * Ask Asset ID Provider (associated with Asset Manager) to assign ID
     * (if needed) to an asset and establish its parentage (dam:parentAssetID)
     *
     * @param asset   The asset to update
     *
     * @return The ID assigned to the asset
     *
     * @throws RepositoryException thrown if an error occurs while accessing the asset
     * @throws PathNotFoundException thrown if no item exists in the path
     */
    String assignAssetID(Asset asset) throws PathNotFoundException, RepositoryException;
}
