/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.wcm.api.components;

import java.util.HashSet;
import java.util.Set;

import javax.servlet.ServletRequest;

import com.day.cq.wcm.api.Page;

/**
 * <code>IncludeOptions</code> provide constants for the next component include
 */
public class IncludeOptions {

    /**
     * Name of the request attribute
     */
    public static String ATTR_NAME = IncludeOptions.class.getName();

    /**
     * forces the current page
     */
    private Page currentPage;

    /**
     * forces the next include to be in the same context
     */
    private boolean sameContext = false;

    /**
     * forces creation of an edit context
     */
    private boolean forceEditContext = false;

    /**
     * forces the cell name of the next include
     */
    private String cellName;

    /**
     * additional css class names
     */
    private Set<String> cssClassNames = new HashSet<String>();

    /**
     * decoration tag name override
     */
    private String decorationTagName;

    /**
     * Returns the include options as stored in the current request. If no such
     * request attribute is defined but <code>force</code> is <code>true</code>,
     * new include options are created and stored as request attribute.
     *
     * @param req the request
     * @param force flag to force creation
     * @return the include options or <code>null</code>
     */
    public static IncludeOptions getOptions(ServletRequest req, boolean force) {
        IncludeOptions opts = (IncludeOptions) req.getAttribute(ATTR_NAME);
        if (opts == null && force) {
            opts = new IncludeOptions();
            req.setAttribute(ATTR_NAME, opts);
        }
        return opts;
    }

    /**
     * Clears the include options from the request attributes and returns it.
     * @param req the request
     * @return the include options or <code>null</code> if never defined.
     */
    public static IncludeOptions clear(ServletRequest req) {
        IncludeOptions opts = (IncludeOptions) req.getAttribute(ATTR_NAME);
        if (opts != null) {
            req.removeAttribute(ATTR_NAME);
        }
        return opts;
    }

    /**
     * Sets the option to force the next include to be in the same edit context
     * as the current one. i.e. no new edit context is created.
     * @param force <code>true</code> if no new edit context is to be created.
     * @return this include options.
     */
    public IncludeOptions forceSameContext(boolean force) {
        this.sameContext = force;
        return this;
    }

    /**
     * Checks if the same edit context is requested.
     * @return <code>true</code> if the next include should use the same edit
     *         context.
     */
    public boolean isSameContext() {
        return sameContext;
    }

    /**
     * Sets the option to force a particular current page for the next include.
     * @param currentPage the forced page
     * @return this include option
     */
    public IncludeOptions forceCurrentPage(Page currentPage) {
        this.currentPage = currentPage;
        return this;
    }

    /**
     * Returns the currentPage for the next include if forced, or null.
     * @return the forced page or <code>null</code>
     */
    public Page getCurrentPage() {
        return currentPage;
    }

    /**
     * Sets the option to force the cell name of the next include
     * @param cellName the cell name
     * @return this include option
     */
    public IncludeOptions forceCellName(String cellName) {
        this.cellName = cellName;
        return this;
    }

    /**
     * Returns the cell name for the next include.
     * @return the cell name or <code>null</code>
     */
    public String getCellName() {
        return cellName;
    }

    /**
     * Returns a modifiable set of additional css class names to be added to the
     * including &lt;div&gt; tag.
     * @return the array of names
     */
    public Set<String> getCssClassNames() {
        return cssClassNames;
    }

    /**
     * Checks a new edit context is forced.
     * @return <code>true</code> if forced.
     */
    public boolean isForceEditContext() {
        return forceEditContext;
    }

    /**
     * Define if a new edit context should be openend.
     * @param forceEditContext <code>true</code> if a new edit context
     * should be openend.
     */
    public void forceEditContext(boolean forceEditContext) {
        this.forceEditContext = forceEditContext;
    }

    /**
     * Returns the decoration tag name override.
     * @return the decoration tag name.
     */
    public String getDecorationTagName() {
        return decorationTagName;
    }

    /**
     * Sets the decoration tag name override. This override any automatic
     * decoation tag name computation in the wcm request filter or explicit
     * definition by the component. If set to an empty string, no decoration
     * tag will be rendered at all. Note that the decoration tag name defined
     * in the include options will only be applied to the component context before
     * inclusion and will not be inheirted to the sub contextes.
     * 
     * @param decorationTagName the decoration tag name.
     * @see ComponentContext#setDecorationTagName(String)
     * @since 5.3
     */
    public void setDecorationTagName(String decorationTagName) {
        this.decorationTagName = decorationTagName;
    }
}