/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2021 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.adobe.cq.dam.cfm.extensions;

import com.adobe.cq.dam.cfm.SemanticDataType;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.osgi.annotation.versioning.ProviderType;

import java.util.Calendar;

/**
 * Provides services that allow to convert different date and time representations.
 *
 * <p>Following representations are supported:</p>
 *
 * <ul>
 *     <li>Original - any {@code Calendar} object or date/time-related {@code String}
 *     representation; may have any timezone setting and also unnormalized fields (see
 *     description of the semantic representation below for more information about
 *     normalization).</li>
 *     <li>Semantic - normalized representations of an original {@code Calendar}
 *     object or date/time-related or {@code String} representation. For the {@code date}
 *     sub-type, the time is guaranteed to be <i>00:00:00.000</i> and the semantic
 *     {@code Calendar} object to be in the <i>UTC</i> timezone. For the {@code time}
 *     sub-type, the date is guaranteed to be set to <i>1970-01-01</i> and the semantic
 *     {@code Calendar} object to be in the <i>UTC</i> timezone.</li>
 * </ul>
 */
@ProviderType
public interface SemanticDateTimeFormatter {

    /**
     * Maps an original Calendar object to a semantic Calendar object.
     *
     * @param value The {@code Calendar} object to map
     * @param semanticType The semantic type to map {@code value} to; supported values are:
     *                     {@link SemanticDataType#DATE}, {@link SemanticDataType#TIME}
     *                     and {@link SemanticDataType#DATETIME}
     * @return The mapped semantic {@code Calendar}
     */
    @Nullable
    Calendar map(@NotNull Calendar value, @NotNull String semanticType);

    /**
     * Parses a {@code String} value to a semantic {@code Calendar} object.
     *
     * <p>The format supported for strings is
     * {@code [HH:mm[:ss]][yyyy-MM-dd['T'HH:mm:ss[.SSS][XXX]]}, which basically resembles
     * ISO-8601.</p>
     *
     * @param value The string to parse
     * @param semanticType The semantic type to map {@code value} to; supported values are:
     *                     {@link SemanticDataType#DATE}, {@link SemanticDataType#TIME}
     *                     and {@link SemanticDataType#DATETIME}
     * @return The semantic {@code Calendar} object that matches the parsed string;
     *         {@code null} if the string doesn't contain a valid date
     */
    @Nullable
    Calendar parse(@NotNull String value, @NotNull String semanticType);

    /**
     * Gets a string representation of the specified {@code Calendar} value for the given
     * semantic type.
     *
     * <p>The string representation will be <i>hh:mm:ss[.n...]</i> (nanosecond portion is
     * optional) for the {@code time} semantic type and <i>yyyy-mm-dd</i> for {@code date}.
     * The string representation for the {@code datetime} semantic type will be formatted
     * as an ISO-8601-compliant string.</p>
     *
     * @param value The {@code Calendar} object to create a string representation for
     * @param semanticType the semantic type for the string representation
     * @return The string representation
     */
    @Nullable
    String format(@NotNull Calendar value, @NotNull String semanticType);

    /**
     * Maps an original {@code String} representation to a suitable semantic {@code String}
     * representation.
     *
     * <p>The format supported for the returned value is
     * {@code [HH:mm[:ss]][yyyy-MM-dd['T'HH:mm:ss[.SSS][XXX]]}, which basically resembles
     * ISO-8601.</p>
     *
     * @param value The string to map
     * @param semanticType The semantic type
     * @return The semantic string
     */
    @NotNull
    String map(@NotNull String value, @NotNull String semanticType);

}
