/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
/**
 * AdobePatentID="3435US01"
 */
package com.adobe.cq.launches.api;

import java.util.Calendar;
import java.util.Collection;
import java.util.List;

import javax.jcr.RangeIterator;

import org.apache.sling.api.resource.Resource;

/**
 * The <code>LaunchManager</code> provides methods to deal with launches.
 */
public interface LaunchManager {

    /**
     * Checks if the provided resource is contained within one or more launches
     *
     * @param resource The resource to check
     * @return <code>true</code> if the provided resource is contained within one or more launches
     */
    public boolean isInLaunch(Resource resource);

    /**
     * Options object used for {@link #createLaunch(CreateOptions)}.
     */
    public static class CreateOptions {
        /**
         * Source resource.
         */
        public Resource resource;
        /**
         * LaunchSource List.
         */
        public List<LaunchSource> launchSourceList;
        /**
         * The title of the new launch.
         */
        public String title;
        /**
         * The target live date of the launch.
         */
        public Calendar liveDate;
        /**
         * If {@code true}, child resources of provided source resource will be included in the launch.
         */
        public boolean isDeep;
        /**
         * If {@code true}, the launch will be a live copy of the current production version.
         */
        public boolean isLiveCopy;
        /**
         * A new template to use for the launch page or null if none
         */
        public String template;
        /**
         * The MSM rollout configurations to use in the source launch relationship.
         */
        public String[] sourceRolloutConfigs;
        /**
         * The MSM rollout configurations to use when promoting this launch.
         */
        public String[] promoteRolloutConfigs;
        /**
         * If {@code true}, after the promotion, the launch sources will be published
         */
        public boolean isProductionReady;
        /**
         * The promotion scope of the launch
         */
        public LaunchPromotionScope promotionScope;
    }

    /**
     * Create a new launch.
     *
     * @param options the options for creating the launch. {@code resource} and {@code title} are required.
     * @return the created launch
     * @throws LaunchException if an error occurs while creating the launch
     */
    public Launch createLaunch(CreateOptions options) throws LaunchException;

    /**
     * Create a new launch
     *
     * @param srcResource Source resource
     * @param title       The title of the new launch
     * @param liveDate    The target live date of the launch
     * @param isDeep      If <code>true</code>, child resources of provided
     *                    source resource will be included in the launch
     * @param isLiveCopy  If <code>true</code>, the launch will
     *                    be a live copy of the current production version
     * @return The created launch
     * @throws LaunchException if an error occurs while creating the launch
     */
    public Launch createLaunch(Resource srcResource, String title, Calendar liveDate, boolean isDeep, boolean isLiveCopy) throws LaunchException;

    /**
     * Update launch source list. This method will add new launchSource and remove existing launchSource from
     * specified launch.
     * @param launch The launch to update
     * @param newLaunchSourceList updated LaunchSource list
     * @return the updated launch
     * @throws LaunchException if an error occurs while updating the launch
     */
    public Launch updateLaunchSources(Launch launch, List<LaunchSource> newLaunchSourceList) throws LaunchException;

    /**
     * Get a launch from its path
     *
     * @param absPath Launch absolute path
     * @return The <code>Launch</code> object loaded from the path or
     * <code>null</code> if the path does not resolve to a launch resource.
     */
    public Launch getLaunch(String absPath);

    /**
     * Get every launches the provided resource is contained in
     *
     * @param resource The resource
     * @return A collection of launches
     * @throws LaunchException if an error occurs while getting the launches
     */
    public Collection<Launch> getLaunches(Resource resource) throws LaunchException;

    /**
     * Promote a launch to become the production version
     *
     * @param launch The launch to promote
     * @param params Launch promotion parameters
     * @throws LaunchException if an error occurs while promoting the launch
     */
    public void promoteLaunch(Launch launch, LaunchPromotionParameters params) throws LaunchException;

    /**
     * Delete a launch
     *
     * @param launch The launch to delete
     * @throws LaunchException if an error occurs while deleting the launch
     */
    public void deleteLaunch(Launch launch) throws LaunchException;

    /**
     * Clone a launch
     *
     * @param launch The launch to clone
     * @param cloneTitle Title to give to the clone
     * @param liveDate   The target live date of the cloned launch
     * @param isLiveCopy If <code>true</code>, the cloned launch will
     *                   be a live copy of the current production version
     * @return launch object
     * @throws LaunchException if an error occurs while cloning the launch
     */
    public Launch cloneLaunch(Launch launch, String cloneTitle, Calendar liveDate, boolean isLiveCopy) throws LaunchException;

    /**
     * Returns a collection of LaunchResourceStatuses comparing resources in a
     * launch to their production version. The given start resource does
     * not need to be a {@link Launch#getSourceRootResource() Launch root resource}.
     *
     * @param launch The launch to compare
     * @param startResource {@link Resource } resource to start comparison from
     * @param deep If <code>true</code> the child resources will be also analyzed
     * @return launch resource statuses
     * @throws LaunchException if an error occurs while getting resource status
     */
    public RangeIterator getResourcesStatus(Launch launch, Resource startResource, boolean deep) throws LaunchException;

    /**
     * Returns a collection of LaunchResourceStatuses comparing resources in a launch to the specified
     * target launch or the production version (if <code>target</code> is <code>null</code>). The given start
     * resource does not need to be a {@link Launch#getSourceRootResource() Launch root resource}.
     *
     * @param launch The launch to compare
     * @param startResource {@link Resource } resource to start comparison from
     * @param deep If <code>true</code> the child resources will be also analyzed
     * @param target The target launch to compare to or <code>null</code> to compare to production
     * @return launch resource statuses
     * @throws LaunchException if an error occurs while getting resource status
     */
    public RangeIterator getResourcesStatus(Launch launch, Resource startResource, boolean deep, Launch target) throws LaunchException;
    
    /**
     * Returns a collection of LaunchResourceStatuses comparing resources in a launch to the specified
     * target launch or the production version (if <code>target</code> is <code>null</code>). The given start
     * resource does not need to be a {@link Launch#getSourceRootResource() Launch root resource}.
     *
     * @param launch The launch to compare
     * @param startResource {@link Resource } resource to start comparison from
     * @param launchPromotionScope based on promotion scope the child resources will be also analyzed
     * @param target The target launch to compare to or <code>null</code> to compare to production
     * @return launch resource statuses
     * @throws LaunchException if an error occurs while getting resource status
     */
    public RangeIterator getResourcesStatus(Launch launch, Resource startResource, LaunchPromotionScope launchPromotionScope,
        Launch target) throws LaunchException;
}
