/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.targetrecommendations.api;

import java.util.List;

import org.osgi.annotation.versioning.ProviderType;

import com.adobe.cq.targetrecommendations.api.model.ProductFeed;
import com.adobe.cq.targetrecommendations.api.model.RecommendationAlgorithm;
import com.adobe.cq.targetrecommendations.api.model.RecommendationMbox;
import com.adobe.cq.targetrecommendations.api.model.RecommendationTemplate;
import com.adobe.cq.targetrecommendations.api.model.TargetRecommendation;
import com.day.cq.wcm.webservicesupport.Configuration;


/**
 * Provides support for the Target Recommendations APIs
 */
@ProviderType
public interface TargetRecommendationsAPIClient {
    
    /**
     * Template management APIs
     */
    
    /**
     * Get the list of available templates associated with a given TesT&amp;Target {@link Configuration}
     * 
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * 
     * @return a {@link List} containing the {@link RecommendationTemplate} objects available for the given T&amp;T client 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public List<RecommendationTemplate> getTemplates(Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Get a single template having a given id
     * 
     * @param id - a {@code Integer} value identifying a recommendation template
     * @param ttConfig - the {@code Configuration} that identifies the Target config
     * 
     * @return a {@code RecommendationTemplate} if a valid template is found for the given id and configuration 
     * or {@code null} if none is found
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}$
     */
    public RecommendationTemplate getTemplate(int id, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Saves a template in Target Recommendations backend using the given T&amp;T cloud configuration
     * 
     * @param template - the {@code RecommendationTemplate} that needs to be persisted in the Recommendations backend
     * @param ttConfig - the Target {@code Configuration}
     * 
     * @return the saved {@code RecommendationTemplate} 
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException} on any issues encountered during template update operation
     */
    public RecommendationTemplate saveTemplate(RecommendationTemplate template, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Deletes a template from the Target Recommendations backend
     * 
     * @param id - the {@code Integer} id of the template that needs to be deleted
     * @param ttConfig - the Target {@code Configuration}
     * @throws TargetRecommendationsException {@link TargetRecommendationsException} on any issues encountered during template deletion
     */
    public void deleteTemplate(int id, Configuration ttConfig) throws TargetRecommendationsException;
    
    
    /**
     * MBoxes APIs
     */
    
    /**
     * Get the list of available MBoxes for a given T&amp;T configuration
     * 
     * @param ttConfig - the Target {@code Configuration}
     * @return a {@code List} containing the MBoxes available for the given T&amp;T client
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public List<RecommendationMbox> getMBoxes(Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Recommendation Algorithm APIs
     */
    
    /**
     * Get a list of available algorithms for the given TesT&amp;Target {@code Configuration}
     * 
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * @return a {@link List} containing the {@link RecommendationAlgorithm} objects available for the given T&amp;T client
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public List<RecommendationAlgorithm> getAlgorithms(Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Get a single algorithm from a given id
     * 
     * @param id - the Target Recommendations id of the needed algorithm
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * 
     * @return a {@code RecommendationAlgorithm} object if a valid algorithm with the given id is found in Target Recommendations
     * or {@code null} if none is found
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public RecommendationAlgorithm getAlgorithm(int id, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Saves an algorithm in Target Recommendations using the given cloud configuration
     * 
     * @param algorithm - the {@code RecommendationAlgorithm} to be updated in Target Recommendations
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     *
     * @return the saved {@code RecommendationAlgorithm}
     * 
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public RecommendationAlgorithm saveAlgorithm(RecommendationAlgorithm algorithm, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Deletes an algorithm from Target Recommendations
     * 
     * @param id - the Target Recommendations id of the algorithm to be deleted
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public void deleteAlgorithm(int id, Configuration ttConfig) throws TargetRecommendationsException;
    
    
    /**
     * Recommendations APIs
     */
    
    /**
     * Get all recommendations related to the given TesT&amp;Target {@code Configuration}
     * 
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * @return a {@code List} containing all the {@code TargetRecommendation} objects available to the given T&amp;T client
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public List<TargetRecommendation> getRecommendations(Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Get a single recommendation from a given id
     * 
     * @param id - the Target Recommendations id of the required recommendation
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * 
     * @return a {@code TargetRecommendation} object if a valid recommendation with the given id is found in Target Recommendations
     * or {@code null} if none is found
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public TargetRecommendation getRecommendation(int id, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Saves a given {@link TargetRecommendation} in Target Recommendations system using the given cloud configuration
     * 
     * @param recommendation - the {@link TargetRecommendation} recommendation object to be persisted in Target Recommendations
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * 
     * @return the saved {@code TargetRecommendation}
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException} 
     */
    public TargetRecommendation saveRecommendation(TargetRecommendation recommendation, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Removes a recommendation from Target Recommendations
     * 
     * @param id - the recommendation id
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public void deleteRecommendation(int id, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Product feed APIs
     */
    
    /**
     * Get all defined product feeds in Target Recommendations related to a given TesT&amp;Target configuration
     * 
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * 
     * @return a {@code List} containing all the {@code ProductFeed} feeds defined for the given T&amp;T account
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public List<ProductFeed> getFeeds(Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Get a single product feed based on a given id
     * 
     * @param id - the Target Recommendations id of the required product feed
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * 
     * @return a {@code ProductFeed} abstraction of the recommendations product feed if one with the given id is found in Target Recommendations backend
     * or {@code null} if none is found
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public ProductFeed getFeed(int id, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Saves the given {@code ProductFeed} in Target Recommendations using the given cloud configuration
     * 
     * @param feed - the {@code ProductFeed} to be updated in Target Recs
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * 
     * @return the updated {@code ProductFeed}
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public ProductFeed saveFeed(ProductFeed feed, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Removes a product feed from Target Recommendations
     * 
     * @param id - the product feed id
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * 
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public void deleteFeed(int id, Configuration ttConfig) throws TargetRecommendationsException;
    
    /**
     * Removes a list of products from the Target Recommendations index
     * 
     * @param productList - the list of product ids to be removed from the Recommendations index
     * @param ttConfig - the {@link Configuration} that identifies the Target cloud configuration
     * @throws TargetRecommendationsException {@link TargetRecommendationsException}
     */
    public void removeProducts(List<String> productList, Configuration ttConfig) throws TargetRecommendationsException;

    /**
     * Checks the connection to Target Recommendations using the given credentials
     *
     * @param clientCode clientCode
     * @param email email
     * @param password password
     * @throws TargetRecommendationsException {@link TargetRecommendationsException} on any connection error
     */
    public void checkConnection(String clientCode, String email, String password) throws TargetRecommendationsException;
}
