/*
*
*	File: Type2Consumer.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.cff;

import com.adobe.fontengine.font.InvalidFontException;

/** Receives events as a Type2 charstring is parsed.
 * 
 * There is one call per Type2 operator. 
 * 
 * <p>We have four virtual operators:
 * <ul>
 * <li> <code>integer</code> and <code>fraction</code>
 * are called when an integer or fraction operand
 * appearing in the Type2 charstring is pushed on the stack
 * <li> <code>implicit_vstemhm</code> is called when there is an
 * implicit vstemhm operator
 * before a hintmask or cntrmask operator
 * <li> <code>width</code> is called if the charstring contains an 
 * advance width; this call can happen in a seemingly random relationship
 * to the other operators
 * </ul>
 * 
 * <p>Each call receives the current stack (which should not be
 * modified, the interpreter will take care of popping the appropriate 
 * number of elements afterwards, and to push the result of operations
 * if needed), and the stack depth. A few calls have additional arguments,
 *  as noted below.
 */
public interface Type2Consumer {
  public void hstem (double stack[], int stackDepth);
  public void vstem (double stack[], int stackDepth);
  public void vmoveto (double stack[], int stackDepth);
  public void rlineto (double stack[], int stackDepth);
  public void hlineto (double stack[], int stackDepth);
  public void vlineto (double stack[], int stackDepth);
  public void rrcurveto (double stack[], int stackDepth);
  
  /** @param subr the index of the local subroutine which is called. */
  public void callsubr (double stack[], int stackDepth, int subr);
  public void return_op (double stack[], int stackDepth);
  public void and (double stack[], int stackDepth);
  public void or (double stack[], int stackDepth);
  public void not (double stack[], int stackDepth);
  public void abs (double stack[], int stackDepth);
  public void add (double stack[], int stackDepth);
  public void sub (double stack[], int stackDepth);
  public void div (double stack[], int stackDepth);
  public void neg (double stack[], int stackDepth);
  public void eq (double stack[], int stackDepth);
  public void drop (double stack[], int stackDepth);
  public void put (double stack[], int stackDepth);
  public void get (double stack[], int stackDepth);
  public void ifelse (double stack[], int stackDepth);
  public void random (double stack[], int stackDepth);
  public void mul (double stack[], int stackDepth);
  public void sqrt (double stack[], int stackDepth);
  public void dup (double stack[], int stackDepth);
  public void exch (double stack[], int stackDepth);
  public void index (double stack[], int stackDepth);
  public void roll (double stack[], int stackDepth);
  public void hflex (double stack[], int stackDepth);
  public void flex (double stack[], int stackDepth);
  public void hflex1 (double stack[], int stackDepth);
  public void flex1 (double stack[], int stackDepth);
  public void endchar (double stack[], int stackDepth);
  public void hstemhm (double stack[], int stackDepth);

  /** @param data holds the charstring (or subroutine) being interpreted
   * @param offset the index of the first byte representing the hintmask
   * @param length the number of bytes in the hintmask
   */
  public void hintmask (double stack[], int stackDepth, CFFByteArray data, int offset, int length)
  throws InvalidFontException;
  
  /** @param data holds the charstring (or subroutine) being interpreted
   * @param offset the index of the first byte representing the cntrmask
   * @param length the number of bytes in the cntrmask
   */
  public void cntrmask (double stack[], int stackDepth, CFFByteArray data, int offset, int length)
  throws InvalidFontException;
  
  public void rmoveto (double stack[], int stackDepth);
  public void hmoveto (double stack[], int stackDepth);
  
  /**
   * An absolute moveto (that is, a moveto that is not relative to the current point).
   * stack[0] contains the new x. stack[1] contains the new y.
   */
  public void moveto(double stack[], int stackDepth);
  
  public void vstemhm (double stack[], int stackDepth);
  
  /** Called when there is vstemhm operator implied by a hintmask or cntrmask operator.*/
  public void implicit_vstemhm (double stack[], int stackDepth);

  public void rcurveline (double stack[], int stackDepth);
  public void rlinecurve (double stack[], int stackDepth);
  public void vvcurveto (double stack[], int stackDepth);
  public void hhcurveto (double stack[], int stackDepth);

  /** 
   * @param subr the index of the global subroutine which is called. */
  public void callgsubr (double stack[], int stackDepth, int subr);

  public void vhcurveto (double stack[], int stackDepth);
  public void hvcurveto (double stack[], int stackDepth);

  /** 
   * @param i the integer which is pushed on the stack by an integer operator. */
  public void integer (double stack[], int stackDepth, int i);

  /** 
   * @param a the real number which is pushed on the stack by a 5 byte number operator.
   */
  public void real (double stack[], int stackDepth, double a);
  
  /** @param w the width recorded in the charstring, to be added to the nominal
   * width of the font; if this method is not called, the width of this outline
   * is the default width of the font.
   * @return true iff parsing should continue */
  public boolean width (double w);
  
  /**
   * @param stack the stack containing the following params:
   * Optionally, the advance width. If the stackDepth is 5, the first element
   * on the stack is the width.
   * adx: the horizontal position of the accent character relative to the
   * base character's origin.
   * ady: the vertical position of the accent character relative to the base
   * character's origin.
   * bchar: the character code of the base character in Standard Encoding.
   * achar: the character code of the accent character in Standard Encoding.
   * @return true if the parser should parse the base and accent characters. false if
   * parsing should not recurse into those characters.
   * 
   */
  public boolean seac(double stack[], int stackDepth);
  
  public void globalColorMe(double stack[], int stackDepth);
}
