/*
 * File: GlyphNamesResourceBuilder.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.adobe.fontengine.SparseArray;
import com.adobe.fontengine.font.opentype.GlyphNames.GlyphNameEntry;
import com.adobe.fontengine.font.opentype.GlyphNames.GlyphNameMultiEntry;

final public class GlyphNamesResourceBuilder {

  protected static final List nameTableDuplicates = new ArrayList ();
  protected static final List charTableDuplicates = new ArrayList ();

  //----------------------------------------------------------------------------

  static int parseNameListFile (String fileName, Map nameTable, List charTable)
  throws IOException  {   
    int linesRead;
    LineNumberReader f = new LineNumberReader (new FileReader (fileName));
    GlyphNameEntry entry = null;

    try {
      String line = f.readLine ();

      while (line != null)  {
        if ((entry = parseLine (line)) != null) {
          if (nameTable.containsKey (entry.getName())) {
            nameTableDuplicates.add (entry); }
          else {
            nameTable.put (entry.getName (), entry); }

          List usvList = (List) charTable.get (entry.getUSV ().intValue ());
          if (usvList == null) {
            usvList = new ArrayList ();
            usvList.add(entry);
            charTable.add (entry.getUSV ().intValue (), usvList); }
          else {
            usvList.add (entry); }}
        line = f.readLine (); }   
      linesRead = f.getLineNumber (); }

    finally {
      f.close (); }
    
    return linesRead;
  }

  static GlyphNameEntry parseLine (String line) {
    // lines beginning with an octothorpe are comment lines - ignore them
    if (line.charAt (0) == '#')  {
      return null; }

    // the name and the usv are semi-colon separated
    int semi = line.indexOf (';');

    String name = line.substring (0, semi).trim ();
    String usv = line.substring (semi + 1).trim ();

    // if there are two or more space separated usv's then we ignore those too
    if (usv.indexOf (' ') != -1) {
      return null; }

    return new GlyphNameEntry (Integer.parseInt (usv, 16), name);
  }

  //----------------------------------------------------------------------------

  static int parseCoolTypeNameListFile (String fileName, Map nameTable, List charTable)
  throws IOException {   
    int linesRead;
    LineNumberReader f = new LineNumberReader (new FileReader (fileName));
    GlyphNameEntry entry = null;

    try {
      String line = f.readLine ();

      while (line != null) {
        if ((entry = parseCoolTypeLine (line)) != null) {
		  if ((entry.getName().indexOf('#') != -1) || 
			  (entry.getName().indexOf('_') != -1)) {
            nameTableDuplicates.add (entry); }
          else {
            nameTable.put (entry.getName(), entry); }

          List usvList = (List) charTable.get (entry.getUSV ().intValue ());
          if (usvList == null) {
            usvList = new ArrayList ();
            usvList.add (entry);
            charTable.add (entry.getUSV ().intValue (), usvList); }
          else {
            if (   (entry.getName().indexOf('_') != -1)
                || (entry.getName().indexOf('#') != -1)) {
              // alternate or non-preferred name - add to the end of the list
              usvList.add (entry); }
            else {
              // preferred name - add to the head of the list
              usvList.add (0, entry); }}}
        line = f.readLine (); }

	  for (int i=0; i<nameTableDuplicates.size(); i++) {
		entry = (GlyphNameEntry)nameTableDuplicates.get(i);
		int nameend = entry.getName().indexOf('#');
		if (nameend == -1) {
		  nameend = entry.getName().indexOf('_');}

		String realname = entry.getName().substring(0, nameend);
		if (!nameTable.containsKey (realname)) {
		  entry = new GlyphNameEntry(entry.getUSV(), realname);
		  nameTable.put (entry.getName(), entry); }}

      linesRead = f.getLineNumber(); }
    finally {
      f.close (); }
    
    return linesRead;
  }
  static int parseCoolTypeMultiNameListFile (String fileName, Map nameTable)
  throws IOException {   
    int linesRead;
    LineNumberReader f = new LineNumberReader (new FileReader (fileName));
    GlyphNameMultiEntry entry = null;

    try {
      String line = f.readLine ();

      while (line != null) {
        if ((entry = parseCoolTypeMultiLine (line)) != null) {
          if (nameTable.containsKey (entry.getName ())) {
            nameTableDuplicates.add (entry); }
          else {
            nameTable.put (entry.getName(), entry); }}
        line = f.readLine (); }   
      linesRead = f.getLineNumber(); }

    finally {
      f.close (); }
    
    return linesRead;
  }
  static GlyphNameEntry parseCoolTypeLine (String line) {
    // lines beginning with an octothorpe are comment lines - ignore them
    if (line.charAt (0) == '*' || line.charAt(0) == '/')  {
      return null;  }

    // the name is surrounded by quotes
    int firstQuote = line.indexOf('\"');
    int secondQuote = line.indexOf('\"', firstQuote + 1);
    String name = line.substring (firstQuote + 1, secondQuote).trim ();

    int beginHex = line.indexOf ("0x", secondQuote);
    int parenthesis = line.indexOf ('}', beginHex);
    String usv = line.substring (beginHex + 2, parenthesis).trim ();

    return new GlyphNameEntry (Integer.parseInt (usv, 16), name);
  }

  static GlyphNameMultiEntry parseCoolTypeMultiLine (String line) {
    // lines beginning with an octothorpe are comment lines - ignore them
    if (line.charAt (0) == '*' || line.charAt(0) == '/')  {
      return null;  }

    // the name is surrounded by quotes
    int firstQuote = line.indexOf('\"');
    int secondQuote = line.indexOf('\"', firstQuote + 1);
	if (line.charAt(secondQuote-1) == '#')
		secondQuote--;
    String name = line.substring (firstQuote + 1, secondQuote).trim ();

    int beginMulti = line.indexOf ('{', secondQuote);
    int endMulti = line.indexOf ('}', beginMulti);
    String multi = line.substring (beginMulti+1, endMulti).trim ();
	String[] hex_usvs = multi.split(",");

	List usvs = new ArrayList();
	for (int i = 0; i < hex_usvs.length-1; i++)
	{
    	int beginHex = hex_usvs[i].indexOf ("0x");
		usvs.add(new Integer(Integer.parseInt(hex_usvs[i].substring(beginHex+2), 16)));
	}
	
    return new GlyphNameMultiEntry (usvs, name);
  }


  public static void main (String args [])
  throws Exception {
    String projectPath = args [0] + File.separator;
    //boolean dumpDuplicates = args.length >= 3 && args[2].equals("dump") ? true : false;

    System.out.println ("--- generating resources for glyph names");

    ObjectOutputStream os = null;
//  BufferedWriter log = null;

    Map nameTable = new HashMap();
    Map nameMultiTable = new HashMap();
    Map dingBatNameTable = new HashMap();
    SparseArray charTable = new SparseArray();
    SparseArray dingBatCharTable = new SparseArray();

    //SparseArray cooltypeCharTable = new SparseArray();

    try {
      int linesRead = 0;
      os = new ObjectOutputStream(
          new FileOutputStream (args[1] + File.separator + "com/adobe/fontengine/font/opentype/agnc"));

      //linesRead = parseNameListFile (projectPath + "outside/agnc/glyphlist.txt", nameTable, charTable);

      linesRead = parseCoolTypeNameListFile(projectPath + "outside/cooltype/UniT1Map.h", nameTable, charTable);
      System.out.println("\t\tglyphlist: " + linesRead + " lines read");

      linesRead = parseCoolTypeMultiNameListFile(projectPath + "outside/cooltype/UniT1MultiMap.h", nameMultiTable);
      System.out.println("\t\tglyphlistMulti: " + linesRead + " lines read");


//    if (dumpDuplicates)
//    {
//    File logFile = new File(args[1] + "logs/output/com/adobe/fontengine/font/agnc_cooltype_comparison.txt");
//    System.out.println("\t\t\tComparison to CoolType dumped to log file - " + logFile.toString());
//    logFile.getParentFile().mkdirs();
//    log = new BufferedWriter (new OutputStreamWriter(
//    new FileOutputStream(logFile), "UTF-8"));
//    log.write("\t\t\tchar table duplicates = ");
//    log.newLine();
//    for (int i = 0; i < charTable.size(); i++)
//    {
//    List usvList = (List) charTable.get(i);
//    List cooltypeList = (List) cooltypeCharTable.get(i);

//    if (((usvList != cooltypeList) && (usvList == null || cooltypeList == null))
//    || ((usvList != null && cooltypeList != null) && !usvList.equals(cooltypeList)))
//    {
//    if (usvList != null && usvList.size() > 1)
//    {
//    log.write("\t\t\t\t+ " + usvList);
//    log.newLine();
//    }

//    if (cooltypeList != null && cooltypeList.size() > 1)
//    {
//    log.write("\t\t\t\t- " + cooltypeList);
//    log.newLine();
//    }
//    }
//    }
//    }

      os.writeObject(nameTable);
      os.writeObject(nameMultiTable);
      os.writeObject(charTable);

      linesRead = parseNameListFile (projectPath + "outside/agnc/zapfdingbats.txt", dingBatNameTable, dingBatCharTable);
      System.out.println("\t\tdingbats: " + linesRead + " lines read");
//    if (dumpDuplicates)
//    {
//    log.write("\t\t\tdingbat char table duplicates = ");
//    log.newLine();
//    for (int i = 0; i < dingBatCharTable.size(); i++)
//    {
//    List usvList = (List) dingBatCharTable.get(i);
//    if (usvList != null && usvList.size() > 1)
//    {
//    log.write("\t\t\t\t" + usvList);
//    log.newLine();
//    }
//    }
//    } 

      os.writeObject (dingBatNameTable);
      os.writeObject (dingBatCharTable); }

    finally {
//    if (log != null)
//    {
//    log.close();
//    }
      if (os != null) {
        os.close (); }}

    System.out.println ("\t\tdone");
  }
}
