/*
*
*	File: Post.java
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2006 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.SubsetSimpleTrueType;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'post' table.
 *
 * <h4>Version handling</h4>
 * 
 * <p>'post' tables have a major/minor version number.
 * This implementation:
 * <ul> 
 * <li>fully supports version 1.0, 2.0, 2.5 and 3.0 tables,</li>
 * <li>partially supports version 4.0 tables,</li>
 * <li>interprets 1.x tables as 1.0 tables,</li>
 * <li>interprets 2.x tables with x < 5 as 2.0 tables</li>
 * <li>interprets 2.x tables with x > 5 as 2.5 tables</li>
 * <li>interprets 3.x tables as 3.0 tables</li>
 * <li>interprets 4.x tables as 4.0 tables</li>
 * <li>rejects other versions with an <code>UnsupportedFontException</code> 
 * at construction time.</li>
 * </ul>
 *
 * <p>Version 2.5 is treated as if it were a major version number (which it 
 * should have been, since a table with version 2.5 cannot be interpreted
 * correctly by a 2.0 client).</p>
 *
 * <p>When some data is not present in the actual version of the table,
 * the accessors return identified values.</p>
 *
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

final public class Post extends Table {
    
  protected Post (FontByteArray buffer) 
  throws UnsupportedFontException, InvalidFontException, java.io.IOException {
    
    super (buffer);
    
    if (buffer.getSize() < 32) {
      throw new InvalidFontException ("'post' table must be at least 32 bytes"); }
    
    int majorVersion = getTableMajorVersion ();
    if (majorVersion < 1 || 4 < majorVersion) {
      throw new UnsupportedFontException 
                   ("'post' tables with major version " 
                        + majorVersion + " are not supported"); }
  }
  
  /** Get the major version of this table. */
  public int getTableMajorVersion () throws InvalidFontException {
    return this.data.getuint16 (0);
  }
 
  /** Get the minor version of this table. */
  public int getTableMinorVersion () throws InvalidFontException {
    return this.data.getuint16 (2);
  }
  
  public int getItalicAngle() throws InvalidFontException {
	  return this.data.getint32(4);
  }

  /** Get the underline position. */
  public int getUnderlinePosition () throws InvalidFontException {
    return this.data.getint16 (8); 
  }
  
  /** Get the underline thickness. */
  public int getUnderlineThickness () throws InvalidFontException {
    return this.data.getint16 (10); 
  }
  
  public boolean isFixedPitch() throws InvalidFontException {
	  return (this.data.getuint32(12) != 0);
  }
  
  /** Get the number of glyphs.
   * @return -1 if the table does not have this data
   */
  public int getNumberOfGlyphs () throws InvalidFontException {
    if (getTableMajorVersion () != 2) {
      return -1; }
    return this.data.getuint16 (32);
  }
  
  /** Get the name of a glyph.
   * @return null if this information is not present, the name of 
   * the glyph otherwise
   */
  public String getGlyphName (int gid){
   
	  try {	 
	    switch (getTableMajorVersion ()) {
	
	      case 1: {
	    	  if (gid >= macNames.length)
	    		  return null;
	        return macNames [gid]; }
	    
	      case 2: {
	        if (getTableMinorVersion () < 5) {
	          int numberOfGlyphs = this.data.getuint16 (32);
	          if (gid >= numberOfGlyphs) {
	            return null; }
	          int position = this.data.getuint16 (34 + gid * 2);
	          if (position < 258) {
	            return macNames [position]; }
	        
	          int offset = 34 + numberOfGlyphs * 2;
	          // skip the names before
	          for (int i = 258; i < position; i++) {
	            int len = this.data.getuint8 (offset);
	            offset += 1 + len; }
	          int len = this.data.getuint8 (offset);
	          char[] s = new char [len];
	          for (int i = 0; i < len; i++) {
	            s [i] = (char) this.data.getuint8 (++offset); }
	          return new String (s); }
	
	        else {
	          int inc = this.data.getint8 (34 + gid * 2);
	          return macNames [gid + inc]; }}
	 
	      case 3: 
	      default: {
	        return null; }}
	  } catch (InvalidFontException e) {
		  // Seeing enough truncated post tables to warrant skipping this
		  // exception and using other methods to look for the glyph name.
		  return null;
	  }
  }
  
  /** Get the index of a glyph given its name.
   * @return  0 if there is no glyph with the name
   */
  public int glyphName2gid (String name) throws InvalidFontException {

    switch (getTableMajorVersion ()) {

      case 1: {
        for (int gid = 0; gid < macNames.length; gid++) {
          if (macNames [gid].equals (name)) {
            return gid; }}
        return 0; }

      case 2: {
        int numberOfGlyphs = this.data.getuint16 (32);
        for (int gid = 0; gid < numberOfGlyphs; gid++) {
          if (name.equals (getGlyphName (gid))) {
            return gid; }}
        return 0; }

      case 3:
      default: {
        return 0; }}
  }
  
  // These names are defined in Apple's TrueType specification.
  private final static String [] macNames = {
      ".notdef", // 0
      ".null", // 1
      "nonmarkingreturn", // 2
      "space", // 3
      "exclam", // 4
      "quotedbl", // 5
      "numbersign", // 6
      "dollar", // 7
      "percent", // 8
      "ampersand", // 9
      "quotesingle", // 10
      "parenleft", // 11
      "parenright", // 12
      "asterisk", // 13
      "plus", // 14
      "comma", // 15
      "hyphen", // 16
      "period", // 17
      "slash", // 18
      "zero", // 19
      "one", // 20
      "two", // 21
      "three", // 22
      "four", // 23
      "five", // 24
      "six", // 25
      "seven", // 26
      "eight", // 27
      "nine", // 28
      "colon", // 29
      "semicolon", // 30
      "less", // 31
      "equal", // 32
      "greater", // 33
      "question", // 34
      "at", // 35
      "A", // 36
      "B", // 37
      "C", // 38
      "D", // 39
      "E", // 40
      "F", // 41
      "G", // 42
      "H", // 43
      "I", // 44
      "J", // 45
      "K", // 46
      "L", // 47
      "M", // 48
      "N", // 49
      "O", // 50
      "P", // 51
      "Q", // 52
      "R", // 53
      "S", // 54
      "T", // 55
      "U", // 56
      "V", // 57
      "W", // 58
      "X", // 59
      "Y", // 60
      "Z", // 61
      "bracketleft", // 62
      "backslash", // 63
      "bracketright", // 64
      "asciicircum", // 65
      "underscore", // 66
      "grave", // 67
      "a", // 68
      "b", // 69
      "c", // 70
      "d", // 71
      "e", // 72
      "f", // 73
      "g", // 74
      "h", // 75
      "i", // 76
      "j", // 77
      "k", // 78
      "l", // 79
      "m", // 80
      "n", // 81
      "o", // 82
      "p", // 83
      "q", // 84
      "r", // 85
      "s", // 86
      "t", // 87
      "u", // 88
      "v", // 89
      "w", // 90
      "x", // 91
      "y", // 92
      "z", // 93
      "braceleft", // 94
      "bar", // 95
      "braceright", // 96
      "asciitilde", // 97
      "Adieresis", // 98
      "Aring", // 99
      "Ccedilla", // 100
      "Eacute", // 101
      "Ntilde", // 102
      "Odieresis", // 103
      "Udieresis", // 104
      "aacute", // 105
      "agrave", // 106
      "acircumflex", // 107
      "adieresis", // 108
      "atilde", // 109
      "aring", // 110
      "ccedilla", // 111
      "eacute", // 112
      "egrave", // 113
      "ecircumflex", // 114
      "edieresis", // 115
      "iacute", // 116
      "igrave", // 117
      "icircumflex", // 118
      "idieresis", // 119
      "ntilde", // 120
      "oacute", // 121
      "ograve", // 122
      "ocircumflex", // 123
      "odieresis", // 124
      "otilde", // 125
      "uacute", // 126
      "ugrave", // 127
      "ucircumflex", // 128
      "udieresis", // 129
      "dagger", // 130
      "degree", // 131
      "cent", // 132
      "sterling", // 133
      "section", // 134
      "bullet", // 135
      "paragraph", // 136
      "germandbls", // 137
      "registered", // 138
      "copyright", // 139
      "trademark", // 140
      "acute", // 141
      "dieresis", // 142
      "notequal", // 143
      "AE", // 144
      "Oslash", // 145
      "infinity", // 146
      "plusminus", // 147
      "lessequal", // 148
      "greaterequal", // 149
      "yen", // 150
      "mu", // 151
      "partialdiff", // 152
      "summation", // 153
      "product", // 154
      "pi", // 155
      "integral", // 156
      "ordfeminine", // 157
      "ordmasculine", // 158
      "Omega", // 159
      "ae", // 160
      "oslash", // 161
      "questiondown", // 162
      "exclamdown", // 163
      "logicalnot", // 164
      "radical", // 165
      "florin", // 166
      "approxequal", // 167
      "Delta", // 168
      "guillemotleft", // 169
      "guillemotright", // 170
      "ellipsis", // 171
      "nonbreakingspace", // 172
      "Agrave", // 173
      "Atilde", // 174
      "Otilde", // 175
      "OE", // 176
      "oe", // 177
      "endash", // 178
      "emdash", // 179
      "quotedblleft", // 180
      "quotedblright", // 181
      "quoteleft", // 182
      "quoteright", // 183
      "divide", // 184
      "lozenge", // 185
      "ydieresis", // 186
      "Ydieresis", // 187
      "fraction", // 188
      "currency", // 189
      "guilsinglleft", // 190
      "guilsinglright", // 191
      "fi", // 192
      "fl", // 193
      "daggerdbl", // 194
      "periodcentered", // 195
      "quotesinglbase", // 196
      "quotedblbase", // 197
      "perthousand", // 198
      "Acircumflex", // 199
      "Ecircumflex", // 200
      "Aacute", // 201
      "Edieresis", // 202
      "Egrave", // 203
      "Iacute", // 204
      "Icircumflex", // 205
      "Idieresis", // 206
      "Igrave", // 207
      "Oacute", // 208
      "Ocircumflex", // 209
      "apple", // 210
      "Ograve", // 211
      "Uacute", // 212
      "Ucircumflex", // 213
      "Ugrave", // 214
      "dotlessi", // 215
      "circumflex", // 216
      "tilde", // 217
      "macron", // 218
      "breve", // 219
      "dotaccent", // 220
      "ring", // 221
      "cedilla", // 222
      "hungarumlaut", // 223
      "ogonek", // 224
      "caron", // 225
      "Lslash", // 226
      "lslash", // 227
      "Scaron", // 228
      "scaron", // 229
      "Zcaron", // 230
      "zcaron", // 231
      "brokenbar", // 232
      "Eth", // 233
      "eth", // 234
      "Yacute", // 235
      "yacute", // 236
      "Thorn", // 237
      "thorn", // 238
      "minus", // 239
      "multiply", // 240
      "onesuperior", // 241
      "twosuperior", // 242
      "threesuperior", // 243
      "onehalf", // 244
      "onequarter", // 245
      "threequarters", // 246
      "franc", // 247
      "Gbreve", // 248
      "gbreve", // 249
      "Idotaccent", // 250
      "Scedilla", // 251
      "scedilla", // 252
      "Cacute", // 253
      "cacute", // 254
      "Ccaron", // 255
      "ccaron", // 256
      "dcroat", // 257
     };
  
    public void subsetAndStreamForCFF(Map tables) throws InvalidFontException
    {
    	OTByteArrayBuilder postData = OTByteArray.getOTByteArrayBuilderInstance(32);
    	postData.setuint32(0, 0x00030000);
    	postData.setuint32(4, data.getint32(4));					// Copy italicAngle
		postData.setuint16(8, data.getint16(8));					// Copy underlinePosition
		postData.setuint16(10, data.getint16(10));					// Copy underlineThickness
		postData.setuint32(12, data.getint32(12));					// Copy isFixedPitch
		postData.setuint32(16, 0);					
		postData.setuint32(20, 0);					
		postData.setuint32(24, 0);					
		postData.setuint32(28, 0);	
		tables.put(new Integer(Tag.table_post), postData);				// Write out finished table
    }

	public void subsetAndStream(Subset subset, SubsetSimpleTrueType ttSubset, Map tables)
		throws UnsupportedFontException, InvalidFontException
	{
		if (ttSubset == null || getTableMajorVersion() > 2)
			return;
		int numGlyphs = subset.getNumGlyphs();
		OTByteArrayBuilder postData = OTByteArray.getOTByteArrayBuilderInstance(34 + numGlyphs * 32);
		postData.setuint32(0, 0x20000);							// Format number, we use 2.0
		postData.setuint32(4, data.getint32(4));					// Copy italicAngle
		postData.setuint16(8, data.getint16(8));					// Copy underlinePosition
		postData.setuint16(10, data.getint16(10));					// Copy underlineThickness
		postData.setuint32(12, data.getint32(12));					// Copy isFixedPitch
		postData.setuint32(16, data.getint32(16));					// Copy minMemType42
		postData.setuint32(20, data.getint32(20));					// Copy maxMemType42
		postData.setuint32(24, data.getint32(24));					// Copy minMemType1
		postData.setuint32(28, data.getint32(28));					// Copy maxMemType1
		postData.setuint16(32, numGlyphs);						// New array length
		int pascalOffset = 0;								// Special strings empty
		int pascalIndex = 258;								// Index starts at 258
		int stringBase = 34 + numGlyphs * 2;						// Beginning of special string section
		for (int subsetGid = 0; subsetGid < subset.getNumGlyphs(); subsetGid++) {
			int gid = subset.getFullGid(subsetGid);					// One entry per subset gid
			int nameIndex = getMacName(gid);					// Get standard index
			if (nameIndex < 0) {
				String specialName = getGlyphName(gid);				// Or special name if not standard
				if (specialName != null) {
					int len = specialName.length();				// Pascal length comes first
					postData.setuint8(stringBase + pascalOffset++, len);	// Cannot be longer than 255
					for (int i = 0; i < len; i++)
						postData.setuint8(stringBase + pascalOffset++, specialName.charAt(i));
					nameIndex = pascalIndex++;				// Assign new ID
				}
			}
			postData.setuint16(34 + subsetGid * 2, nameIndex);			// Write name index to array
		}
		tables.put(new Integer(Tag.table_post), postData);				// Write out finished table
	}

	/**
	 * Local hack to get macName offset when subsetting
	 */
	private int getMacName(int gid)
		throws InvalidFontException
	{
		switch (getTableMajorVersion()) {
		case 1:
			return gid;
		case 2:
			if (getTableMinorVersion() == 0) {
				int numberOfGlyphs = data.getuint16(32);
				if (gid >= numberOfGlyphs)
					return -1;
				int position = data.getuint16(34 + gid * 2);
				if (position < 258)
					return position;
				return -1;
			} else {
				int inc = data.getint8(34 + gid * 2);
				return gid + inc;
			}
		default:
			return -1;
		}
	}
	
	 public void stream(Map tables) {
		  OTByteArrayBuilder newData = this.getDataAsByteArray();
		  tables.put (new Integer (Tag.table_post), newData);
	  }
}
