/*
 *
 *	File: FontResolutionPriority.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.fontmanagement;

import java.io.Serializable;


/**
 * This class provides an enumeration of the allowable values for use in setting the resolution
 * priority in the variety of font sets that are stored in CAFE.  Not every font set uses every
 * font resolution priority.
 * 
 * Examples of the use of these priority levels are available in:
 * {@link com.adobe.fontengine.inlineformatting.css20.CSS20FontSet#setResolutionPriority CSS20FontSet.setResolutionPriority}.
 * {@link com.adobe.fontengine.fontmanagement.postscript.PSNameResolver#setResolutionPriority PSNameResolver.setResolutionPriority}.
 * 
 * <h4>Concurrency</h4>
 *  
 * Instances of this class are immutable after construction and contain no mutable static
 * data. Therefore, they are threadsafe.
 */
public final class FontResolutionPriority implements Serializable
{
	/**
	 * In the case of conflicts choose the font using an "intelligent" font resolution process.
	 * If two fonts appear to be the same under "intelligent" selection choose the one that was
	 * added last.
	 */
	public static final FontResolutionPriority INTELLIGENT_LAST = 
		new FontResolutionPriority(new String [] {"intelligent last", "intelligent"}, 1);
	/**
	 * This is an alias for {@link #INTELLIGENT_LAST}.
	 */
	public static final FontResolutionPriority INTELLIGENT = INTELLIGENT_LAST;
	/**
	 * In the case of conflicts choose the font using an "intelligent" font resolution process.
	 * If two fonts appear to be the same under "intelligent" selection choose the one that was
	 * added last.
	 */
	public static final FontResolutionPriority INTELLIGENT_FIRST = 
		new FontResolutionPriority(new String[] {"intelligent first"}, 4);
	/**
	 * In the case of conflicts choose the font which was added to the font set first.
	 */
	public static final FontResolutionPriority FIRST = 
		new FontResolutionPriority(new String[] {"first"}, 2);
	/**
	 * In the case of conflicts choose the font which was added to the font set last.
	 */
	public static final FontResolutionPriority LAST = 
		new FontResolutionPriority(new String[] {"last"}, 3);

	private static final FontResolutionPriority[] priorityValues = {
		INTELLIGENT_LAST, INTELLIGENT_FIRST, FIRST, LAST
	};

	/* Serialization signature is explicitly set and should be 
	 * incremented on each release to prevent compatibility.
	 */
	static final long serialVersionUID = 1;

	// This is used to ensure that these remain singleton objects on deserialization.
	private final int id;
	private final String[] names;

	/**
	 * @param names the array of names to use for this priority; 
	 * place the preferred name first as that will be used for output
	 * @param id
	 */
	private FontResolutionPriority (String[] names, int id)
	{
		this.names = names;
		this.id = id;
	}

	/**
	 * Resolve the object during de-serialization.  We want to return only one of our singletons.
	 * @return the correct singleton
	 */
	private Object readResolve()
	{
		for (int i = 0; i < priorityValues.length; i++)
		{
			if (this.id == priorityValues[i].id)
			{
				return priorityValues[i];
			}
		}

		return null;
	}

	/**
	 * Parse the text and return the FontResolutionPriority object which is represented if any.
	 * 
	 * @param text textual representation of the resolution priority
	 * @return FontResolutionPriority represented by the text value
	 */
	public static FontResolutionPriority parse(String text)
	{
		for (int priority = 0; priority < priorityValues.length; priority++)
		{
			for (int name = 0; name < priorityValues[priority].names.length; name++)
			{
				if (priorityValues[priority].names[name].equalsIgnoreCase(text))
				{
					return priorityValues[priority];
				}
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		return names[0];
	}

}