/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.taskmanagement;

import com.adobe.granite.workflow.exec.InboxItem;
import com.adobe.granite.workflow.exec.Status;

import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.osgi.annotation.versioning.ProviderType;

/**
 * Task class describes a task instance.
 */
@ProviderType
public interface Task extends InboxItem {

    /**
     * Default task type.
     */
    String DEFAULT_TASK_TYPE = "default";

    /**
     * Gets the ID of the task.
     * @return String containing the id of the task.
     */
    String getId();

    /**
     * Gets the UUID for the task
     * @return String containing the UUID of the Task
     */
    String getUUId();

    /**
     * Returns the ID of the parent task.
     * @return the ID of the parent task, or null if this is not a sub-task.
     */
    String getParentId();

    /**
     * Gets the name of the task.
     * @return String containing the name of the task.
     */
    String getName();

    /**
     * Sets the name of the task.
     * @param name String containing the new name of the task.
     */
    void setName(String name);

    /**
     * Gets the description of the task.
     * @return String containing the description of the task.
     */
    String getDescription();

    /**
     * Sets the description of the task.
     * @param description String containing the new description of the task.
     */
    void setDescription(String description);

    /**
     * Gets the current status of the task.
     * @return Status instance representing the current task status.
     */
    Status getStatus();

    /**
     * Gets the id of the owner of the task.
     * @return String containing the id of the owner of the task.
     */
    String getCurrentAssignee();

    /**
     * Sets the id of the owner of the task.
     * @param ownerId String containing the id of the owner of the task.
     */
    void setCurrentAssignee(String ownerId);

    /**
     * Gets the type of this task.
     * @return String containing the name of the type of this task.
     */
    String getTaskTypeName();

    /**
     * Gets the time this task was created.
     * @return Date containing the time this task was created.
     */
    Date getTimeStarted();

    /**
     * Gets the ID for the user who created the task
     * @return String containing the create user's ID
     */
    String getCreatedBy();

    /**
     * Gets the time this task was last updated.
     * @return Date containing the time this task was last updated.
     */
    Date getLastModified();

    /**
     * Gets the ID for the user who last modified the task
     * @return String containing the user's ID
     */
    String getLastModifiedBy();

    /**
     * Gets the time this task was last completed.
     * @return Date containing the time this task was completed.
     */
    Date getTimeEnded();

    /**
     * Gets the ID for the user who completed the task
     * @return String containing the complete user's ID
     */
    String getCompletedBy();

    /**
     * Sets the property.
     * @param name Name of the property to set.
     * @param value value of named property.
     */
    void setProperty(String name, Object value);

    /**
     * Gets a property with the given name.
     * @param aPropertyName Name of the property to retrieve.
     * @return value of the specified property.
     */
    Object getProperty(String aPropertyName);

    /**
     * Gets an iterator of the property names for this task.
     * @return an task property name iterator.
     */
    Iterator<String> getPropertyNames();

    /**
     * Sets the instructions for this task.
     * @param aInstructions String containing the new instructions for this task.
     */
    void setInstructions(String aInstructions);

    /**
     * Gets the instructions for this task.
     * @return String containing the instructions for this task.
     */
    String getInstructions();

    /**
     * returns the list of action objects for this task.
     * @return list of action objects
     */
    List<TaskAction> getActions();

    /**
     * sets the list of action objects for this task.
     * @param aListOfActions list of action objects
     */
    void setActions(List<TaskAction> aListOfActions);

    /**
     * gets the selected action used to complete the task.
     * @return action TaskAction represented the action used to compete the task
     */
    TaskAction getSelectedAction();

    /**
     * Returns the list of subtasks for this task
     * @return the list of subtasks
     */
    List<Task> getSubTasks();

    /**
     * Indicates if this task object is fully populated with sub task information.
     * For performance reasons some task objects may not be fully populated with sub task data.
     * @return true if the sub tasks are loaded in this task object
     */
    boolean isSubTaskInfoLoaded();

    /**
     * Returns whether or not there are sub-tasks under this task.
     * Task instances from the server (ie: it does not depend on sub-tasks being actually loaded)
     * @return true if this task has sub-tasks, false otherwise
     */
    boolean hasSubTasks();

    /**
     * Returns a content path this task is associated with.
     * @return the associated content path
     */
    String getContentPath();

    /**
     * Sets a content path that this task is associated with.
     * @param contentPath the new content path for this task
     */
    void setContentPath(String contentPath);

    /**
     * Sets the due time of this task.
     * @param dueTime the new due time of this task
     */
    void setDueTime(Date dueTime);

    /**
     * Sets the progress begin time for this task.
     * @param progressBeginTime the time for when progress should start on this task.
     */
    void setProgressBeginTime(Date progressBeginTime);

    /**
     * Sets the priority for this task.
     * @param priority the priority of this task.
     */
    void setPriority(Priority priority);

}
