/*
 * File: InputByteStreamImpl.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.io.stream;

import java.io.IOException;
import java.io.InputStream;

/**
 * Only for internal engineering use. This api can change without notice.
 */
public abstract class InputByteStreamImpl implements InputByteStream
{
	protected InputByteStreamImpl(){}

	// The "marked" position for use in mark() and reset().
	protected int mark = 0;

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Create a new <code>InputByteStream</code> that that provides another view on the underlying
	 * data.  This new <code>InputByteStream</code> is completely independent and can be read from
	 * and positioned without affecting the original or any other <i>slice</i>
	 * <code>InputByteStream</code> objects.  The initial position of the <i>slice</i> is at zero -
	 * the beginning of the <i>slice</i>.
	 * @param begin	Offset within the current <code>InputByteStream</code> to start the <i>slice</i>.
	 * @param length	The length of the <i>slice</i>.
	 * @return A <i>slice</i> of the original <code>InputByteStream</code>.
	 * @throws IOException
	 */
	public abstract InputByteStream slice(long begin, long length)
	throws IOException;

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Create a new <code>InputByteStream</code> that that provides another view on the underlying
	 * data.  This new <code>InputByteStream</code> is completely independent and can be read from
	 * and positioned without affecting the original or any other <i>slice</i>
	 * <code>InputByteStream</code> objects.  The initial position of the <i>slice</i> is at zero -
	 * the beginning of the <i>slice</i>.  The <i>slice</i> has the same bounds as the original
	 * <code>InputByteStream</code>.
	 *
	 * One common but probably incorrect usage of this method is to create a temporary "view"
	 * on the data that is used briefly within a method and then discarded.  This is pretty
	 * wasteful and creates a number of small objects.  In that case it would be better
	 * to find the current position using <code>getPosition()</code> and store it locally
	 * in the method and then restore the position with <code>setPosition()</code>
	 * before returning from the method.
	 * @return A <i>slice</i> of the original <code>InputByteStream</code>.
	 * @throws IOException
	 */
	public InputByteStream slice()
	throws IOException
	{
		return this.slice(0, this.length());
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#read()
	 */
	public abstract int read()
	throws IOException;

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#read(byte[], int, int)
	 */
	public abstract int read(byte[] bytes, int position, int length)
	throws IOException;

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#read(byte[])
	 */
	public int read(byte[] bytes)
	throws IOException
	{
		return this.read(bytes, 0, bytes.length);
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#unget()
	 */
	public int unget()
	throws IOException
	{
		this.seek(this.getPosition() - 1);
		int byteRead = this.read();
		this.seek(this.getPosition() - 1);
		return byteRead;
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Set the current position in the underlying data.  This position is pegged
	 * to be no less than zero and no greater than the length of the InputByteStream.
	 * @param position Where to set the current position.
	 * @return This object.
	 * @throws IOException
	 */
	public abstract InputByteStream seek(long position)
	throws IOException;

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#getPosition()
	 */
	public abstract long getPosition()
	throws IOException;

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#length()
	 */
	public abstract long length()
	throws IOException;

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#bytesAvailable()
	 */
	public long bytesAvailable()
	throws IOException
	{
		return this.length() - this.getPosition();
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#eof()
	 */
	public boolean eof()
	throws IOException
	{
		return this.getPosition() >= this.length();
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#close()
	 */
	public abstract void close() 
	throws IOException;

	/**
	 * Only for internal engineering use. This api can change without notice.
	 */
	// for debug only
	public String toString()
	{
		StringBuilder message = null;
		byte[] buf = new byte[100];
		InputByteStream slice=null;
		try {
			slice = this.slice(this.getPosition(), Math.min(this.length() - this.getPosition(), buf.length));
			slice.read(buf);
		} catch (IOException e) {
			// Do nothing here - we will return an empty string
		}finally{
			try {
				if(slice != null)
					slice.close();
			} catch (IOException e) {
				//
			}
		}
		try {
			message = new StringBuilder("InputByteStream [ position = ").append(this.getPosition()).
					append(", limit = ").append(this.length()).
					append(", available = ").append(this.bytesAvailable()).
					append(", mark = ").append(this.mark).
			append(" ] ").append(new String(buf, "US-ASCII"));
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
		return message.toString();
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.InputByteStream#toInputStream()
	 */
	public InputStream toInputStream() 
	throws IOException
	{
		InputByteStream ibs = this.slice();
		return new InputStreamImpl(ibs);
	}
}
