/*
 * File: PaddedInputByteStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2009 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/



package com.adobe.internal.io.stream;

import java.io.IOException;

/**
 * Only for internal engineering use. This api can change without notice.
 * 
 * This class provides access to a seekable input stream with support of padding without allocating any extra memory. This class's 
 * object wraps inside it an InputByteStream. It extends the <code>InputByteStreamImpl</code> abstract class.
 */

public class PaddedInputByteStream extends InputByteStreamImpl {

	private InputByteStream baseIBS;
	private int paddedChar = ' ';
	private long padLength = 0;
	
	private long padMark = 0;
	
	/**
	 * constructor with InputByteStream, character to be padded and padding length as parameters.
	 * @param ibs
	 * @param paddedChar
	 * @param padLength
	 */
	public PaddedInputByteStream(InputByteStream ibs, int paddedChar, long padLength) {
		this.baseIBS = ibs;
		this.paddedChar = paddedChar;
		this.padLength = padLength;
	}
	 
	/**
	 * Closes this InputByteStream and releases any system resources associated with the stream.
	 * After this method is called then any further calls to this instance are errors.
	 * @throws IOException
	 */
	public void close() throws IOException {
	 	baseIBS.close();
	}
	
	/**
	 * Get the current position.
	 * @return The current position.
	 * @throws IOException
	 */
	public long getPosition() throws IOException {
		if(baseIBS.eof())
			return baseIBS.length() + padMark;
		return baseIBS.getPosition();
	}

	/**
	 * The maximum number of bytes that can be read from this <code>InputByteStream</code>.
	 *
	 * @return Total number of bytes available in this <code>InputByteStream</code>.
	 * @throws IOException
	 */
	public long length() throws IOException {
		return baseIBS.length() + padLength;
	}

	/**
	 * Read the byte at the current position.  If the current position is at or beyond
	 * the end of the underlying data return a <code>-1</code>.  If not beyond the end
	 * of the underlying data the current position is incremented by 1.
	 * @return The byte at the current position or <code>-1</code> if at or beyond EOF.
	 * @throws IOException
	 */
	public int read() throws IOException {
		if(this.eof())
			return -1;
		if(baseIBS.eof()) {
			padMark++;
			return paddedChar;
		}
		return baseIBS.read();
	}

	
	
	/**
	 * Read an array of bytes starting at the current  position. The
	 * position is incremented by the length of the array that has been read.
	 * @param bytes	The destination array.
	 * @param position	The offset in the byte array to put the first byte read.
	 * @param length	 The number of bytes to read.
	 * @throws IOException
	 */
	public int read(byte[] bytes, int position, int length) throws IOException {
		if(this.eof())
			return InputByteStream.EOF;
		if(getPosition() + length <= baseIBS.length())
			return baseIBS.read(bytes, position, length);
		else{
			if(getPosition() + length > length())
				length = (int) (length() - getPosition());
			int bytesToBeCopied = (int) (baseIBS.length() - getPosition());
			bytesToBeCopied = bytesToBeCopied > 0?bytesToBeCopied:0;
			baseIBS.read(bytes, position, bytesToBeCopied);
			position = position + bytesToBeCopied;
			for(int i=position; i<position + length - bytesToBeCopied; i++)
				bytes[i] = (byte) paddedChar;
			padMark += length - bytesToBeCopied;
			return length;
		}
	}

	/**
	 * Set the current position in the underlying data.  This position is pegged
	 * to be no less than zero and no greater than the length of the InputByteStream.
	 * @param position Where to set the current position.
	 * @return This object.
	 * @throws IOException
	 */
	public InputByteStream seek(long position) throws IOException {
		if(position > this.length()){
			padMark = padLength;
			baseIBS.seek(baseIBS.length());
		}
		else if(position < baseIBS.length()){
			padMark = 0;
			baseIBS.seek(position);
		}
		else{
			baseIBS.seek(baseIBS.length());
			padMark = position - baseIBS.length();
		}
		return this;
	}

	
	/**
	* Create a new <code>InputByteStream</code> that provides another view on the underlying
	 * data.  This new <code>InputByteStream</code> is completely independent and can be read from
	 * and positioned without affecting the original or any other <i>slice</i>
	 * <code>InputByteStream</code> objects.  The initial position of the <i>slice</i> is at zero -
	 * the beginning of the <i>slice</i>.
	 * @param begin	Offset within the current <code>InputByteStream</code> to start the <i>slice</i>.
	 * @param length	The length of the <i>slice</i>.
	 * @return A <i>slice</i> of the original <code>InputByteStream</code>.
	 * @throws IOException
	 */
	
	public InputByteStream slice(long begin, long length) throws IOException {
		long padding = 0;
		if ((begin < 0) || (length < 0) || ((begin + length) > this.length()))
		{
			throw new IOException("Invalid slice of PaddedBytestream");
		}
		if(begin + length > baseIBS.length()){
			if(begin > baseIBS.length()){
				padding = length;
				begin = 0;
				length = 0;
			}
			else{
				padding = (int) (begin + length - baseIBS.length());
				length = length - padding;
			}
		}
		return new PaddedInputByteStream(this.baseIBS.slice(begin, length), paddedChar, padding);
	}
}
