/*
 *
 *	File: StreamManager.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.internal.io.stream;

import java.io.IOException;

import com.adobe.internal.io.ByteArrayByteWriter;
import com.adobe.internal.io.ByteBufferByteReader;
import com.adobe.internal.io.ByteReader;
import com.adobe.internal.io.ByteWriter;
import com.adobe.internal.io.ByteWriterFactory;
import com.adobe.internal.io.ByteWriterFactory.EncryptionStatus;
import com.adobe.internal.io.ByteWriterFactory.Fixed;
import com.adobe.internal.io.ByteWriterFactory.Longevity;

/**
 * Only for internal engineering use. This api can change without notice.
 * 
 * The source for all <code>InputByteStream</code> and <code>OutputByteStream</code> instances.
 * 
 * @exclude
 */
public final class StreamManager
{
	private final static boolean ignoreSlicesOfMaster = false;

	private ByteReader masterByteReader;
	private ByteWriterFactory byteWriterFactory;
	private StreamDatabase streamDB;

	private StreamManager(ByteWriterFactory byteWriterFactory, ByteReader masterByteReader)
	{
		this.byteWriterFactory = byteWriterFactory;
		this.masterByteReader = masterByteReader;
		this.streamDB = new StreamDatabase(masterByteReader, 
				true,  /* lenient on master byte reader closing*/
				true,  /* ignore closing nonexistent byte reader */
				true); /* ignore closing nonexistent byte writer */
	}

	/**
	 * Creates a new instance of the <code>StreamManager</code>.
	 * The <code>ByteWriterFactory</code> is used as the source of all new data sinks that are
	 * created.
	 * The "master" <code>ByteReader</code> is the main <code>ByteReader</code> used in the
	 * document processing.  It is the backing store for the document.  This will be closed and so
	 * will all <code>InputByteStream</code> instances made from it when the <code>StreamManager</code>
	 * is closed or the master <code>ByteReader</code> is reset.
	 * @param byteWriterFactory factory to use for creating new <code>OutputByteStream</code> instances
	 * @param masterByteReader the "master" byte reader, can be null if there is no master
	 * @return a newly created <code>StreamManager</code>
	 */
	public static StreamManager newInstance(ByteWriterFactory byteWriterFactory, ByteReader masterByteReader)
	{
		return new StreamManager(byteWriterFactory, masterByteReader);
	}

	/**
	 * Reset the master <code>ByteReader</code> and close it and all <code>InputByteStream</code>
	 * instances made from it.
	 * @param masterByteReader the new master <code>ByteReader</code>, can be null
	 * @throws IOException
	 */
	public void resetMasterByteReader(ByteReader masterByteReader) 
	throws IOException
	{
		this.masterByteReader = masterByteReader;
		ByteReader oldMasterByteReader = this.streamDB.resetMasterByteReader(masterByteReader);
		if (oldMasterByteReader != null)
		{
			oldMasterByteReader.close();
		}
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Create an <code>InputByteStream</code> from a <code>ByteReader</code>.
	 * @param byteReader A <code>ByteReader</code> to be used for creating the <code>InputByteStream</code>.
	 * @return An <code>InputByteStream</code>.
	 * @throws IOException
	 */
	public InputByteStream getInputByteStream(ByteReader byteReader)
	throws IOException
	{
		return new ByteReaderInputByteStream(this, byteReader, true);
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Create an <code>InputByteStream</code> from a <code>ByteReader</code>.
	 * @param byteReader A <code>ByteReader</code> to be used for creating the <code>InputByteStream</code>.
	 * @return An <code>InputByteStream</code>.
	 * @throws IOException
	 */
	public InputByteStream getInputByteStream(ByteReader byteReader, long startOffset, long length)
	throws IOException
	{
		return new ByteReaderInputByteStream(this, byteReader, startOffset, length, true);
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Create an <code>InputByteStream</code> from a byte array.
	 * @param b An array of bytes to be used for creating the <code>InputByteStream</code>.
	 * @return An <code>InputByteStream</code>.
	 * @throws IOException
	 */
	public InputByteStream getInputByteStream(byte[] b)
	throws IOException
	{
		java.nio.ByteBuffer buffer = java.nio.ByteBuffer.wrap(b);
		ByteReader reader = null;
		reader = new ByteBufferByteReader(buffer);
		return this.getInputByteStream(reader);
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Create an <code>OutputByteStream</code>.
	 * This creates an <code>OutputByteStream</code> that wraps multiple <code>OutputByteStream</code>
	 * objects and makes them appear as one large <code>OutputByteStream</code>.  The <code>OutputByteStream</code>
	 * objects that are passed must <b>not</b> be modified in any way until the wrapping
	 * <code>OutputByteStream</code> returned from this method is no longer in use.
	 * @param byteStreams An array of <code>OutputByteStream</code> objects.
	 * @return An <code>OutputByteStream</code>.
	 * @throws IOException
	 */
	public InputByteStream getInputByteStream(InputByteStream[] byteStreams)
	throws IOException
	{
		if (byteStreams != null)
		{
			return new ChainedInputByteStream(byteStreams);
		}
		return null;
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Create an <code>OutputByteStream</code> from a <code>ByteWriter</code>.
	 * @param byteWriter A <code>ByteWriter</code> to be used for creating the <code>OutputByteStream</code>.
	 * @return An <code>OutputByteStream</code>.
	 * @throws IOException
	 */
	public OutputByteStream getOutputByteStream(ByteWriter byteWriter)
	throws IOException
	{
		return new ByteWriterOutputByteStream(this, byteWriter, true /*register?*/);
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Create an <code>OutputByteStream</code> from a <code>ByteWriter</code>.  The
	 * returned <code>OutputByteStream</code> is unregistered and so will not be managed
	 * by the stream manager.  It is the responsibility of the caller to ensure that the
	 * <code>ByteWriter</code> and the <code>OutputByteStream</code> are properly closed.
	 * @param byteWriter A <code>ByteWriter</code> to be used for creating the <code>OutputByteStream</code>.
	 * @return An <code>OutputByteStream</code>.
	 * @throws IOException
	 */
	public OutputByteStream getUnregisteredOutputByteStream(ByteWriter byteWriter)
	throws IOException
	{
		return new ByteWriterOutputByteStream(this, byteWriter, false /*register?*/);
	}

	/**
	 * A utility function that calls 
	 * {@link #getOutputByteStream(Longevity, EncryptionStatus, Fixed, long, boolean)}
	 * with the following parameters:
	 * <br>Longevity - Longevity.TEMPORARY
	 * <br>EncryptionStatus - EncryptionStatus.CLEAR
	 * 
	 * This is intended to create an <code>OutputByteStream</code> whose duration is temporary
	 * (lasting the duration of a single method) and the data to be stored is not encrypted nor is
	 * it decrypted.
	 * @param fixed whether the memory size is fixed or growable
	 * @param size the number of bytes for <code>OutputByteStream</code>
	 * @return the newly created <code>OutputByteStream</code>
	 * @throws IOException
	 */
	public OutputByteStream getOutputByteStreamClearTemp(Fixed fixed, long size) 
	throws IOException
	{
		return this.getOutputByteStream(Longevity.TEMPORARY, EncryptionStatus.CLEAR, fixed, size, false);
	}

	/**
	 * A utility function that calls 
	 * {@link #getOutputByteStream(Longevity, EncryptionStatus, Fixed, long, boolean)}
	 * with the following parameters:
	 * <br>Longevity - Longevity.TEMPORARY
	 * <br>EncryptionStatus - EncryptionStatus.ENCRYPTED
	 * 
	 * This is intended to create an <code>OutputByteStream</code> whose duration is temporary
	 * (lasting the duration of a single method) and the data to be stored is encrypted.
	 * @param fixed whether the memory size is fixed or growable
	 * @param size the number of bytes for <code>OutputByteStream</code>
	 * @return the newly created <code>OutputByteStream</code>
	 * @throws IOException
	 */
	public OutputByteStream getOutputByteStreamEncryptedTemp(Fixed fixed, long size) 
	throws IOException
	{
		return this.getOutputByteStream(Longevity.TEMPORARY, EncryptionStatus.ENCRYPTED, fixed, size, false);
	}

	/**
	 * A utility function that calls 
	 * {@link #getOutputByteStream(Longevity, EncryptionStatus, Fixed, long, boolean)}
	 * with the following parameters:
	 * <br>Longevity - Longevity.TEMPORARY
	 * <br>EncryptionStatus - EncryptionStatus.DECRYPTED
	 * 
	 * This is intended to create an <code>OutputByteStream</code> whose duration is temporary
	 * (lasting the duration of a single method) and the data to be stored is decrypted.
	 * @param fixed whether the memory size is fixed or growable
	 * @param size the number of bytes for <code>OutputByteStream</code>
	 * @return the newly created <code>OutputByteStream</code>
	 * @throws IOException
	 */
	public OutputByteStream getOutputByteStreamDecryptedTemp(Fixed fixed, long size) 
	throws IOException
	{
		return this.getOutputByteStream(Longevity.TEMPORARY, EncryptionStatus.DECRYPTED, fixed, size, false);
	}

	/**
	 * A utility function that calls 
	 * {@link #getOutputByteStream(Longevity, EncryptionStatus, Fixed, long, boolean)}
	 * with the following parameters:
	 * <br>Longevity - Longevity.LONG
	 * <br>EncryptionStatus - EncryptionStatus.CLEAR
	 * 
	 * This is intended to create an <code>OutputByteStream</code> whose expected duration 
	 * of existence is of similar length to that of the <code>PDFDocument</code> and the 
	 * data to be stored is not encrypted nor is it decrypted.
	 * @param fixed whether the memory size is fixed or growable
	 * @param size the number of bytes for <code>OutputByteStream</code>
	 * @return the newly created <code>OutputByteStream</code>
	 * @throws IOException
	 */
	public OutputByteStream getOutputByteStreamClearDocument(Fixed fixed, long size) 
	throws IOException
	{
		return this.getOutputByteStream(Longevity.LONG, EncryptionStatus.CLEAR, fixed, size, false);
	}

	/**
	 * A utility function that calls 
	 * {@link #getOutputByteStream(Longevity, EncryptionStatus, Fixed, long, boolean)}
	 * with the following parameters:
	 * <br>Longevity - Longevity.LONG
	 * <br>EncryptionStatus - EncryptionStatus.ENCRYPTED
	 * 
	 * This is intended to create an <code>OutputByteStream</code> whose expected duration 
	 * of existence is of similar length to that of the <code>PDFDocument</code> and the 
	 * data to be stored is encrypted.
	 * @param fixed whether the memory size is fixed or growable
	 * @param size the number of bytes for <code>OutputByteStream</code>
	 * @return the newly created <code>OutputByteStream</code>
	 * @throws IOException
	 */
	public OutputByteStream getOutputByteStreamEncryptedDocument(Fixed fixed, long size) 
	throws IOException
	{
		return this.getOutputByteStream(Longevity.LONG, EncryptionStatus.ENCRYPTED, fixed, size, false);
	}

	/**
	 * A utility function that calls 
	 * {@link #getOutputByteStream(Longevity, EncryptionStatus, Fixed, long, boolean)}
	 * with the following parameters:
	 * <br>Longevity - Longevity.LONG
	 * <br>EncryptionStatus - EncryptionStatus.DECRYPTED
	 * 
	 * This is intended to create an <code>OutputByteStream</code> whose expected duration 
	 * of existence is of similar length to that of the <code>PDFDocument</code> and the 
	 * data to be stored is decrypted.
	 * @param fixed whether the memory size is fixed or growable
	 * @param size the number of bytes for <code>OutputByteStream</code>
	 * @return the newly created <code>OutputByteStream</code>
	 * @throws IOException
	 */
	public OutputByteStream getOutputByteStreamDecryptedDocument(Fixed fixed, long size) 
	throws IOException
	{
		return this.getOutputByteStream(Longevity.LONG, EncryptionStatus.DECRYPTED, fixed, size, false);
	}

	/**
	 * Create an <code>OutputByteStream</code> for use within the COS layer of the PDF toolkit.  
	 * This <code>OutputByteStream</code> will <b>not</b> be kept track of through an internal registry 
	 * and if not used correctly and carefully can lead to problems.  This is <b>not</b> to be used
	 * outside of the COS layer.
	 * 
	 * If it is desired to turn the 
	 * <code>OutputByteStream</code> into an <code>InputByteStream</code> this can be done
	 * by using the {@link OutputByteStream#closeAndConvert()} method.
	 * @param fixed is the <code>OutputByteStream</code> of a fixed size or not
	 * @param size the actual size of a fixed <code>OutputByteStream</code> or an estimate of a growable one
	 * @return a stream for writing data to
	 * @throws IOException
	 */
	public OutputByteStream getUnregisteredOutputByteStream(Fixed fixed, int size) 
	throws IOException
	{
		OutputByteStream obs;
		ByteWriter byteWriter = null;
//		try
//		{
		//byteWriter = byteWriterFactory.getMemoryByteWriter(fixed, size);
		byteWriter = new ByteArrayByteWriter(size);
		obs = new ByteWriterOutputByteStream(this, byteWriter, false);
//		} catch (IOException e) {
//		if (byteWriter != null)
//		{
//		byteWriter.close();
//		}
//		throw new PDFIOException("Error during creation of ByteWriter.", e);
//		}
		return obs;
	}

	/**
	 * Create an <code>OutputByteStream</code> for use within the PDF toolkit.  This <code>OutputByteStream</code>
	 * will be kept track of through an internal registry and will need to be properly closed by the caller.  If
	 * it is desired to turn the <code>OutputByteStream</code> into an <code>InputByteStream</code> this can be done
	 * by using the {@link OutputByteStream#closeAndConvert()} method.
	 * @param longevity an estimate of how long the <code>OutputByteStream</code> will remain in use
	 * @param encryption the encryption status of the contents of the <code>OutputByteStream</code>
	 * @param fixed is the <code>OutputByteStream</code> of a fixed size or not
	 * @param size the actual size of a fixed <code>OutputByteStream</code> or an estimate of a growable one
	 * @param fast a request for a faster <code>OutputByteStream</code> which is only a suggestion
	 * @return a stream for writing data to
	 * @throws IOException
	 */
	public OutputByteStream getOutputByteStream(Longevity longevity, EncryptionStatus encryption, 
			Fixed fixed, long size, boolean fast) 
	throws IOException
	{
		OutputByteStream obs;
		ByteWriter byteWriter;
		byteWriter = byteWriterFactory.getByteWriter(longevity, encryption, fixed, size, fast);
		obs = this.getOutputByteStream(byteWriter);
		return obs;
	}

	/**
	 * Create a <code>ByteWriter</code>.  This <code>ByteWriter</code> is not kept track of and 
	 * will need to be properly closed by the caller.  This is a potentially large resource leak if used
	 * improperly.  Be <b>extremely</b> careful.  This method is only in place for one place within
	 * the code that will soon be rewritten.  Do not use for new code.
	 * <p>This method has similar parameters as
	 * {@link #getOutputByteStream(Longevity, EncryptionStatus, Fixed, long, boolean)}
	 * but makes the following default choices:
	 * <br>Longevity - Longevity.LONG
	 * <br>EncryptionStatus - EncryptionStatus.CLEAR
	 * @param fixed is the <code>OutputByteStream</code> of a fixed size or not
	 * @param size the actual size of a fixed <code>OutputByteStream</code> or an estimate of a growable one
	 * @return a stream for writing data to
	 * @throws IOException
	 */
	public ByteWriter getByteWriter(Fixed fixed, long size) 
	throws IOException
	{
		return byteWriterFactory.getByteWriter(Longevity.LONG, EncryptionStatus.CLEAR, fixed, size, false);
	}

	/**
	 * Close this <code>StreamManager</code>.  This closes all open <code>ByteReader</code> instances and
	 * any <code>InputByteStream</code> instances made from.  The <code>ByteWriterFactory</code> given
	 * during construction is also closed.
	 * 
	 * In the future if there are open <code>ByteReader</code> instances this will be considered
	 * an error state and an exception will be thrown.
	 * @throws IOException
	 */
	public void close() 
	throws IOException
	{
		this.close(false);
	}

	/**
	 * Close this <code>StreamManager</code>.  This closes all open <code>ByteReader</code> instances and
	 * any <code>InputByteStream</code> instances made from.  The <code>ByteWriterFactory</code> given
	 * during construction is also closed.
	 * 
	 * In the future if there are open <code>ByteReader</code> instances this will be considered
	 * an error state and an exception will be thrown.
	 * @param ignoreMaster if <code>true</code> the master ByteReader is ignored for closing and
	 * it will be the responsibility of the caller to close it
	 * @throws IOException
	 */
	public void close(boolean ignoreMaster) 
	throws IOException
	{
		try
		{
//			if (!ignoreMaster)
//			{
//				this.streamDB.resetMasterByteReader(null);
//			}
//			if (!this.streamDB.isEmptyExceptForMasterByteReader())
			{
				// TODO - force all closed
				this.streamDB.closeAllOpen(ignoreMaster);
//				System.out.println(this.streamDB.toString());
//				throw new RuntimeException("StreamManager closing with Streams and/or ByteWriters still open.");
			}
		} finally {
			try
			{
				this.byteWriterFactory.closeFactory();
			}
			catch (IOException e)
			{
				IOException ioe = new IOException("Unable to close the ByteWriterFactory.");
				ioe.initCause(e);
				throw ioe;
			}
		}
	}

	//////////////// Input/Output ByteStream Communication

	/*package protected*/ void deregisterInputByteStream(ByteReaderInputByteStream ibs, 
			ByteReader byteReader, boolean isSlice) 
	throws IOException
	{
		//if (closeStream)
		if (ignoreSlicesOfMaster && (byteReader == this.masterByteReader) && isSlice)
		{
			return;
		}

		if (this.streamDB.removeIBS(ibs, byteReader))
		{
			try
			{
				byteReader.close();
			} catch (IOException e) {
				IOException ioe = new IOException("Unable to close ByteReader");
				ioe.initCause(e);
				throw ioe;
			}
		}
	}

	/*package protected*/ void registerInputByteStream(ByteReaderInputByteStream ibs, 
			ByteReader byteReader, boolean isSlice)
	{
		if (ignoreSlicesOfMaster && (byteReader == this.masterByteReader) && isSlice)
		{
			return;
		}
		this.streamDB.addIBS(ibs, byteReader);
	}

	/*package protected*/ boolean deregisterOutputByteStream(OutputByteStream obs, ByteWriter byteWriter) 
	throws IOException
	{
		boolean closeStream = this.streamDB.removeOBS(obs, byteWriter);
		if (closeStream)
		{
			try
			{
				this.byteWriterFactory.closeByteWriter(byteWriter);
			} catch (IOException e) {
				IOException ioe = new IOException("Unable to close ByteReader");
				ioe.initCause(e);
				throw ioe;
			}
		}
		return closeStream;
	}

	/*package protected*/ void registerOutputByteStream(ByteWriterOutputByteStream obs, ByteWriter byteWriter)
	{
		this.streamDB.addOBS(obs, byteWriter);
	}	
}
