/* ****************************************************************************
 *
 *	File: XRefTable.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASNumber;
import com.adobe.internal.pdftoolkit.core.types.ASObject;
import com.adobe.internal.pdftoolkit.core.types.ASString;
import com.adobe.internal.pdftoolkit.core.util.ByteOps;

/**
 * Represents a PDF cross reference table. The cross reference table is
 * defined in section 3.4.3 of the PDF Reference Manual version 1.4.
 *
 * TODO: Needs to support the cross-reference stream and hybrid
 * cross-references introduced in PDF 1.5.
 */
class XRefTable
{
	static final int tTable = 0;
	static final int tStream = 1;
	static final int tHybrid = 2;
	private static final int mMaxChunk = 262144;
	private static final int EOF_CHECK_THRESHOLD = 1024;
	private int mXrefType;

	private InputByteStream mFileBuf;	// Underlying data stream for entire pdf file
	private InputByteStream mBuf;		// XRef table stream, which may be different
	private CosDocument mDoc;		// Document containing the xref
	private ArrayList mXRefSubSections;	// List of xref table subsections
	private CosDictionary mTrailer;		// Trailer dictionary
	private ArrayList mTrailerList;		// Trailer dictionary list
	private long[] mRevisions;		// EOF values for document revisions
	private boolean mNoPreload;		// Do no preload main XRef section if true
	private boolean mIsFDF;			// This is an FDF document if true
	private int highestObjectNumInXRefEntries = 0; // Highest object number found in all Xref tables
	private TableXRefSubSection mainXrefSubSection = null;// This is the first subsection in the main Xref section.
	  // If main xref section is present as stream then this will be bull.

	private boolean isXrefIntialized;
	/**
	 * Constructs a cross reference table.
	 */
	XRefTable(CosDocument doc, InputByteStream buf, boolean noPreload, boolean isFDF)
	{
		mFileBuf = buf;
		mBuf = buf;
		mDoc = doc;
		mNoPreload = noPreload;
		mIsFDF = isFDF;
	}

	void loadMainInfos()
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		if (mIsFDF) {
			rebuild();
		} else {
			mXRefSubSections = new ArrayList();
			mTrailerList = new ArrayList();
			long nextXRefOffset;
			ArrayList eofList = new ArrayList();
			nextXRefOffset = getLastXRefSectionPosition();
			mBuf.seek(nextXRefOffset);
			byte b = CosToken.skipWhitespace(mBuf);
			mBuf.unget();
			if (nextXRefOffset == 0)
				throw new PDFCosParseException("could not find xref section");
			if (b == 'x') {
				/*
				 * This is either a classical 1.4 xref table or maybe a hybrid one.
				 */
				mXrefType = tTable;	// until proven otherwise
				parseTableXrefChain(nextXRefOffset, eofList);
			} else if (ByteOps.isDigit(b)) {
				/* This is one of the new xref stream situations */
				mXrefType = tStream;
				parseStreamXrefChain(nextXRefOffset, eofList);
			} else {
				throw new PDFCosParseException("could not find xref section");
			}
			setRevisions(eofList);
			// Set begin entry of main subsection to 0 only when PDF has been repaired.
			// Although it is required by PDF specification (if no incremental updates)
			// have been done), there are some files(Watson#3491172) whose begin entry 
			// starts as 1.
			if(mDoc.getRepairTypes().contains(REPAIRTYPE.xrefRepair))
			{
				// XRef repair hack follows, main table XRef section MUST begin with zero
				if (mainXrefSubSection != null) {
					mainXrefSubSection.mEnd -= mainXrefSubSection.mBegin;
					mainXrefSubSection.mBegin = 0;
				}
			}			
			if (!mTrailer.containsKey(ASName.k_Size)) {
				XRefSubSection lastSection = (XRefSubSection)mXRefSubSections.get(0);
				mTrailer.put(ASName.k_Size, lastSection.mEnd);
			}
		}
		isXrefIntialized = true;
	}

	void rebuild()
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		mXRefSubSections = new ArrayList();
		mTrailerList = new ArrayList();
		mRevisions = null;
		ArrayList tempTrailerList = new ArrayList();
		ArrayList tempEOFList = new ArrayList();
		mBuf.seek(0);
		boolean first = true;
		boolean isLinearized = false;
		int catalogObjNum = 0;
		while (true) {
			CosToken.skipWhitespace(mBuf);
			mBuf.unget();
			long objPos = mBuf.getPosition();
			ASObject asObj = CosToken.readPrimitive(mBuf);
			if (asObj == null)
				break;
			if (asObj instanceof ASNumber) {
				CosObject cosObj = scanIndirectObj((ASNumber)asObj, objPos);
				if (first && cosObj != null) {
					first = false;
					if (cosObj instanceof CosDictionary) {
						isLinearized = ((CosDictionary)cosObj).containsKey(ASName.k_Linearized);
					}
				}
				if (cosObj instanceof CosDictionary) {
					CosObject cosType = ((CosDictionary)cosObj).get(ASName.k_Type);
					if (cosType instanceof CosName && ((CosName)cosType).nameValue() == ASName.k_Catalog) {
						catalogObjNum = ((ASNumber)asObj).intValue();
					}
				}
			} else if (asObj instanceof ASString) {
				String str = asObj.toString();
				if ("startxref".equals(str)) {
					long nextEOF = 0;
					try {
						nextEOF = findNextEOF();
						mBuf.seek(nextEOF);
					} catch (PDFCosParseException e) {
						nextEOF = mBuf.getPosition();
					}
					tempEOFList.add(Long.valueOf(nextEOF));
				} else if ("trailer".equals(str)) {
					long savePos = mBuf.getPosition();
					try {
						CosObject trailerDict = CosToken.readObject(mDoc, mBuf, null);
						if (!(trailerDict instanceof CosDictionary))
							throw new PDFCosParseException("Invalid trailer dictionary");
						tempTrailerList.add(trailerDict);
					} catch (PDFCosParseException e) {
						mBuf.seek(savePos);
					}
				}
			}
		}
		if (tempTrailerList.isEmpty() && catalogObjNum > 0) {
			CosDictionary trailer = mDoc.createDirectCosDictionary();
			trailer.put(ASName.k_Root, new CosObjectRef(mDoc, mDoc.getIndexedInfo(catalogObjNum)));
			trailer.put(ASName.k_Size, mDoc.createCosNumeric(mDoc.getNumObjectsInternal()));
			trailer.put(ASName.k_ID, mDoc.createUpdateDocID(true));
			tempTrailerList.add(trailer);
		}
		while (!tempTrailerList.isEmpty())
			mTrailerList.add(tempTrailerList.remove(tempTrailerList.size() - 1));
		if (isLinearized && mTrailerList.size() > 1) {
			Object x = mTrailerList.get(mTrailerList.size() - 1);
			mTrailerList.set(mTrailerList.size() - 1, mTrailerList.get(mTrailerList.size() - 2));
			mTrailerList.set(mTrailerList.size() - 2, x);
		}
		if (!mTrailerList.isEmpty()) {
			mTrailer = (CosDictionary)mTrailerList.get(0);
			if (catalogObjNum > 0)
				mTrailer.put(ASName.k_Root, new CosObjectRef(mDoc, mDoc.getIndexedInfo(catalogObjNum)));
			mTrailer.put(ASName.k_Size, mDoc.getNumObjectsInternal());
		}
		ArrayList eofList = new ArrayList();
		while (!tempEOFList.isEmpty())
			eofList.add(tempEOFList.remove(tempEOFList.size() - 1));
		if (isLinearized && eofList.size() > 1) {
			Object x = eofList.get(eofList.size() - 1);
			eofList.set(eofList.size() - 1, eofList.get(eofList.size() - 2));
			eofList.set(eofList.size() - 2, x);
		}
		if (mTrailer == null && !mIsFDF)
			throw new PDFCosParseException("Rebuilt document still has no trailer");
		setRevisions(eofList);
	}

	CosObject scanIndirectObj(ASNumber asObjNum, long objPos)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		long savePos = mBuf.getPosition();
		ASObject asObjGen = CosToken.readPrimitive(mBuf);
		ASObject asObjTag = CosToken.readPrimitive(mBuf);
		if (asObjGen instanceof ASNumber && asObjTag instanceof ASString) {
			Number nmObjNum = asObjNum.numberValue();
			Number nmObjGen = ((ASNumber)asObjGen).numberValue();
			String objTag = ((ASString)asObjTag).toString();
			if ((nmObjNum instanceof Integer || nmObjNum instanceof Long) && (nmObjGen instanceof Integer || nmObjGen instanceof Long)) {
				int objNum = nmObjNum.intValue();
				int objGen = nmObjGen.intValue();
				if (objNum > 0 && objGen >= 0 && objGen < 65535 && "obj".equals(objTag)) {
					CosObject cosObj = null;
					try {
						cosObj = CosToken.readIndirectObject(mDoc, mBuf, null);
					} catch (PDFCosParseException e) {
					}
					if (cosObj != null) {
						CosObjectInfo info = new CosObjectInfo(mDoc, objNum, objGen);
						info.setPosInternal(objPos);
						info.markAddressed();
						mDoc.putRebuiltInfo(objNum, info);
						if (cosObj instanceof CosStream) {
							CosObject lenObj = ((CosStream)cosObj).get(ASName.k_Length);
							if (lenObj instanceof CosNumeric) {
								int stmLen = ((CosNumeric)lenObj).intValue();
								long saveStmPos = mBuf.getPosition();
								mBuf.seek(saveStmPos + stmLen);
								ASObject asTokenObj = CosToken.readPrimitive(mBuf);
								if (asTokenObj instanceof ASString && asTokenObj.toString().equals("endstream"))
									return cosObj;
								mBuf.seek(saveStmPos);
								while (true) {
									CosToken.skipWhitespace(mBuf);
									mBuf.unget();
									asTokenObj = CosToken.readPrimitive(mBuf);
									if (asTokenObj == null)
										return cosObj;
									if (asTokenObj instanceof ASString && asTokenObj.toString().equals("endstream"))
										return cosObj;
								}
							}
						} else {
							mBuf.unget();
						}
						return cosObj;
					}
				}
			}
		}
		mBuf.seek(savePos);
		return null;
	}

	void rebuildLate()
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		mBuf.seek(0);
		while (true) {
			CosToken.skipWhitespace(mBuf);
			mBuf.unget();
			long objPos = mBuf.getPosition();
			ASObject asObj = CosToken.readPrimitive(mBuf);
			if (asObj == null)
				break;
			if (asObj instanceof ASNumber) {
				long savePos = mBuf.getPosition();
				try{
					scanIndirectObj((ASNumber)asObj, objPos);
				}catch(PDFCosParseException e)
				{
					// ignore cos-parsing exceptions while rebuilding only if this flag is set.
					if(! mDoc.getOptions().skipCorruptObjects())
					{
						throw e;
					}
				}
				finally{
					if(mBuf.getPosition() < objPos)
						mBuf.seek(savePos);//if we don't do this, then we may struck in infinite loop.
										   // we should always move forward loading CosObjects.
				}
			}
		}
	}

	/**
	 * @author gish
	 * Constructs an empty cross reference table for a new doc.
	 *
	 * @param doc Document containing the cross reference table
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 *
	 */
	XRefTable(CosDocument doc)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mDoc = doc;
		mXRefSubSections = new ArrayList();
		mTrailer = mDoc.createCosDictionary(CosObject.DIRECT);
	}

	/**
	 * Close any resources used by this object. After this call returns this object is
	 * no longer in a usable state.
	 *
	 * @throws IOException
	 */
	// kept separate from closeStreams()
	// this one can close down more - closeStreams() is used internally
	void close()
		throws IOException
	{
		try {
			closeStreams();
		} finally {
			closeXRefSubSections();
		}
	}

	private void closeStreams()
		throws IOException
	{
		// only want to close mBuf if NOT the same as mFileBuf
		try {
			if (mBuf != null && mBuf != mFileBuf) {
				mBuf.close();
				mBuf = null;
			}
		} finally {
			if (mFileBuf != null) {
				mFileBuf.close();
				mFileBuf = null;
			}
		}
	}

	private void closeXRefSubSections()
		throws IOException
	{
		IOException ioEx = null;
		Iterator iter = mXRefSubSections.iterator();
		while (iter.hasNext()) {
			XRefSubSection subSection = (XRefSubSection)iter.next();
			try {
				if (subSection != null)
					subSection.close();
			} catch (IOException e) {
				ioEx = e;
			}
		}
		if (ioEx != null)
			throw ioEx;
		mXRefSubSections.clear();
	}

	private long find(InputByteStream byteStream, char[] key)
		throws IOException
	{
		long location = -1;
		while (byteStream.bytesAvailable() > 0) {
			char c = (char)byteStream.read();
			if(key[0] == c) {
				location = byteStream.getPosition() - 1;
				for(int i = 1; i < key.length; i++) {
					c = (char)byteStream.read();
					if(key[i] != c) {
						location = -1;
						break;
					}
				}
				if (location != -1) return location;
			}
		}
		return location;
	}

	/**
	 * Parse the linked list of xref sections of the type in 1.4 and earlier file formats
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	private void parseTableXrefChain(long nextXRefOffset, ArrayList eofList)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		do {
			mBuf.seek(nextXRefOffset);
			int nextXrefSubSectionIndex = mXRefSubSections.size();
			byte b = readTableXRefTable();
			// We are assuming the first subsection of the just read xref section as main xref subsection.
			// If there are more XREF sections to read then this shall automatically be overwritten.
			if(mXRefSubSections.size() > nextXrefSubSectionIndex)
				mainXrefSubSection = (TableXRefSubSection) mXRefSubSections.get(nextXrefSubSectionIndex);
			CosDictionary trailer = readTrailer(b);
			mTrailerList.add(trailer);
			if (mTrailer == null) {
				mTrailer = trailer;
			}
			CosNumeric prev = (CosNumeric)trailer.get(ASName.k_Prev);
			if (prev != null) {
				long prevValue = prev.longValue();
				if (prevValue == nextXRefOffset)
					throw new PDFCosParseException("Fatal XRef chain loop at position " + Long.toString(prevValue));
				nextXRefOffset = prevValue;
				CosNumeric xref = (CosNumeric)trailer.get(ASName.k_XRefStm);
				if (xref != null) {
					long pos = mBuf.getPosition();
					try{
						readStreamXRefTable(xref.longValue());
					}finally{
						mBuf.seek(pos);
						mainXrefSubSection = null;// we just read stream XREF, if this is main xref section 
						  //then we don't need this to store.
					}
					mXrefType = tHybrid;
				}
			} else
				nextXRefOffset = 0;
			if (eofList != null && mBuf.getPosition() > nextXRefOffset)
				eofList.add(Long.valueOf(findNextEOF()));
		} while (nextXRefOffset != 0);
		
		int size = mXRefSubSections.size();
		boolean objectNumberZeroFound = true;
		
		if(size > 0)
		{
			objectNumberZeroFound = false;
			for(int i = 0; i<mXRefSubSections.size(); i++) {
				// Get ith xref subsection
				XRefSubSection sub = (XRefSubSection) mXRefSubSections.get(i);
				if(sub instanceof TableXRefSubSection) {
					if(((TableXRefSubSection) sub).mBegin == 0) {
						objectNumberZeroFound = true;
					}
				}
			}
		}
		
		if(!objectNumberZeroFound) {
			// Throw a PDFCosParseException which triggers repair
			throw new PDFCosParseException("Object number 0 is not present in any of the Xref subsections.");
		}
		
	}

	/**
	 * Parses the cross reference table.
	 *
	 * @return First byte not in the cross reference table.
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private byte readTableXRefTable()
		throws PDFCosParseException, IOException
	{
		try{
			/* The spec doesn't allow it, but Acrobat seems willing to skip white space
			 * at the beginning of an xref table. We'll do the same.
			 */
			byte b = 0;
			CosToken.skipWhitespace(mBuf);
			mBuf.unget();
			// TODO - move to generic parse routines
			if (!CosToken.readLine(mBuf, false).startsWith("xref")) {
				throw new PDFCosParseException("Expected 'xref' : " + Long.toString(mBuf.getPosition()));
			}
			b = CosToken.skipWhitespace(mBuf);
			while (ByteOps.isDigit(b)) {
				mXRefSubSections.add(new TableXRefSubSection(b));
				b = CosToken.skipWhitespace(mBuf);
			}
			return b;
		}
		catch(IOException e)
		{
			// Wrap IOException as PDFCosparseEx, so that xref repair code is triggered.
			throw new PDFCosParseException(e);
		}
		
	}

	/**
	 * Parse a chain of xref streams
	 * @param nextXRefOffset
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	private void parseStreamXrefChain(long nextXRefOffset, ArrayList eofList)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		do {
			mBuf.seek(nextXRefOffset);
			CosStream xrefStm = readStreamXRefTable(nextXRefOffset);
			mTrailerList.add(xrefStm);
			if (mTrailer == null)
				mTrailer = xrefStm;
			CosNumeric prev = (CosNumeric)xrefStm.get(ASName.k_Prev);
			if (prev != null) {
				long prevValue = prev.longValue();
				if (prevValue == nextXRefOffset)
					throw new PDFCosParseException("Fatal XRef chain loop at position " + Long.toString(prevValue));
				nextXRefOffset = prevValue;
			} else
				nextXRefOffset = 0;
			if (eofList != null && mBuf.getPosition() > nextXRefOffset)
				eofList.add(Long.valueOf(findNextEOF()));
		} while (nextXRefOffset != 0);
	}

	private CosStream readStreamXRefTable(long nextXRefOffset)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		/* make a new stream object to stay reentrant */
		mBuf.seek(nextXRefOffset);
		CosParseBuf pBuf = new CosParseBuf(mBuf, 128);
		int[] xStmID = CosToken.readObjID(mDoc, pBuf, CosToken.skipWhitespace(pBuf));
		CosObjectInfo xStmInfo = mDoc.getIndexedInfo(xStmID[0]);
		if (xStmInfo != null) {
			xStmInfo.setObjNum(xStmID[0]);
			xStmInfo.setObjGen(xStmID[1]);
		} else {
			xStmInfo = mDoc.getObjectInfo(xStmID[0], xStmID[1]);
		}
		CosObject xStmObj = CosToken.readIndirectObject(mDoc, pBuf, xStmInfo);
		pBuf.close();
		xStmInfo.setPosInternal(nextXRefOffset);
		xStmInfo.markLoaded();
		mDoc.putIndexedInfo(xStmID[0], null);
		if (!(xStmObj instanceof CosStream))
			throw new PDFCosParseException("Expected CosStream : " + Long.toString(nextXRefOffset));
		CosStream xStm = (CosStream)xStmObj;
		InputByteStream xStmData = xStm.getStreamDecoded();

		/*
		 * Extract some information out of the dictionary part of the stream object, but
		 * don't touch the stream data at this point. If we're lucky, we'll never have
		 * to decompress the stream data.
		 */
		Object xTypeObj = xStm.get(ASName.k_Type);
		if (!(xTypeObj != null && (xTypeObj instanceof CosName) && (((CosName)xTypeObj).nameValue().equals(ASName.k_XRef))))
			throw new PDFCosParseException("Expected CosName: " + Long.toString(nextXRefOffset));
		CosObject indexObj = xStm.get(ASName.k_Index);
		CosObject wObj = xStm.get(ASName.k_W);
		if (!(wObj instanceof CosArray) || (((CosArray)wObj).size() != 3))
			throw new PDFCosParseException("UExpected CosArray : " + Long.toString(nextXRefOffset));
		CosArray wArray = (CosArray)wObj;
		int[] w = new int[3];
		int totalW = 0;
		for (int i = 0; i < 3; i++) {
			w[i] = ((CosNumeric)wArray.get(i)).intValue();
			totalW += w[i];
		}
		int[] intArray = null;
		if (indexObj == null) {
			/* the default is [0..size] */
			intArray = new int[2];
			intArray[0] = 0;
			intArray[1] = ((CosNumeric)xStm.get(ASName.k_Size)).intValue();
		} else {
			if (!(indexObj instanceof CosArray))
				throw new PDFCosParseException("Expected CosArray : " + Long.toString(nextXRefOffset));
			CosArray indexArray = (CosArray)indexObj;
			intArray = new int[indexArray.size()];
			for (int i = 0; i < indexArray.size(); i++) {
				intArray[i] = ((CosNumeric)indexArray.get(i)).intValue();
			}
		}
		long streamOffset = 0;
		for (int i = 0; i < intArray.length; i += 2) {
			int first = intArray[i];
			int count = intArray[i + 1];
			while (count > 0) {
				int subCount = count;
				if (count * totalW > mMaxChunk)
					subCount = mMaxChunk / totalW;
				mXRefSubSections.add(new StreamXRefSubSection(first, subCount, xStm, xStmData, streamOffset, w, totalW));
				first += subCount;
				count -= subCount;
				streamOffset += subCount * totalW;
			}
		}
		xStmData.close();
		return xStm;
	}

	/**
	 * Obtains the PDF document trailer dictionary.
	 *
	 * @return COSDictionary
	 */
	CosDictionary getTrailer()
	{
		return mTrailer;
	}

	CosDictionary[] getTrailerList()
	{
		if (mTrailerList.isEmpty())
			return null;
		CosDictionary[] list = new CosDictionary[mTrailerList.size()];
		for (int i = 0; i < mTrailerList.size(); i++)
			list[i] = (CosDictionary)mTrailerList.get(mTrailerList.size() - i - 1);
		return list;
	}

	/**
	 * @author gish
	 * Parses a CosObject from the input stream and returns it as a
	 * CosObject instance
	 * @param info - CosObjectInfo of indirect object to parse
	 * @return CosObject parsed object from input bytestream or
	 * compressed object stream bytestream (returns null if object
	 * not found or if object entry in xref is free)
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	CosObject getIndirectObject(CosObjectInfo info)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		CosObject result = null;
		if (info != null) {
			if (!info.isAddressed())
				info = getInfo(info);
			if (info == null )
				return result;
			if (!info.isFree()) {
				if (info.isCompressed()) {
					CosObject stmObject;
					int objNum = info.getObjNum();
					CosObjectInfo stmInfo = info.getStreamInfo();
					CosLinearization cosLin = mDoc.getLinearization();
					if (cosLin != null && cosLin.getOldToOldObjStmMap() != null) {
						objNum = cosLin.mapNewToOldObjNum(objNum);
						stmObject = cosLin.mapOldToOldObjStm(stmInfo.getObjNum());
					} else {
						stmObject = mDoc.getIndirectObject(stmInfo);
					}
					if(stmObject instanceof CosObjectStream)
					{
						InputByteStream stm = ((CosObjectStream)stmObject).getBytesForObject(objNum);
						info.setNextObjPos(CosToken.DEFAULT_NEXT_OBJ_POS);
						result = CosToken.readObject(mDoc, stm, info);
					}
					else
					{
						throw new PDFCosParseException("Expected cosobject stream.");
					}
					
				} else {
					long pos = info.getPos();
					if(pos == 0)
						throw new PDFCosParseException("The position 0 for an object is not valid."); // Position 0 is for PDF version header
					if(pos >= mFileBuf.length())
						throw new PDFCosParseException("The position of (obj "+info.getObjNum()+" 0) lies outside the file boundary");
					mFileBuf.seek(pos);
					CosParseBuf pBuf = new CosParseBuf(mFileBuf, 128);
					CosToken.skipObjID(mDoc, pBuf, CosToken.skipWhitespace(pBuf));
					result = CosToken.readIndirectObject(mDoc, pBuf, info);
					pBuf.close();
				}
				info.markLoaded();
			}
		}
		return result;
	}

	/**
	 * Verify this object number is in use and return an info object for it if so.
	 *
	 * @param objNum Indirect object number of COS object whose byte location is desired
	 *
	 * @return Byte offset from the start of the file.
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 */
	CosObjectInfo getInfo(int objNum)
		throws PDFCosParseException, IOException, PDFSecurityException
	{
		Iterator iter = mXRefSubSections.iterator();
		while (iter.hasNext()) {
			XRefSubSection xref = (XRefSubSection) iter.next();
			int begin = xref.getBegin();
			if (begin <= objNum && objNum < xref.getEnd()) {
				if (xref.getXRefUsed(objNum - begin)) {
					CosObjectInfo info = new CosObjectInfo(mDoc,objNum, xref.getXRefGeneration(objNum - begin));
					return xref.getInfo(objNum - begin, info);
				}
			}
		}
		return null;
	}

	/**
	 * Verify this object number and generation are in use. Return null if not.
	 *
	 * @param info Info of the COS object whose byte location is desired
	 *
	 * @return Byte offset from the start of the file.
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 */
	private CosObjectInfo getInfo(CosObjectInfo info)
		throws PDFCosParseException, IOException, PDFSecurityException
	{
		int id = info.getObjNum();
		int generation = info.getObjGen();
		Iterator iter = mXRefSubSections.iterator();
		CosObjectInfo rslt = null;
		while (iter.hasNext() && rslt == null) {
			XRefSubSection xref = (XRefSubSection) iter.next();
			int begin = xref.getBegin();
			if (begin <= id && id < xref.getEnd()) {
				if (xref.getXRefGeneration(id - begin) == generation && xref.getXRefUsed(id - begin))
					rslt = xref.getInfo(id - begin, info);
				return rslt;
			}
		}
		return rslt;
	}

	/**
	 * Preload addresses and create CosObjectInfo records for all sections
	 * EXCEPT the main section. Permits unique resolution of references early on.
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 */
	void loadUpdateInfos()
		throws PDFCosParseException, IOException, PDFSecurityException
	{
		Iterator iter = mXRefSubSections.iterator();
		while (iter.hasNext()) {
			XRefSubSection xref = (XRefSubSection)iter.next();
			if (mNoPreload && mRevisions != null && getXRefSectionPos(xref) < mRevisions[0])
				break;
			int begin = xref.getBegin();
			for (int id = begin; id < xref.getEnd(); id++) {
				if (id != 0) {
					int gen = (xref.getXRefUsed(id - begin)) ? xref.getXRefGeneration(id - begin) : 65535;
					CosObjectInfo info = mDoc.getObjectInfo(id, gen);
					if (info != null && (info.isAddressed() || info.isLoaded() || info.isDirty()))
						continue;
					xref.getInfo(id - begin, info);
				}
			}
		}
	}

	/**
	 * Reset XRef at the end of a save operation.
	 * Then rebuild the XRef sections array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	void resetXRef(InputByteStream stm)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		// FIXME - we are potentially dropping a stream on the ground here
		// it will be caught at document close time but not before
		// TODO - bring the closing back to life

		mFileBuf = stm;
		mBuf = stm;
		mXRefSubSections.clear();
		ArrayList eofList = new ArrayList();
		mTrailerList = new ArrayList();
		long nextXRefOffset = getLastXRefSectionPosition();
		mBuf.seek(nextXRefOffset);
		byte b = CosToken.skipWhitespace(mBuf);
		mBuf.unget();
		if (b == 'x') {
			mXrefType = tTable;
			parseTableXrefChain(nextXRefOffset, eofList);
		} else if (ByteOps.isDigit(b)) {
			mXrefType = tStream;
			parseStreamXrefChain(nextXRefOffset, eofList);
		} else {
			throw new PDFCosParseException("could not find xref section");
		}
		setRevisions(eofList);
	}

	/**
	 * Find the next EOF marker position.
	 * Deal wit the line endings that follow it.
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private long findNextEOF()
		throws PDFCosParseException, IOException
	{
		char[] key_eof = {'%', '%', 'E', 'O', 'F'};
		long eofPos = find(mBuf, key_eof);
		if (eofPos < 0)
			throw new PDFCosParseException("Could not find EOF");
		byte nextByte = 0;
		while (!mBuf.eof() && nextByte != (byte)'\n' && nextByte != (byte)'\r')
			nextByte = (byte)mBuf.read();
		if (mBuf.eof())
			return mBuf.length();
		nextByte = (byte)mBuf.read();
		if (mBuf.eof())
			return mBuf.length();
		eofPos = mBuf.getPosition();
		if (nextByte != (byte)'\n' && nextByte != (byte)'\r')
			eofPos--;
		return eofPos;
	}

	/**
	 * Get the EOF position of the update section in which this
	 * object resides. Returns zero if the object is direct or
	 * unsaved.
	 */
	long getObjEOF(CosObjectInfo objInfo)
	{
		if (objInfo == null || mRevisions == null)
			return 0;
		if (objInfo.isCompressed())
			objInfo = objInfo.getStreamInfo();
		long objPos = objInfo.getPos();
		for (int i = 0; i < mRevisions.length; i++)
			if (objPos < mRevisions[i])
				return mRevisions[i];
		return 0;
	}

	/**
	 * Get the EOF position of this update section.
	 * Returns zero if there is no such update section.
	 */
	long getRevisionEOF(int revision)
	{
		if (mRevisions == null || revision < 0 || revision >= mRevisions.length)
			return 0;
		return mRevisions[revision];
	}

	/**
	 * Get the revision number associated with this object.
	 * Returns zero for direct objects. Retunrs zero if the
	 * object is from the original file or -1 if the object
	 * is direct or unsaved.
	 */
	int getObjRevision(CosObjectInfo objInfo)
	{
		if (objInfo == null || mRevisions == null)
			return -1;
		if (objInfo.isCompressed())
			objInfo = objInfo.getStreamInfo();
		long objPos = objInfo.getPos();
		for (int i = 0; i < mRevisions.length; i++)
			if (objPos < mRevisions[i])
				return i;
		return -1;
	}

	/**
	 * Get total number of revisions for this XRef
	 */
	int getNumRevisions()
	{
		if (mRevisions == null)
			return 0;
		return mRevisions.length;
	}

	void getChangedObjects(long eof, CosList modList)
		throws PDFCosParseException, IOException, PDFSecurityException
	{
		Iterator iter = mXRefSubSections.iterator();
		while (iter.hasNext()) {
			XRefSubSection subSec = (XRefSubSection)iter.next();
			if (getXRefSectionPos(subSec) < eof)
				return;
			int first = subSec.getBegin();
			int end = subSec.getEnd();
			int subSecSize = end - first;
			for (int index = 0; index < subSecSize; index++) {
				int objNum = first + index;
				CosObjectID id = (CosObjectID)modList.get(objNum);
				if (id == null) {
					if (subSec.getXRefUsed(index))
						id = new CosObjectID(objNum, subSec.getXRefGeneration(index));
					else
						id = new CosObjectID(objNum, -1);
					modList.add(objNum, id);
				}
			}
		}
	}

	private long getXRefSectionPos(XRefSubSection sec)
		throws PDFCosParseException, IOException, PDFSecurityException
	{
		if (sec instanceof TableXRefSubSection)
			return ((TableXRefSubSection)sec).mXRefPos;
		CosStream xrefStm = ((StreamXRefSubSection)sec).mXStm;
		CosObjectInfo info = xrefStm.getInfo();
		if (!info.isFree() && !(info.isCompressed()))
			return info.getPos();
		return 0;
	}

	/**
	 * Filter trash entries out of the EOF list. Linear save produces a trash
	 * entry because it has to sections, but both of them logically belong to
	 * revision zero. Hybrid update sections can also produce a trash entry
	 * which is a duplicate, even though it all belongs to the same revision.
	 */
	private void setRevisions(ArrayList eofList)
	{
		for (int i = 0; i < eofList.size() - 1; i++) {
			if (((Long)eofList.get(i)).longValue() <= ((Long)eofList.get(i + 1)).longValue())
				eofList.remove(i);
		}
		if (!eofList.isEmpty()) {
			int size = eofList.size();
			mRevisions = new long[size];
			for (int i = 0; i < size; i++)
				mRevisions[i] = ((Long)eofList.get(size - i - 1)).longValue();
		} else {
			mRevisions = null;
		}
	}

	/**
	 * Parses the PDF trailer dictionary.
	 *
	 * @param b First byte of the "trailer"
	 *
	 * @return COSDictionary
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	private CosDictionary readTrailer(byte b)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		// TODO - replace with generic parsing code
		CosParseBuf pBuf = new CosParseBuf(mBuf, 128);
		if (!(b == (byte) 't'
		      && pBuf.read() == (byte) 'r'
		      && pBuf.read() == (byte) 'a'
		      && pBuf.read() == (byte) 'i'
		      && pBuf.read() == (byte) 'l'
		      && pBuf.read() == (byte) 'e'
		      && pBuf.read() == (byte) 'r')) {
			throw new PDFCosParseException("Expected 'trailer' : " + Long.toString(pBuf.getPosition() - 1));
		}
		CosObject trailerDict = CosToken.readObject(mDoc, pBuf, null);
		pBuf.close();
		if (!(trailerDict instanceof CosDictionary))
			throw new PDFCosParseException("Invalid trailer dictionary");
		return (CosDictionary)trailerDict;
	}

	/**
	 * Returns the position of the last cross reference table. This value
	 * is stored right before the %%EOF at the end of the PDF file. Refer
	 * to section 3.4.4 of the PDF Reference Manual version 1.4 for more
	 * information.
	 *
	 * @return Location of the last cross reference table as a byte offset
	 * from the start of the file.
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	long getLastXRefSectionPosition()
		throws PDFCosParseException, IOException
	{
		// Get the starting location of the %%EOF
		long pos = getEOFPosition();

		// Move backwards to end of the number right before the %%EOF.
		byte b;
		do {
			pos--;
			mBuf.seek(pos);
			// FIXME_IO - check for EOF!
			b = (byte) mBuf.read();
		} while (!ByteOps.isDigit(b));

		// Read the number and return it.
		long rslt = 0;
		long multiplier = 1;
		do {
			rslt += ((char) b - '0') * multiplier;
			multiplier *= 10;
			pos--;
			mBuf.seek(pos);
			// FIXME_IO - check for EOF!
			b = (byte) mBuf.read();
		} while (ByteOps.isDigit(b));
		return rslt;
	}

	/**
	 * Obtains the location of the %%EOF portion of the PDF file.
	 *
	 * @return Starting location of the "%%EOF" string as a byte offset
	 * from the start of the file.
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private long getEOFPosition()
		throws PDFCosParseException, IOException
	{
		long length = mBuf.length();
		final int bufferSize = length >=EOF_CHECK_THRESHOLD ? EOF_CHECK_THRESHOLD : (int)length;
		int curPos = 0;
		int bufPos = bufferSize;
		long garbageLength = 0;
		byte[] buffer = new byte[bufferSize];
		mBuf.seek(length - bufferSize);
		mBuf.read(buffer, 0, bufferSize);
		bufPos = bufferSize;
		while(bufPos >= 4){
			curPos = bufPos;
			if(buffer[--bufPos] == 'F' && buffer[--bufPos] == 'O' && buffer[--bufPos] == 'E' &&
					buffer[--bufPos] == '%' && buffer[--bufPos] == '%'){
				mDoc.setGarbageLength(garbageLength);
				return length - (bufferSize - bufPos);
			}
			if(garbageLength == 0 && (curPos - bufPos) == 1 && ByteOps.isWhitespace(buffer[bufPos]))
				continue;
			garbageLength +=curPos - bufPos;
		}
		throw new PDFCosParseException("could not find %%EOF");
	}

	/**
	 * @throws PDFSecurityException
	 */
	void setupTrailerEncryption()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (mTrailerList != null) {
			for (int i = 0; i < mTrailerList.size(); i++) {
				CosDictionary trailer = (CosDictionary)mTrailerList.get(i);
				if (trailer != null) {
					boolean oldDecrypt = mDoc.getEncryption().setEncryptionState(false);
					if (trailer.containsKey(ASName.k_ID)) {
						CosArray idArray = trailer.getCosArray(ASName.k_ID);
						idArray.setEncryptionState(false);
					}
					if (trailer.containsKey(ASName.k_Encrypt)) {
						CosDictionary encryptDict = trailer.getCosDictionary(ASName.k_Encrypt);
						encryptDict.setEncryptionState(false);
					}
					mDoc.getEncryption().setEncryptionState(oldDecrypt);
				}
			}
		}
	}

	/**
	 * Parses a positive integer from the PDF byte stream.
	 *
	 * @param buf Buffer to parse
	 * @param b First byte of the integer
	 *
	 * @return Parse positive integer
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static long readInt(InputByteStream buf, byte b)
		throws PDFCosParseException, IOException
	{
		int sign = 1;
		if (b == '+') {
			sign = 1;
			// FIXME_IO - check for EOF!
			b = (byte) buf.read();
		}
		if (!ByteOps.isDigit(b))
			throw new PDFCosParseException("Expected digit : " + Long.toString(buf.getPosition() - 1));
		long rslt = 0;
		while (ByteOps.isDigit(b)) {
			rslt *= 10;
			rslt += b - '0';
			// FIXME_IO - check for EOF!
			b = (byte) buf.read();
		}
		rslt *= sign;
		buf.unget();
		return rslt;
	}

	int getType()
	{
		return mXrefType;
	}

	/**
	 * @author gish
	 * If this XRef belongs to a new CosDocument (not from an input stream) it
	 * cannot be used as an index to retrieve indirect objects.
	 * @return true if there are no XRef entries to search
	 */
	boolean isNew()
	{
		boolean result = false;
		if (mXRefSubSections.isEmpty())
			result = true;
		return result;
	}
	
	/**
	 * @return Number of objects defined in cross reference tables
	 */
	 int getNumObjectsDefinedInXRefEntries()
	 {
	  	return highestObjectNumInXRefEntries + 1;
	 }
	 
	 boolean isXrefIntialized() {
		return isXrefIntialized;
	}

	/**
	 * Base class to handle both classical and stream xref table subsections
	 */
	private abstract class XRefSubSection
	{
		int mBegin, mEnd;

		/**
		 * Close any resources used by this XRefSubSection.
		 *
		 * @throws IOException
		 */
		abstract void close()
			throws IOException;

		/**
		 * Obtains the starting object ID number for the subsection.
		 *
		 * @return Starting ID number.
		 */
		final int getBegin()
		{
			return mBegin;
		}

		/**
		 * Obtains the final object ID number for the subsection.
		 *
		 * @return Ending ID number.
		 */
		final int getEnd()
		{
			return mEnd;
		}

		/**
		 * Obtains a CosObjectInfo that can be used to obtain the bytes of
		 * the given object so it can be parsed.
		 * @param index Zero-based index specifying the subsection
		 * entry whose data is desired.
		 * @param cosObjectInfo The info of the object expected at index. It is
		 * used for consistency checking when parsing
		 * @return A source for the bytes of the object, either in the
		 * main file or in a compressed object stream
		 * @throws PDFCosParseException
		 * @throws IOException
		 * @throws PDFSecurityException
		 */
		abstract CosObjectInfo getInfo(int index, CosObjectInfo cosObjectInfo)
			throws PDFCosParseException, IOException, PDFSecurityException;

		/**
		 * Obtains the generation number from the specified entry.
		 *
		 * @param index	Zero-based index specifying the subsection
		 * entry whose generation field is desired.
		 *
		 * @return Generation number from the specified entry.
		 * @throws PDFCosParseException
		 * @throws IOException
		 * @throws PDFSecurityException
		 */
		abstract int getXRefGeneration(int index)
			throws PDFCosParseException, IOException, PDFSecurityException;

		/**
		 * Indicates whether the specified entry's object is in use.
		 *
		 * @param index	Zero-based index specifying the subsection entry.
		 *
		 * @return true if the object is in use.
		 * @throws PDFCosParseException
		 * @throws IOException
		 * @throws PDFSecurityException
		 */
		abstract boolean getXRefUsed(int index)
			throws PDFCosParseException, IOException, PDFSecurityException;
	}

	/**
	 * Represents a cross reference table subsection. Refer to section 3.4.3
	 * of the PDF Reference Manual for information about the cross reference
	 * table subsection.
	 */
	private class TableXRefSubSection extends XRefSubSection {
		// Every cross reference entry is 20 bytes.
		private static final int ENTRY_SIZE = 20;

		// The first portion of the entry is the object offset.
		private static final int ENTRY_POSITION_OFFSET = 0;

		// The second portion of the entry starts at byte 11
		// and it is the object generation number.
		private static final int ENTRY_GENERATION_OFFSET = 11;

		// The third portion of the entry starts at byte 17
		// and it is the in-use indicator character.
		private static final int ENTRY_IN_USE_OFFSET = 17;

		private long mXRefPos;		// Starting position of first entry

		// For optimization
		private long mCurIndex = -1;
		private byte[] mCurEntry = new byte[ENTRY_SIZE];

		/**
		 * Constructs a cross reference subsection.
		 *
		 * @param b First byte of the subsection
		 * @throws PDFCosParseException
		 * @throws IOException
		 */
		TableXRefSubSection(byte b)
			throws PDFCosParseException, IOException
		{
			// Parse the starting object ID number.
			mBegin = (int)readInt(mBuf, b);
			b = CosToken.skipWhitespace(mBuf);

			// Parse the number of objects in the subsection and add it
			// to the beginning object ID to get the ending object ID.
			mEnd = mBegin + (int)readInt(mBuf, b);
			CosToken.skipWhitespace(mBuf);

			highestObjectNumInXRefEntries = highestObjectNumInXRefEntries > mEnd ? highestObjectNumInXRefEntries : mEnd;
			
			// Record the starting position of the first entry in
			// the cross reference table.
			mXRefPos = mBuf.getPosition() - 1;

			// Set the stream to the end of the subsection
			mBuf.seek(mXRefPos + ENTRY_SIZE * (mEnd - mBegin));
		}

		@Override
		CosObjectInfo getInfo(int index, CosObjectInfo info)
			throws PDFCosParseException, IOException
		{
			if (info != null && info.isAssigned()) {
				if (getXRefUsed(index)) {
					if (index != mCurIndex) {
						mBuf.seek(mXRefPos + ((long)ENTRY_SIZE) * index + ENTRY_POSITION_OFFSET);
						mBuf.read(mCurEntry, 0, ENTRY_SIZE);
						byte b = mCurEntry[ENTRY_IN_USE_OFFSET];
						if (b != 'n' && b != 'f')
							throw new PDFCosParseException("Bad xref table entry");
						mCurIndex = index;
					}
					long pos = 0;
					for (int i = ENTRY_POSITION_OFFSET; i < ENTRY_GENERATION_OFFSET - 1; i++) {
						pos *= 10;
						pos += mCurEntry[i] - '0';
					}
					info.setPos(pos);
					info.markAddressed();
				} else {
					info.markFree();
				}
			}
			return info;
		}

		/**
		 * Obtains the generation number from the specified entry.
		 * @param index	Zero-based index specifying the subsection
		 * entry whose generation field is desired.
		 * @return Generation number from the specified entry.
		 * @throws PDFCosParseException
		 * @throws IOException
		 */
		@Override
		final int getXRefGeneration(int index)
			throws PDFCosParseException, IOException
		{
			/* get our own copy of the stream position so we can be
			 * reentrant
			 */
			if (index != mCurIndex) {
				mBuf.seek(mXRefPos + ((long)ENTRY_SIZE) * index + ENTRY_POSITION_OFFSET);
				mBuf.read(mCurEntry, 0, ENTRY_SIZE);
				byte b = mCurEntry[ENTRY_IN_USE_OFFSET];
				if (b != 'n' && b != 'f')
					throw new PDFCosParseException("Bad xref table entry");
				mCurIndex = index;
			}
			int generation = 0;
			for (int i = ENTRY_GENERATION_OFFSET; i < ENTRY_IN_USE_OFFSET - 1; i++) {
				generation *= 10;
				generation += mCurEntry[i] - '0';
			}
			return generation;
		}

		/**
		 * Indicates whether the specified entry's object is in use.
		 * @param index	Zero-based index specifying the subsection entry.
		 * @return true if the object is in use.
		 * @throws IOException
		 */
		@Override
		final boolean getXRefUsed(int index)
			throws PDFCosParseException, IOException
		{
			/* get our own copy of the stream position so we can be
			 * reentrant
			 */
			if (index != mCurIndex) {
				mBuf.seek(mXRefPos + ((long)ENTRY_SIZE) * index + ENTRY_POSITION_OFFSET);
				mBuf.read(mCurEntry, 0, ENTRY_SIZE);
				byte b = mCurEntry[ENTRY_IN_USE_OFFSET];
				if (b != 'n' && b != 'f')
					throw new PDFCosParseException("Bad xref table entry");
				mCurIndex = index;
			}
			return mCurEntry[ENTRY_IN_USE_OFFSET] == 'n';
		}

		/**
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString()
		{
			return "TableXRefSubSection (base=" + mBegin + " end=" + mEnd + ")";
		}

		@Override
		void close() throws IOException {
			//NOP			
		}
	}

	/**
	 * Xref subsections obtained from an Xref stream
	 *
	 */
	private class StreamXRefSubSection extends XRefSubSection {
		/*
		 * The position of the data for the entry for mBegin in the stream data
		 * contained in mStreamData.
		 */
		long mXrefPos;

		/*
		 * Parsed values of the /W array in the dictionary part of the xref stream
		 */
		int[] mW;

		/*
		 * The sum of the numbers in mW, the size of an entry
		 */
		int mTotalW;

		/*
		 * The sum of the numbers in mW fields 0 and 1, an optimization
		 */
		int mW0plus1;

		/*
		 * This subsection's CosStream object
		 */
		CosStream mXStm;

		/*
		 * This subsection's portion of the stream data
		 */
		byte[] mXRefData;

		StreamXRefSubSection(int first, int count, CosStream cosStm, InputByteStream stm, long streamOffset, int[] w, int totalW)
			throws PDFCosParseException, IOException, PDFSecurityException
		{
			mBegin = first;
			mEnd = first + count;
			highestObjectNumInXRefEntries = highestObjectNumInXRefEntries > mEnd ? highestObjectNumInXRefEntries : mEnd;    
			mXrefPos = streamOffset;
			mW = w;
			mTotalW = totalW;
			mW0plus1 = w[0] + w[1];
			mXStm = cosStm;
			mXRefData = new byte[count * totalW];
			stm.seek(mXrefPos);
			stm.read(mXRefData, 0, count * totalW);
		}

		@Override
		void close()
		{
			mXRefData = null;
		}

		@Override
		final int getXRefGeneration(int index)
			throws PDFCosParseException, IOException, PDFSecurityException
		{
			int result = 0;
			if (mW[2] != 0) {
				int entryBase = index * mTotalW;
				if (getXrefEntryType(index) == 1) {
					for (int i = entryBase + mW0plus1; i < entryBase + mTotalW; i++) {
						result <<= 8;
						result |= (mXRefData[i] & 0xFF);
					}
				}
			}
			return result;
		}

		@Override
		CosObjectInfo getInfo(int index, CosObjectInfo info)
			throws PDFCosParseException, IOException, PDFSecurityException
		{
			if (info != null && info.isAssigned()) {
				int entryBase = index * mTotalW;
				int type = getXrefEntryType(index) ;
				if (type == 0) {
					info.markFree();
				} else if (type == 1) {
					/*
					 * An object in the main data stream
					 */
					long position = 0;
					for (int i = entryBase + mW[0]; i < entryBase + mW0plus1; i++) {
						position <<= 8;
						position |= (mXRefData[i] & 0xFF);
					}
					info.setPos(position);
					info.markAddressed();
				} else if (type == 2) {
					/*
					 * An object in a compressed stream
					 */
					int stmNumber = 0;
					for (int i = entryBase + mW[0]; i < entryBase + mW0plus1; i++) {
						stmNumber <<= 8;
						stmNumber |= (mXRefData[i] & 0xFF);
					}
					int objNdx = 0;
					for (int i = entryBase + mW0plus1; i < entryBase + mTotalW; i++) {
						objNdx <<= 8;
						objNdx |= (mXRefData[i] & 0xFF);
					}
					CosObjectInfo stmInfo = mDoc.getObjectInfo(stmNumber, 0);
					info.setStreamInfo(stmInfo);
					info.setStreamNdx(objNdx);
					info.markAddressed();
				}
				else {
					throw new PDFCosParseException("Undefined XRef stream entry type");
				}
			}
			return info;
		}

		/**
		 * Returns the type of entry at specified index.
		 * According to PDF reference Xref stream data contains multiple XREF entries. Each Xref entry
		 * contains same number of integers which is equal to the length of W in XREF dict. Now each size of these integers
		 * is represented by the corresponding value in the W array. The first integer in each entry shall
		 * always represent the type of entry which can only be 0,1 or 2 according to pdf reference. So we ideally need a single
		 * byte to represent the type but that's not enforced by the reference So 'Type' integer shall have W[0] number of bytes and
		 * if it's not 0,1, or 2 then should point to null object.
		 * @param index
		 * @return int
		 */
		private int getXrefEntryType(int index){
			if(mW[0] == 0)
				return 1;// default type is 1
			int entryBase = index * mTotalW;
			int type = 0;
			for (int i = entryBase; i < entryBase + mW[0]; i++) {
				type <<= 8;
				type |= (mXRefData[i] & 0xFF);
			}
			return type;
		}
		@Override
		final boolean getXRefUsed(int index)
			throws PDFCosParseException, PDFSecurityException
		{
			return (getXrefEntryType(index)!= 0);
		}
	}
}
