/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.FindBugsSuppress;


/**
 * An implementation of the NodeList interface that uses an ArrayList as its
 * storage.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class ArrayNodeList extends NodeList {
	private final List<Node> mArray;

	/**
	 * default constructor
	 */
	public ArrayNodeList() {
		mArray = new ArrayList<Node>();
	}

	/**
	 * Create a new ArrayNodeList from an existing one.
	 * 
	 * @param list
	 *            to clone
	 */
	public ArrayNodeList(List<? extends Node> list) {
		mArray = new ArrayList<Node>(list);
	}

	/**
	 * Create an ArrayNodeList from the children of an Element. All children of
	 * the element will be added to the list.
	 * 
	 * @param e
	 *            The Element to grab children from.
	 */
	public ArrayNodeList(Element e) {
		mArray = new ArrayList<Node>();
		Node child = e.getFirstXFAChild();
		while (child != null) {
			mArray.add(child);
			child = child.getNextXFASibling();
		}
	}

	/**
	 * @see ListBase#append(Obj)
	 */
	public void append(Obj newNode) {
		if (isReadOnly() || newNode == null)
			return;
		
		if (!(newNode instanceof Node))
			throw new ExFull(ResId.ArgumentMismatchException);
		
		mArray.add((Node)newNode);
	}

	public Object clone() {
		return new ArrayNodeList(mArray);
	}

	/**
	 * @see Obj#getClassAtom()
	 */
	public String getClassAtom() {
		return STRS.NODELIST;
	}

	/**
	 * @see Obj#getClassName()
	 */
	public String getClassName() {
		return STRS.NODELIST;
	}

	public Node getNamedItem(String name) {
		if (name == null)
			return null;

		String aName = name.intern();

		int n = mArray.size();

		for (int i = 0; i < n; i++) {
			Node child = mArray.get(i);
			if (aName == child.getName()) {
				return child;
			}
		}
		return null;
	}
	
	/**
	 * @see NodeList#getNamedItem(String, String, int)
	 */
	@FindBugsSuppress(code="ES")
	public Node getNamedItem(String aName, String aClassName, int nTargetOccurrence/*0*/) {
	    if (aName == null && aClassName == null)
	        return null;

	    int nLength = length();
	    int nOccurrence = 0;

	    if (nTargetOccurrence > nLength)
	        return null;

	    for (int i = 0; i < nLength; i++) {
	        boolean bMatched = false;
	        Node tmp = (Node)item(i);
	        if (aName != null && aClassName != null) {
	            if (aClassName == tmp.getClassAtom() && aName == tmp.getName())
	                bMatched = true;
	        }
	        else if ( aClassName != null && aClassName == tmp.getClassAtom() )
	            bMatched = true;
	        else if (aName != null && aName == tmp.getName())
	            bMatched = true;

	        if (bMatched)
	        {
	            if (nOccurrence == nTargetOccurrence)
	                return tmp;
	            else
	                nOccurrence++;
	        }
	    }
	    return null;
	}
	
	/**
	 * @see NodeList#getOccurrence(Node)
	 */
	@FindBugsSuppress(code="ES")
	Integer getOccurrence(Node oNode) {
	    int nLength = length();
	    int nOccurrence = 0;
	    
	    String aName = oNode.getName();
	    String aClassName = oNode.getClassAtom();

	    for (int i = 0; i < nLength; i++) {
	        Node oTemp = mArray.get(i);

	        if (aClassName == oTemp.getClassAtom() && aName == oTemp.getName()) {
	            if (oNode == oTemp)
	                return Integer.valueOf(nOccurrence);
	            nOccurrence++;
	        }
	    }
		return null;
	}

	/**
	 * @see ListBase#insert(Obj, Obj)
	 */
	public void insert(Obj newNode, Obj refNode) {
		if (isReadOnly() || newNode == null)
			return;
		
		if (refNode == null)
			throw new ExFull(ResId.InsertFailedException);
		
		if (!(newNode instanceof Node))
			throw new ExFull(ResId.ArgumentMismatchException);
		
		if (!(refNode instanceof Node))
			throw new ExFull(ResId.ArgumentMismatchException);

		for (int i = 0; i < mArray.size(); i++) {
			if (mArray.get(i) == refNode) {
				mArray.add(i, (Node)newNode);
			}
		}
	}

	/**
	 * @see ListBase#item(int)
	 */
	public Obj item(int index) {
		return mArray.get(index);
	}

	/**
	 * @see ListBase#length()
	 */
	public int length() {
		return mArray.size();
	}

	/**
	 * @see ListBase#remove(Obj)
	 */
	public void remove(Obj removeNode) {
		if (isReadOnly() || removeNode == null)
			return;
		
		if (!(removeNode instanceof Node))
			throw new ExFull(ResId.ArgumentMismatchException);

		mArray.remove(removeNode);
	}
	
	/* @exclude from published api. */
	boolean appendPermsCheck() {
		return true;
	}
	
	/* @exclude from published api. */
	boolean removePermsCheck(Obj obj) {
		return true;
	}
	
	/* @exclude from published api. */
	boolean insertPermsCheck() {
		return true;
	}
}
