/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa;

import com.adobe.xfa.ut.StringUtils;

/**
 * An immutable class to represent any attribute defined by a simple string.
 */
public final class StringAttr extends Attribute {
	private final String mNS;

	/**
	 * Instantiates an attribute with the given attribute parameters.
	 * @param NS the namespace.
	 * @param localName the local name.
	 * @param qName the qualified name.
	 * @param value the attribute value.
	 */
	public StringAttr(String NS, String localName, String qName, String value) {
		this(NS, localName, qName, value, true);
	}

	/**
	 * Instantiates an attribute with the given value.
	 * @param qName the attribute name.
	 * @param value the attribute value.
	 */
	public StringAttr(String qName, String value) {
		super(null, null, qName, value);
		mNS = null;
	}

	/**
	 * Create a new StringAttr, given all attribute parameters
	 * @param NS the namespace for this attribute
	 * @param localName the local name for this attribute
	 * @param qName the qualified name for this attribute 
	 * @param value the string to use to create the new attribute
	 * @param internSymbols indicates whether the symbols in other parameters need to be interned.
	 *
	 * @exclude from published api.
	 */
	public StringAttr(String NS, String localName, String qName, String value, boolean internSymbols) {
		super(NS, localName, qName, value, internSymbols);

		if (internSymbols && NS != null) {
			NS = NS.intern();
		}
		
		mNS = NS;
	}

	/**
	 * @see Attribute#newAttribute(String)
	 *
	 * @exclude from published api.
	 */
	public Attribute newAttribute(String value) {
		return newAttribute(getNS(), getName(), getQName(), value, false);
	}

	/**
	 * @see Attribute#newAttribute(String, String, String, String, boolean)
	 *
	 * @exclude from published api.
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value, boolean internSymbols) {
		
		boolean bIsEmpty = StringUtils.isEmpty(value);
		
		if (bIsEmpty)
			value = getAttrValue();
		
		return new StringAttr(NS, localName, qName, value, internSymbols);
	}
	
	/**
	 * Gets this attribute's namespace.
	 * @return the namespace.
	 *
	 * @exclude from published api.
	 */
	public String getNS() {
		return mNS;
	}

	
	/**
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 *
	 * @exclude from published api.
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value) {
		return newAttribute(NS, localName, qName, value, true);
	}
}
