//----------------------------------------------------------------------
//
//	ADOBE CONFIDENTIAL
//	__________________
//
//		Copyright 1995 - 2004 Adobe Systems Incorporated.  All
//		Rights Reserved.
//
//		NOTICE:  All information contained herein is, and remains
//		the property of Adobe Systems Incorporated and its
//		suppliers, if any.	The intellectual and technical
//		concepts contained herein are proprietary to Adobe Systems
//		Incorporated and its suppliers and may be covered by U.S.
//		and Foreign Patents, patents in process, and are protected
//		by trade secret or copyright law.  Dissemination of this
//		information or reproduction of this material is strictly
//		forbidden unless prior written permission is obtained from
//		Adobe Systems Incorporated.
//
//----------------------------------------------------------------------

package com.adobe.xfa.gfx;

/**
 * The graphic attribute object is a base class for different
 * display attributes and holds information about common attibute
 * values such as background and foreground colour, shading, style
 * etc.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public class GFXAttr {
/**
 * enumeration StyleCode: Specifies the style of the attributes.
 * <p> 
 * STYLE_UNKNOWN to STYLE_DIAG_CROSS inclusive is valid for line, fill
 * and text attributes. STYLE_DOT, STYLE_DASH, STYLE_DOT_DASH and
 * STYLE_DOT_DOT_DASH are only valid style values for line attributes
 */
	public final static int STYLE_UNKNOWN = 0;
	public final static int STYLE_NONE = 1;
	public final static int STYLE_SOLID = 2;
	public final static int STYLE_HORZ = 3;
	public final static int STYLE_VERT = 4;
	public final static int STYLE_CROSS = 5;
	public final static int STYLE_DIAG_LEFT = 6;
	public final static int STYLE_DIAG_RIGHT = 7;
	public final static int STYLE_DIAG_CROSS = 8;
	public final static int STYLE_DOT = 9;
	public final static int STYLE_DASH = 10;
	public final static int STYLE_DOT_DASH = 11;
	public final static int STYLE_DOT_DOT_DASH = 12;

	public final static int DEFAULT_SHADESCALE = 100;

	private int mnStyle; // constructor initialization // style code.
	private int mnShadeScale; // upper bound for shading.
	private int mnShade; // shading value from 0 (invisible) to

	private GFXColour mpoColour; // foreground colour.
	private GFXColour mpoColourBg; // background colour.

	private GFXGraphicContext mpoContext; // graphic context

/**
 * Default constructor.
 * <p>
 * Populates the graphic attribute with the following settings:
 * <pre>
 *	 Style			   = solid
 *	 Shading scale	   = full shading
 *	 Shade			   = solid
 *	 Foreground colour = black
 *	 Background colour = white
 * </pre>
 */
	public GFXAttr () {
		initialize();
	}

/**
 * Copy consturctor.
 * <p>
 * Copies all attribute values.
 * @param oSource Source attribute object to copy
 */
	public GFXAttr (GFXAttr oSource) {
		initialize();
		copyFrom (oSource);
	}

/**
 * Constructor.
 * <p>
 * Creates an Attribute object with the settings as specified
 * by the input values.
 * @param nNewStyle Specifies the style of the attribute
 * @param lNewShade Specifies the shade of the attribute
 * @param oNewColour Specifies the foreground colour
 * @param oNewColourBg Specifies the background colour
 */
	public GFXAttr (int nNewStyle, int lNewShade, GFXColour oNewColour, GFXColour oNewColourBg) {
		mnStyle = nNewStyle;
		mnShadeScale = DEFAULT_SHADESCALE;
		shade (lNewShade);
		colour (oNewColour);
		colourBg (oNewColourBg);
	}

/**
 * Get the foreground colour for this attribute.
 * @return The foreground colour as a <code>jfGfxColour</code> object
 */
	public GFXColour colour () {
		return mpoColour;
	}

/**
 * Set the foreground colour for this attribute.
 * @param oNewColour The new foreground colour as a
 * <code>jfGfxColour</code> object
 */
	public void colour (GFXColour oNewColour) {
		int fg = GFXColour.getStandardColourIndex (oNewColour);
		if (fg == GFXColour.OTHER_INDEX) {
			mpoColour = oNewColour;
		} else {
			mpoColour = GFXColour.getStandardColour (fg);
		}
	}

/**
 * Get the background colour for this attribute.
 * @return The background colour as a <code>jfGfxColour</code> object
 */
	public GFXColour colourBg () {
		return mpoColourBg;
	}

/**
 * Set the background colour for this attribute.
 * @param oNewColourBg The new background colour as a
 * <code>jfGfxColour</code> object
 */
	public void colourBg (GFXColour oNewColourBg) {
		int bg = GFXColour.getStandardColourIndex (oNewColourBg);
		if (bg == GFXColour.OTHER_INDEX) {
			mpoColourBg = oNewColourBg;
		} else {
			mpoColourBg = GFXColour.getStandardColour (bg);
		}
	}

/**
 * Get the shading value for this attribute. This is a value
 * between 0 (invisible) to the value of ShadeScale (solid).
 * @return The shading value
 */
	public int shade () {
		return mnShade;
	}

/**
 * Set the shading value for this attribute. This is a value
 * between 0 (invisible) to the value of ShadeScale (solid).
 * @param lNewShade The new shading value
 */
	public void shade (int lNewShade) {
		if (lNewShade < 0) {
			mnShade = 0;
		} else if (lNewShade > mnShadeScale) {
			mnShade = mnShadeScale;
		} else {
			mnShade = lNewShade;
		}
	}

/**
 * Get the upper bound for shading.
 * @return The upper bound for shading.
 */
	public int shadeScale () {
		return mnShadeScale;
	}

/**
 * Set the upper bound for shading.
 * @param lNewScale The new upper bound for shading.
 */
	public void shadeScale (int lNewScale) {
		double dOldShadeScale = mnShadeScale;
		double dShade = mnShade;
		double dFactor;

		if (lNewScale > 0) {
			mnShadeScale = lNewScale;
			dFactor = dShade / dOldShadeScale;
			mnShade = (int) Math.round (dFactor * mnShadeScale);
		}
	}

/**
 * Get the shading colour.
 * @return The shading colour as a <code>jfGfxColour</code> object
 */
	public GFXColour shadeColour () {
		double dShade = (double) mnShade / (double) mnShadeScale;
		return GFXColour.weightedAverage (colourBg(), colour(), dShade);
	}

/**
 * Get the default upper bound for shading
 * @return The default upper bound for shading.
 */
	public static int defaultShadeScale () {
		return DEFAULT_SHADESCALE;
	}

/**
 * Get the style for this attribute
 * @return The style as a StyleCode enumeration
 */
	public int style () {
		return mnStyle;
	}

/**
 * Set the style for this attribute
 * @param nNewStyle The new style as a StyleCode enumeration
 */
	public void style (int nNewStyle) {
		mnStyle = nNewStyle;
	}

/**
 * Return the graphic context associated with this graphic attribute
 * object.
 * @return Pointer to the graphic context.	A null pointer is returned
 * if the context has never been set.
 */
	public GFXGraphicContext graphicContext () {
		return mpoContext;
	}

/**
 * Set a new graphic context for this attribute object.
 * @param poGraphicContext - New graphic context to associate with this graphic
 * attribute object.
 */
	public void graphicContext (GFXGraphicContext poGraphicContext) {
		mpoContext = poGraphicContext;
	}

/**
 * Equality comparison.
 * <p>
 * Compares on an attribute value basis. Two attributes are
 * considered equal if their values compare for equality.
 * @param oCompare Attribute object to compare against.
 * @return TRUE if all members are equal, FALSE otherwise.
 */
	public boolean equivalent (GFXAttr oCompare) {
		return (mnStyle == oCompare.mnStyle)
			&& (mnShadeScale == oCompare.mnShadeScale)
			&& (mnShade == oCompare.mnShade)
			&& GFXGraphicContext.match (mpoContext, oCompare.mpoContext)
			&& mpoColour.equals (oCompare.mpoColour)
			&& mpoColourBg.equals (oCompare.mpoColourBg);
	}

/**
 * Equality comparison operator.
 * <p>
 * Compares on an attribute value basis. Two attributes are
 * considered equal if their values compare for equality.
 * @param object Attribute object to compare against.
 * @return TRUE if the attribute objects are considered equal;
 * FALSE otherwise.
 */
	public boolean equals (Object object) {
		
		if (this == object)
			return true;
		
		// This overrides Object.equals(boolean) directly, so...
		if (object == null)
			return false;
		
		if (object.getClass() != getClass())
			return false;
		
		return equivalent ((GFXAttr) object);
	}

	public int hashCode() {
		int hash = 73;
		hash = (hash * 31) ^ mnStyle;
		hash = (hash * 31) ^ mnShadeScale;
		hash = (hash * 31) ^ mnShade;
		hash = (hash * 31) ^ mpoColour.hashCode();
		hash = (hash * 31) ^ mpoColourBg.hashCode();
		hash = (hash * 31) ^ mpoContext.hashCode();
		return hash;
	}

/**
 * Replace all attributes with those from the source object.
 * <p>
 * The standard assignment copies everything, including enabled and
 * disabled status.  Graphic source information is also copied.
 * @param oSource Source attribute object to copy.
 */
	public void copyFrom (GFXAttr oSource) {
		if (this != oSource) {
			mnStyle = oSource.mnStyle;
			mnShadeScale = oSource.mnShadeScale;
			mnShade = oSource.mnShade;
			mpoColour = oSource.mpoColour;
			mpoColourBg = oSource.mpoColourBg;
			mpoContext = oSource.mpoContext;
		}
	}

	private void initialize () {
		mnStyle = STYLE_SOLID;
		mnShadeScale = DEFAULT_SHADESCALE;
		mnShade = DEFAULT_SHADESCALE;
		mpoColour = GFXColour.BLACK;
		mpoColourBg = GFXColour.WHITE;
		mpoContext = null;
	}
}
