/*************************************************************************
 *
 *	File: PDF417Encoder.java
 *
 **************************************************************************
 * 
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.xfa.pmp.adobepdf417pmp;

import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.pmp.common.BarcodeEncoder;
import com.adobe.xfa.pmp.common.BarcodeGenerationParams;
import com.adobe.xfa.pmp.common.IntegerHolder;

/**
 * Ported from PDF417Encoder.cpp
 */
public class PDF417Encoder implements BarcodeEncoder
{
	
	
//////////////////////////////////////////////////////////////////////
	/**
	    Encode the message into a PDF417 symbol.

	    param message - The byte message to be placed in the symbol.
	    param parameters - The encoding parameters.
	    param numberOfParameters - The number of encoding parameters.

	    The encoding parameters are:

	    Resolution
	        Type: int32.
	        Required: yes.
	        Range: must be greater than 0.
	        Description:
	           The DPI resolution.  This is used along with the 
	           XSymWidth and XSymHeight parameters to calculate
	           the size of the output image in pixels.

	    ECC
	        Type: int32
	        Required: no.
	        Default: 4.
	        Range: 0 - 8.
	        Description:
	            The ECC level to use in the barcode.

	    Width
	        Type: double.
	        Required: yes.
	        Range: must be greater than 0.0.
	        Description:
	            The width of the barcode image in inches.

	    Height
	        Type: double.
	        Required: yes.
	        Range: must be greater than 0.0.
	        Description:
	            The height of the barcode image in inches.

	    XSymWidth
	        Type: int.
	        Required: no.
	        Default: 4.
	        Range: must be greater than 0.
	        Description:
	           The x-dimension of the barcode in pixels.

	    XSymHeight
	        Type: int.
	        Required: no.
	        Default: 8.
	        Range: must be greater than 0.
	        Description:
	           The y-dimension of the barcode in pixels.

	    nCodeWordCol
	        Type: int.
	        Required: yes/no.
	        Range: must be greater than 0.
	        Description:
	            The number of data columns in the barcode.

	    nCodeWordRow
	        Type: int.
	        Required: yes/no.
	        Range: must be greater than 0.
	        Description:
	            The number of data rows in the barcode.
	 * @throws PDF417EncoderException 

	*/
	public //////////////////////////////////////////////////////////////////////

	BufferedImage encode(
	    char[] message,    // In: The message to be encoded in the barcode.
	    BarcodeGenerationParams pmpParams) throws PDF417EncoderException        // In: barcode parameters                
	    // Out: The image containing the barcode.    
	{
	 
		

	    // Extract Resolution
		double resolution = pmpParams.getResolution();
	    if(resolution < 1)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.RESOLUTION);

	    // Extract the ECC
		int ecc = pmpParams.getEccLevel();
		if((ecc < 0) || (ecc > 8))
			throw new PDF417EncoderException(PDF417EncoderErrorCode.ECC);
	    
	    // Extract the image width
		IntegerHolder width = new IntegerHolder((int) ((pmpParams.getWidth()) * resolution + 0.5));
		

	    // Extract the image height
		IntegerHolder height = new IntegerHolder((int) (pmpParams.getHeight() * resolution + 0.5));
		

	    // Extract the module size
		int xSymWidth = (pmpParams.getXSymbolWidth());// * resolution + 0.5);
		
	    if(xSymWidth < 1)
			throw new PDF417EncoderException(PDF417EncoderErrorCode.XSYMWIDTH);

	    // Extract the module size
		int xSymHeight = (pmpParams.getXSymbolHeight());// * resolution + 0.5);
		
	    if(xSymHeight < 1)
	            throw new PDF417EncoderException(PDF417EncoderErrorCode.XSYMHEIGHT);
		
		// This is an special odd ball case to handle the default module w/h values
		// from XFA
		if ( xSymWidth == 3 && ( xSymHeight == 59 || xSymHeight == 60 ) )
			xSymHeight = 9;
	    
	    // Get the symbol X sequence.
	    IntegerHolder dataCols = new IntegerHolder(0);
	    IntegerHolder dataRows = new IntegerHolder(0);;
	    IntegerHolder actualXDimension = new IntegerHolder(0);
	    IntegerHolder actualYDimension = new IntegerHolder(0);
	    String xSequence = getXSequence2(
	        message, 
	        width, 
	        height,
	        xSymWidth, 
	        xSymHeight,
	        ecc, 
	        false,
	        dataCols,
	        dataRows,
	        actualXDimension,
	        actualYDimension);

	    // Create the image.
	    BufferedImage image = PDF417ImageBuilder.buildCentered(
	        xSequence, 
	        actualXDimension.getValue(), 
	        actualYDimension.getValue(),
	        0, 
	        0,
	        dataCols.getValue(),
	        dataRows.getValue(),
	        width.getValue(),
	        height.getValue());
	    
	    return image;
	}
	
	//////////////////////////////////////////////////////////////////////
	/**
	    Encode the message into a PDF417 symbol
	*/
	//////////////////////////////////////////////////////////////////////

	static String getXSequence2(
	    char[] message, 
	    IntegerHolder imageCols,
	    IntegerHolder imageRows,
	    int xDimension,
	    int yDimension,
	    int eccLevel,
	    boolean readerInitialization,
	    IntegerHolder dataCols,
	    IntegerHolder dataRows,
	    IntegerHolder actualXDimension,
	    IntegerHolder actualYDimension) throws PDF417EncoderException
	{
	    // Compact the data
	    List<Integer> data = new ArrayList<Integer>();
	    PDF417CompactorManager.compact(message, data);

	    // Add the Reader Initialization code, if needed.
	    if(readerInitialization)
	    {
	        // The reader initialization code word must appear after the size codeword.
	        // Add a new element to the end of the vector.
	        data.add(0);
	        for(int idx = data.size() - 1; idx > 1; idx--)
	            data.set(idx,data.get(idx - 1));
	        data.set(1,PDF417SpecialCode.INITIALIZER.getValue());
	    }

	    // Calculate the size of the symbol
	    dataCols.setValue(1);
	    dataRows.setValue(3);
	    actualXDimension.setValue(4);
	    actualYDimension.setValue(8);
	    if((imageCols.getValue() <= 0) && (imageRows.getValue() <= 0))
	        calculateRowsAndColsWidthAndHeightVariable(
	            data.size() + PDF417ReedSolomon.getNumberOfECC(eccLevel), 
	            imageCols, 
	            imageRows,
	            xDimension,
	            yDimension,
	            dataCols, 
	            dataRows,
	            actualXDimension,
	            actualYDimension);
	    else
	    if(imageCols.getValue() <= 0)
	        calculateRowsAndColsWidthVariable(
	            data.size() + PDF417ReedSolomon.getNumberOfECC(eccLevel), 
	            imageCols, 
	            imageRows.getValue(),
	            xDimension,
	            yDimension,
	            dataCols, 
	            dataRows,
	            actualXDimension,
	            actualYDimension);
	    else
	    if(imageRows.getValue() <= 0)
	        calculateRowsAndColsHeightVariable(
	            data.size() + PDF417ReedSolomon.getNumberOfECC(eccLevel), 
	            imageCols.getValue(), 
	            imageRows,
	            xDimension,
	            yDimension,
	            dataCols, 
	            dataRows,
	            actualXDimension,
	            actualYDimension);
	    else
	        calculateRowsAndCols2(
	            data.size() + PDF417ReedSolomon.getNumberOfECC(eccLevel), 
	            imageCols.getValue(), 
	            imageRows.getValue(),
	            xDimension,
	            yDimension,
	            dataCols, 
	            dataRows,
	            actualXDimension,
	            actualYDimension);

	    // Pad the Data with NULL codewords
	    padData(data, dataCols.getValue(), dataRows.getValue(), eccLevel);

	    // Get the Reed-Solomon Error Correction code words
	    List<Integer> ecc = new ArrayList<Integer>();
	    PDF417ReedSolomon.getECC(data, eccLevel, ecc);

	    // Append the ECC on the data
	    int size = ecc.size();
	    int idx;
	    for(idx = size - 1; idx >= 0; idx--)
	        data.add(ecc.get(idx));

	    //toString(*pData);

	    // Generate the X-Sequence
	    return PDF417XSequenceBuilder.build(data, dataCols.getValue(), dataRows.getValue(), eccLevel);
	}

//////////////////////////////////////////////////////////////////////
	/**
	    Calculate the symbol parameters given the image size.

	    param messageCodeWord - The total number of codewords in the 
	        message including ECC.
	    param imageCols - The desired image columns.
	    param imageRows - The desired image rows.
	    param dimX - The symbol X dimension. On input, if 1 or greater
	        the desired X dimension.  If less than 1, the best X dimension
	        will be calculated. On return, the X dimension to use for the
	        symbol.
	    param dimY - The symbol Y dimension. On input, if 1 or greater
	        the desired Y dimension.  If less than 1, the best Y dimension
	        will be calculated.  On return, the Y dimension to use for the 
	        symbol.
	    param codeWordCols - On return, the number of code word columns 
	        in the symbol.
	    param codeWordRows - On return, the number of code word rows 
	        in the symbol.
	 * @throws PDF417EncoderException 
	*/
	//////////////////////////////////////////////////////////////////////

	static void calculateRowsAndCols2(
	    int messageCodeWords,  // including ECC
	    int imageCols,
	    int imageRows,
	    int xDimension,
	    int yDimension,
	    IntegerHolder codeWordCols,
	    IntegerHolder codeWordRows,
	    IntegerHolder actualXDimension,
	    IntegerHolder actualYDimension) throws PDF417EncoderException
	{
	    // Calculate the maximum and minimum X dimension.
	    /* Find the largest x-dimension. Removed search to fix the x-dimension
	    /*
	    PMPInt32 xDimensionMax = imageCols/(17 + 17 + 17 + 17 + 18 + 4);
	    PMPInt32 xDimensionMin = imageCols/(30 * 17 + 17 + 17 + 17 + 18 + 4);
	    if(xDimensionMin < xDimension)
	        xDimensionMin = xDimension;
	    */
	    // Fix the x-dimension
	    int xDimensionMax = xDimension;
	    int xDimensionMin = xDimension - 1;

	    // Iterate thru x dimension widths, starting with the largest.
	    int idxX = 4;
	    int idxY = 8;
	    int cwCols = 0;
	    int cwRows = 0;
	    boolean found = false;
	    for(idxX = xDimensionMax; idxX > xDimensionMin; idxX--)
	    {
	        // Calculate the number of codes word in a line
	        cwCols = (int)((Math.floor( (double)(imageCols - idxX * (17 + 17 + 17 + 18 + 4))/(double)(17 * idxX) )));

	        // Must be capable of a single codeword column.
	        if(cwCols < 1)
	            continue;

	        // The maximum number of codeword columns is 30.
	        if(cwCols > 30)
	            cwCols = 30;

	        // Calculate the needed number of rows
	        int neededCodeWordRows = (int)Math.ceil((double)(messageCodeWords)/(double)(cwCols));

	        // Must have at least 3 rows
	        if(neededCodeWordRows < 3)
	            neededCodeWordRows = 3;
	        
	        // Must be less than or equal to 90 rows
	        if(neededCodeWordRows > 90)
	            continue;

	        // Calculate the maximum and minimum Y dimension
	        // compensate for quiet zone. Minumum of 3 code word rows.
	        // Assume the minimum Y dimension is 2 * idxX
	        int yDimensionMax = (int)(Math.floor( (double)(imageRows - 4 * idxX)/(double)neededCodeWordRows  ));  // needed  rows
	        int yDimensionMin = (int)(Math.floor( (imageRows - 4 * idxX)/90.0 ));  // 90 rows
	        if(yDimensionMin < yDimension)
	            yDimensionMin = yDimension;

	        // Enforce minimum Y dimension of 2 * X dimension
	        if(yDimensionMin < (2 * idxX))
	            yDimensionMin = 2 * idxX;

	        // Enforce maximum Y dimension of 20 * X dimension
	        if(yDimensionMax > (20 * idxX))
	            yDimensionMax = 20 * idxX;

	        // Iterate thru Y dimension heights, starting with the largest.
	        for(idxY = yDimensionMax; idxY >= yDimensionMin; idxY--)
	        {
	            
	            cwRows = (int)(Math.floor( (double)(imageRows - idxX * 4) / (double)(idxY) ));

	            if(cwRows > 90)
	                continue;

	            found = (cwCols * cwRows >= messageCodeWords);

	            if(found) break;
	        }
	        if(found) break;
	    }

	    if(!found)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.MESSAGE_TOO_BIG);

	    actualXDimension.setValue(idxX);
	    actualYDimension.setValue(idxY);

	    codeWordCols.setValue(cwCols);
	    codeWordRows.setValue(cwRows);

	    if(codeWordRows.getValue() < 3)
	        codeWordRows.setValue(3);

	    if(codeWordCols.getValue() < 1)
	        codeWordCols.setValue(1);

	    if(codeWordRows.getValue() > 90)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.MESSAGE_TOO_BIG);

	    if(codeWordCols.getValue() > 30)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.TOO_MANY_COLS);

	    if(codeWordCols.getValue() * codeWordRows.getValue() > 928)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.SYMBOL_TOO_BIG);

	    if(codeWordCols.getValue() * codeWordRows.getValue() < messageCodeWords)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.MESSAGE_TOO_BIG);
	}
	
	
//////////////////////////////////////////////////////////////////////
	/**
	    Calculate the symbol parameters where the image width is 
	    variable and the image height is fixed.

	    param messageCodeWord - The total number of codewords in the 
	        message including ECC.
	    param imageCols - returns the calculated image width.
	    param imageRows - The desired image rows.
	    param dimX - The symbol X dimension. On input, if 1 or greater
	        the desired X dimension.  If less than 1, the best X dimension
	        will be calculated. On return, the X dimension to use for the
	        symbol.
	    param dimY - The symbol Y dimension. On input, if 1 or greater
	        the desired Y dimension.  If less than 1, the best Y dimension
	        will be calculated.  On return, the Y dimension to use for the 
	        symbol.
	    param codeWordCols - On return, the number of code word columns 
	        in the symbol.
	    param codeWordRows - On return, the number of code word rows 
	        in the symbol.
	 * @throws PDF417EncoderException 
	*/
	//////////////////////////////////////////////////////////////////////

	static void calculateRowsAndColsWidthVariable(
	    int messageCodeWords,  // including ECC
	    IntegerHolder imageCols,
	    int imageRows,
	    int xDimension,
	    int yDimension,
	    IntegerHolder codeWordCols,
	    IntegerHolder codeWordRows,
	    IntegerHolder actualXDimension,
	    IntegerHolder actualYDimension) throws PDF417EncoderException
	{
	    if(messageCodeWords > 928)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.MESSAGE_TOO_BIG);

	    // Calculate the number of lines.
	    int cwRows = (imageRows - 4 * xDimension)/yDimension;

	    if(cwRows > messageCodeWords)
	        cwRows = messageCodeWords;

	    if(cwRows < 3)
	        cwRows = 3;

	    if(cwRows > 90)
	        cwRows = 90;

	    // Calculate the needed number of code word cols.
	    int cwCols = (int)Math.ceil((double)(messageCodeWords)/(double)(cwRows));

	    if(cwCols < 1)
	        cwCols = 1;

	    if(cwCols > 30)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.MESSAGE_TOO_BIG);

	    // Calculate the image rows
	    imageCols.setValue((17 * cwCols + 17 + 17 + 17 + 18 + 4) * xDimension);
	        
	    actualXDimension.setValue(xDimension);
	    actualYDimension.setValue(yDimension);

	    codeWordCols.setValue(cwCols);
	    codeWordRows.setValue(cwRows);

	    if(codeWordCols.getValue() * codeWordRows.getValue() > 928)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.SYMBOL_TOO_BIG);
	}

	//////////////////////////////////////////////////////////////////////
	/**
	    Calculate the symbol parameters where the image width is fixed
	    and the image height is variable.

	    param messageCodeWord - The total number of codewords in the 
	        message including ECC.
	    param imageCols - The desired image columns.
	    param imageRows - The desired image rows.
	    param dimX - The symbol X dimension. On input, if 1 or greater
	        the desired X dimension.  If less than 1, the best X dimension
	        will be calculated. On return, the X dimension to use for the
	        symbol.
	    param dimY - The symbol Y dimension. On input, if 1 or greater
	        the desired Y dimension.  If less than 1, the best Y dimension
	        will be calculated.  On return, the Y dimension to use for the 
	        symbol.
	    param codeWordCols - On return, the number of code word columns 
	        in the symbol.
	    param codeWordRows - On return, the number of code word rows 
	        in the symbol.
	 * @throws PDF417EncoderException 
	*/
	//////////////////////////////////////////////////////////////////////

	static void calculateRowsAndColsHeightVariable(
	    int messageCodeWords,  // including ECC
	    int imageCols,
	    IntegerHolder imageRows,
	    int xDimension,
	    int yDimension,
	    IntegerHolder codeWordCols,
	    IntegerHolder codeWordRows,
	    IntegerHolder actualXDimension,
	    IntegerHolder actualYDimension) throws PDF417EncoderException
	{
	    if(messageCodeWords > 928)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.MESSAGE_TOO_BIG);

	    // Calculate the number of codes word in a line
	    int cwCols = (imageCols/xDimension - 17 - 17 - 17 - 18 - 4) / 17;

	    if(cwCols > messageCodeWords)
	        cwCols = messageCodeWords;

	    // Must be capable of a single codeword column.
	    if(cwCols < 1)
	        cwCols = 1;

	    // The maximum number of codeword columns is 30.
	    if(cwCols > 30)
	        cwCols = 30;

	    // Calculate the needed number of rows
	    int cwRows = (int)Math.ceil((double)(messageCodeWords)/(double)(cwCols));

	    // Calculate the image rows
	    imageRows.setValue(cwRows * yDimension + 4 * xDimension);

	    // Must have at least 3 rows
	    if(cwRows < 3)
	        cwRows = 3;
	        
	    // Must be less than or equal to 90 rows
	    if(cwRows > 90)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.MESSAGE_TOO_BIG);

	    actualXDimension.setValue(xDimension);
	    actualYDimension.setValue(yDimension);

	    codeWordCols.setValue(cwCols);
	    codeWordRows.setValue(cwRows);

	    if(codeWordCols.getValue() * codeWordRows.getValue() > 928)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.SYMBOL_TOO_BIG);
	}

	//////////////////////////////////////////////////////////////////////
	/**
	    Calculate the symbol parameters where both the image height
	    and width are variable.  This tries to make the symbol as square
	    as possibe.

	    param messageCodeWord - The total number of codewords in the 
	        message including ECC.
	    param imageCols - The desired image columns.
	    param imageRows - The desired image rows.
	    param dimX - The symbol X dimension. On input, if 1 or greater
	        the desired X dimension.  If less than 1, the best X dimension
	        will be calculated. On return, the X dimension to use for the
	        symbol.
	    param dimY - The symbol Y dimension. On input, if 1 or greater
	        the desired Y dimension.  If less than 1, the best Y dimension
	        will be calculated.  On return, the Y dimension to use for the 
	        symbol.
	    param codeWordCols - On return, the number of code word columns 
	        in the symbol.
	    param codeWordRows - On return, the number of code word rows 
	        in the symbol.
	 * @throws PDF417EncoderException 
	*/
	//////////////////////////////////////////////////////////////////////

	static void calculateRowsAndColsWidthAndHeightVariable(
	    int messageCodeWords,  // including ECC
	    IntegerHolder imageCols,
	    IntegerHolder imageRows,
	    int xDimension,
	    int yDimension,
	    IntegerHolder codeWordCols,
	    IntegerHolder codeWordRows,
	    IntegerHolder actualXDimension,
	    IntegerHolder actualYDimension) throws PDF417EncoderException
	{
	    if(messageCodeWords > 928)
	        throw new PDF417EncoderException(PDF417EncoderErrorCode.MESSAGE_TOO_BIG);
	        
	    int cwCols;
	    int cwRows = 0;

	    cwCols = (int)Math.ceil(Math.sqrt(4.0 +
	        ((double)(messageCodeWords) * (double)(yDimension))/
	        ((xDimension) * 17.0)
	        )) - 2;

	    // The maximum number of codeword columns is 30.
	    if(cwCols > 30)
	        cwCols = 30;
	    if(cwCols < 1)
	        cwCols = 1;

	    boolean found = false;
	    while(cwCols <= 30)
	    {
	        cwRows = messageCodeWords/cwCols;
	        if(cwRows < 3)
	            cwRows = 3;
	        if(cwRows > 90)
	            cwRows = 90;

	        int cwRowsMax = cwRows + (17 * xDimension/yDimension) + 2;
	        if(cwRowsMax > 90)
	            cwRowsMax = 90;

	        while(cwRows <= cwRowsMax)
	        {
	            found = (cwCols * cwRows >= messageCodeWords);
	            if(found) break;
	            cwRows++;
	        }
	        if(found) break;
	        cwCols++;
	    }

	    if(!found)
	    {
	        // Try a maximum sized barcode.
	        cwCols = 16;
	        cwRows = 58;
	    }

	    if(codeWordCols.getValue() * codeWordRows.getValue() > 928)
	    {
	        // Try a maximum sized barcode.
	        cwCols = 16;
	        cwRows = 58;
	    }

	    actualXDimension.setValue(xDimension);
	    actualYDimension.setValue(yDimension);

	    codeWordCols.setValue(cwCols);
	    codeWordRows.setValue(cwRows);

	    imageCols.setValue((17 * cwCols + 17 + 17 + 17 + 18 + 4) * xDimension);
	    imageRows.setValue((cwRows * yDimension) + (4 * xDimension));
	}
	//////////////////////////////////////////////////////////////////////
	/**
	    Pad the data with NULL codewords.
	*/
	static //////////////////////////////////////////////////////////////////////

	void padData(
	    List<Integer> data, 
	    int dataCols, 
	    int dataRows, 
	    int eccLevel)
	{
	    int eccSize = PDF417ReedSolomon.getNumberOfECC(eccLevel);
	    int symbolSize = dataCols * dataRows;
	    int dataSize = data.size();
	    int pad = symbolSize - eccSize - dataSize;

	    for(int idx = 0; idx < pad; idx++)
	        data.add(PDF417SpecialCode.TextCompactionLatch.getValue());

	    // Set the size of the data set
	    data.set(0,data.size());
	}
}
