package com.adobe.xfa.text;


/**
 * A text field extends the text stream class (TextStream) with the
 * ability to behave as a field.  As a text stream, a text field
 * represents a container of rich text, embedded fields and other
 * objects.
 * </p>
 * <p>
 * While a text field may stand alone as a container of data in an
 * application, it does not support layout and display on its own.
 * Instead, a field must be embedded in another text stream, possibly
 * another field.
 * </p>
 * <p>
 * Layout and rendering are operations on a displayable text stream (see
 * class TextDispStr).	For a field to participate in layout, its root
 * ancestor in the text stream hierarchy must be a displayable stream.
 * </p>
 * <p>
 * The application may choose to use instances of this class directly,
 * or it may further derive its own class.	By inserting instances of
 * its own class into the parent stream, the application can cache its
 * own information with each field.
 * </p>
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public class TextField extends TextStream {
	public final static int EMBEDTYPE_SOM = 0;
	public final static int EMBEDTYPE_URI = 1;
	public final static int EMBED_RAW = 0;
	public final static int EMBED_FORMATTED = 1;

	private TextPosn moPosn; // in parent stream
	private int meEmbedMode = EMBED_FORMATTED;
	private int meEmbedType = EMBEDTYPE_SOM;
	private String msExpression = "";

/**
 * Default constructor.
 * <p>
 * The text field contains no content, has no pool/mapping
 * assocotiation and is not embedded in any other stream.
 */
	public TextField () {
	}

/**
 * Copy constructor.
 * <p>
 * Copy all content from the source field.	Also copies the graphic
 * source of the given stream.	The field is initially not embedded in
 * any stream.
 * @param oSource Source text field to copy content from.
 */
	public TextField (TextField oSource) {
		super (oSource, null);
		meEmbedMode = oSource.meEmbedMode;
		meEmbedType = oSource.meEmbedType;
		msExpression = oSource.msExpression;
	}

	public TextField (String sFieldText, int eEmbedMode, int eEmbedType) {
		super (sFieldText);
		meEmbedMode = eEmbedMode;
		meEmbedType = eEmbedType;
	}

	public TextField (String sFieldText, int eEmbedMode, int eEmbedType, String sExpression) {
		this (sFieldText, eEmbedMode, eEmbedType);
		msExpression = sExpression;
	}

	public TextField (int eEmbedMode, int eEmbedType, String sExpression) {
		meEmbedMode = eEmbedMode;
		meEmbedType = eEmbedType;
		msExpression = sExpression;
	}

/**
 * Constructor with source text string.
 * <p>
 * Create a text field whose initial content is copied from the given
 * string.	The text stream initially has no attribute pool or font
 * mapper association, and is not embedded in any stream.
 * @param sFieldText String whose contents are to be copied to the field.
 */
	public TextField (String sFieldText) {
		super (sFieldText);
	}

/**
 * Assign this field's content from the given field.
 * <p>
 * Replace this stream's content with a copy of the content of the given
 * stream.	The graphic source information is <b>not</b> copied.  In
 * other words, fonts will be re-mapped in this stream's font mapper and
 * attributes will be re-pooled in any attribute pool associated with
 * this stream.
 * @param oSource Field containing source content to copy.
 */
	public void copyFrom (TextField oSource) {
		super.copyFrom (oSource);
		meEmbedType = oSource.meEmbedType;
		meEmbedMode = oSource.meEmbedMode;
		msExpression = oSource.msExpression;
	}

/**
 * Compare text fields for content equality.
 * <p>
 * Compare this field against the one passed on the parameter object
 * for content equality.  The graphics sources of the fields are not
 * compared.  To be equal, the fields' content must match in all
 * aspects: raw text, attributes, embedded field content, and so on.
 * @param object Field to compare against
 * @return TRUE if the fields are equal; FALSE otherwise.
 */
	public boolean equals (Object object) {
		
		if (this == object)
			return true;
		
		if (!super.equals(object))
			return false;

		TextField test = (TextField) object;
		
		if ((msExpression != null) && (test.msExpression != null)) {
			if (! msExpression.equals (test.msExpression)) {
				return false;
			}
		} 
		else if (msExpression != test.msExpression) {		// null/non-null mismatch
			return false;
		}
		
		return meEmbedType == test.meEmbedType &&
			   meEmbedMode == test.meEmbedMode;
	}

	public int hashCode() {
		int hash = 53;
		if (msExpression != null)
			hash = (hash * 31) ^ msExpression.hashCode();
		hash = (hash * 31) ^ super.hashCode();
		hash = (hash * 31) ^ meEmbedType;
		hash = (hash * 31) ^ meEmbedMode;
		return hash;
	}

/**
 * Compare text fields for content inequality.
 * <p>
 * Compare this field against the one passed on the parameter oCompare
 * for content inequality.	The graphics sources of the fields are not
 * compared.  This is the exact opposite of the equality comparison.
 * @param oCompare Field to compare against
 * @return TRUE if the fields are unequal; FALSE otherwise.
 */
	public boolean notEqual (TextField oCompare) {
		return ! equals (oCompare);
	}

/**
 * Overridable equality comparison.
 * <p>
 * Compare this field against the one passed on the parameter oCompare
 * for equality.  The default implementation simply defers to
 * operator==().  If you derive a class from this, you can do your own
 * comparisons.
 * @param oCompare Field to compare against
 * @return TRUE if the fields are equal; FALSE otherwise.
 */
	public boolean isEqual (TextField oCompare) {
		return equals (oCompare);
	}

/**
 * Overridable method to create a new instance.
 * <p>
 * When the caller works with a derived class, it must provide an
 * implementation of Clone() that at least creates a new field of the
 * correct (derived) type.	This will be invoked by the parent stream
 * when the caller adds one of its fields to the stream, in order to
 * create the copy that the parent stream holds on to.	The default
 * implementation creates a copy of the field class through the copy
 * constructor.
 * @return Pointer to cloned copy.	Note that this is a true clone that
 * copies all attributes of this field, as opposed to creating an empty
 * new instance.
 */
	public TextField cloneField () {
		return new TextField (this);
	}

// Inherited from class TextStream.
/**
 * Return the position of this field in the parent stream.
 * <p>
 * This method is inherited from class TextStream.	The implementation
 * in Text Field returns a pointer to a position object that allows
 * the caller to determine the parent stream and the position within
 * that stream.  Derived classes need not override this method.
 * @return Pointer to a const text position object that represents this
 * field's position in its parent stream.  If the field object is not
 * embedded in any stream, the method returns NULL.
 */
	public TextPosn position () {
		return ((moPosn == null) || (moPosn.stream() == null)) ? null : moPosn;
	}

	public void setEmbedMode (int eEmbedMode) {
		meEmbedMode = eEmbedMode;
	}

	public int getEmbedMode () {
		return meEmbedMode;
	}

	public void setEmbedType (int eEmbedType) {
		meEmbedType = eEmbedType;
	}

	public int getEmbedType () {
		return meEmbedType;
	}

	public void setExpression (String sExpression) {
		msExpression = sExpression;
	}

	public String getExpression () {
		return msExpression;
	}

	void positionSet (TextStream poStream, int nIndex) {
		if (poStream == null) {
			moPosn = null;
		} else if (moPosn == null) {
			moPosn = new TextPosn (poStream, nIndex, TextPosn.POSN_AFTER);
		} else {
			moPosn.associate (poStream, nIndex, TextPosn.POSN_AFTER);
		}
	}

//	protected void Invalidate () {
//		TextDisplay poDisplay = Display();
//		if (poDisplay == null) {
//			return;
//		}
//
//		TextField poNonConst = (TextField) this;
//		TextRange oRange (poNonConst);
//
//		poDisplay.UpdateSelected (oRange);
//	}
}
