package com.adobe.xfa.text;

import com.adobe.xfa.ut.UnitSpan;

/**
 * A text label is a displayable stream (see class TextDispStr) that
 * has unlimited height and width, with minimums going right down to
 * zero.  The application does not set these dimensions directly; they
 * are always derived from the text content.
 * @exclude from published api.
 */

public class TextLabel extends TextDispStr {

/**
 * Default constructor.
 * <p>
 * The text stream contains no content and has no pool/mapping
 * assocotiation.
 */
	public TextLabel () {
		setTraditionalFrame (new LabelFrame());
	}

/**
 * Copy constructor.
 * <p>
 * Copies all contents of the text label.
 */
	public TextLabel (TextLabel oSource) {
		super (oSource);
		setTraditionalFrame (new LabelFrame());
	}

/**
 * Copy constructor with graphics source information.
 * <p>
 * Copy all stream content from the source stream, using the optional
 * graphic attribute pool.	The display is not automatically created.
 * @param oSource - Source text stream to copy content from.
 * @param poPool - Graphic attribute pool to use.
 */
//	public TextLabel (TextStream oSource, GFXAttrPool poPool) {
//		TextDispStr = oSource, poPool;
//		setTraditionalFrame (new TextFrameLabel());
//	}

/**
 * Constructor with source text string.
 * <p>
 * Create a text stream whose initial content is copied from the given
 * string.	The text stream initially has no attribute pool association.
 * The display is not automatically created.
 * @param sSource - String whose contents are to be copied to the text
 * stream.
 */
	public TextLabel (String sSource) {
		super (sSource);
		setTraditionalFrame (new LabelFrame());
	}

/**
 * Assign this text label's content from the given text label.
 * <p>
 * Replace this stream's content with a copy of the content of the given
 * stream.	The graphic source information is <b>not</b> copied.  In
 * other words, fonts will be re-mapped in this stream's font service
 * and attributes will be re-pooled in any attribute pool associated
 * with this stream.	The display pointer is not copied, as each
 * stream has its own display; nor is a display automatically created.
 * @param oSource - Stream containing source content to copy.
 */
	public void copyFrom (TextLabel oSource) {
		if (this != oSource) {
			super.copyFrom (oSource);
			getLabelFrame().copyFrom (oSource.getLabelFrame());
		}
	}

/**
 * Compare text blocks for equality.
 * <p>
 * Compare this stream against the one passed on the parameter oCompare
 * for equality.  The graphics sources of the streams are not compared.
 * To be equal, the streams' content must match in all aspects: raw
 * text, attributes, embedded field content, and so on.  The display
 * does not participate in the comparison.
 * @param oCompare - Stream to compare against
 * @return TRUE if the streams are equal; FALSE otherwise.
 */
// Javaoprt: not needed when it merely calls super.
//	public boolean equals (Object oCompare) {
//		return super.equals (oCompare);
//	}


/**
 * Compare text streams for inequality.
 * <p>
 * Compare this stream against the one passed on the parameter oCompare
 * for inequality.	The graphics sources of the streams are not
 * compared.  This is the exact opposite of the equality comparison.
 * The display does not participate in the comparison.
 * @param oCompare - Stream to compare against
 * @return TRUE if the streams are unequal; FALSE otherwise.
 */
	public boolean notEqual (TextLabel oCompare) {
		return ! equals (oCompare);
	}

	private LabelFrame getLabelFrame () {
		return (LabelFrame) (getTraditionalFrame());
	}
}

/*
 *TextFrameLabel - Class to represent geometry of a text label.
 */

class LabelFrame extends TraditionalFrame {
	static final UnitSpan NO_SIZE = new UnitSpan (UnitSpan.INCHES_72K, -1);

	public UnitSpan minWidth () {
		return NO_SIZE;
	}

	public UnitSpan minHeight () {
		return NO_SIZE;
	}

	public UnitSpan maxWidth () {
		return NO_SIZE;
	}

	public UnitSpan maxHeight () {
		return NO_SIZE;
	}
}
