/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.search.eval;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.query.Row;

import org.apache.felix.scr.annotations.Component;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.resource.collection.ResourceCollection;

import com.day.cq.search.Predicate;

/**
 * Finds items that are member of a specific
 * <a href="https://docs.adobe.com/docs/en/aem/6-2/develop/ref/javadoc/org/apache/sling/resource/collection/ResourceCollection.html">
 * sling resource collection</a>.
 *
 * <p>
 * This is a filtering-only predicate and cannot leverage a search index.
 * Does not support facet extraction.
 *
 * <h3>Name:</h3>
 * memberOf
 *
 * <h3>Properties:</h3>
 * <dl>
 * <dt>memberOf</dt>
 * <dd>path of sling resource collection</dd>
 * </dl>
 *
 * @since 6.0
 */
@Component(metatype = false, factory = "com.day.cq.search.eval.PredicateEvaluator/memberOf")
public class CollectionPredicateEvaluator extends AbstractPredicateEvaluator {
    public static final String COLLECTION = "memberOf";

    private static final String COLLECTION_PATH_SET_KEY = "collectionPathSet";
    
    private static final String COLLECTION_FOLDER_SET_KEY = "collectionFolderSet";

    @Override
    public boolean includes(Predicate p, Row row, EvaluationContext context) {
        if (!p.hasNonEmptyValue(COLLECTION)) {
            return true;
        }
        // Set of paths obtained from sling:members property
        Set<String> collectionPathSet = (Set<String>)context.get(COLLECTION_PATH_SET_KEY);
        // Set of path of members which are folders
        Set<String> collectionFolderSet = (Set<String>)context.get(COLLECTION_FOLDER_SET_KEY);
        if (collectionPathSet == null) {
            Resource colRes = context.getResourceResolver().getResource(p.get(COLLECTION));
            collectionPathSet = Collections.<String>emptySet();
            collectionFolderSet = Collections.<String>emptySet();
            if (colRes != null) {
                ResourceCollection collection = colRes.adaptTo(ResourceCollection.class);
                Iterator<Resource> collectionItr = collection.getResources();
                collectionPathSet = new HashSet<String>();
                collectionFolderSet = new HashSet<String>();
                while(collectionItr.hasNext()) {
                    Resource collectionMember = collectionItr.next();
                    collectionPathSet.add(collectionMember.getPath());
                    if (isFolder(collectionMember)) {
                        collectionFolderSet.add(collectionMember.getPath() + "/");
                    }
                }  
            } 
            // Put the set of paths in context so that it can be reused subsequently.
            context.put(COLLECTION_PATH_SET_KEY, collectionPathSet);
            context.put(COLLECTION_FOLDER_SET_KEY, collectionFolderSet);
        }     
        // returns true if either it is a direct member or is member of a folder which is a direct member
        return collectionPathSet.contains(context.getPath(row)) || isFolderMember(collectionFolderSet, context.getPath(row));
    }

    @Override
    public boolean canXpath(Predicate predicate, EvaluationContext context) {
        return false;
    }

    @Override
    public boolean canFilter(Predicate predicate, EvaluationContext context) {
        return true;
    }
    
    private boolean isFolder(Resource resource){
        Node n = resource.adaptTo(Node.class);
        try {
            return n.isNodeType("nt:folder");
        } catch (RepositoryException e) {
            return false;
        }
    }
    
    private boolean isFolderMember(Set<String> collectionFolderSet, String path) {
        for (String folderPath : collectionFolderSet) {
            if(path.indexOf(folderPath) == 0) {
                return true;
            }
        }
        return false;
    }
}
