/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.api;

import java.util.EnumSet;
import java.util.Set;
import java.util.Collections;

import javax.servlet.ServletRequest;

/**
 * Enumeration of the various cq5 debug flags
 */
public enum DebugFlag {

    /**
     * displays layout information
     */
    LAYOUT,
    
    /**
     *  display mobile device info as inline HTML
     */
    MDEV,
    
    /**
     *  display mobile device info as HTML comments
     */
    MDEVC;

    /**
     * name of the request attribute
     */
    public static final String REQUEST_ATTRIBUTE_NAME = DebugFlag.class.getName();

    /**
     * Default parameter name of the debug request parameter
     */
    public static final String DEBUG_PARAM_NAME = "debug";

    /**
     * Returns the debug mode for the current request, if set in the request attributes
     * @param req request
     * @return debug mode, empty Set if not defined in request attributes
     */
    @SuppressWarnings("unchecked")
    public static Set<DebugFlag> fromRequest(ServletRequest req) {
        Set<DebugFlag> flag = (Set<DebugFlag>) req.getAttribute(REQUEST_ATTRIBUTE_NAME);
        if (flag == null) {
            flag = Collections.emptySet();
        }
        return flag;
    }
    
    /** Return the current set of debug modes defined by supplied request parameter.
     *  If the set was already stored in the request attributes, return that.
     *  @param request the current request
     *  @param parameterName the (multi-valued) request parameter that defines debug modes
     *  @return the Set of DebugFlag, from request attributes if present, else computed,
     *      empty Set if no debug modes defined.
     */
    @SuppressWarnings("unchecked")
    public static Set<DebugFlag> fromRequestParameter(ServletRequest request, String parameterName) {
        Set<DebugFlag> dFlag = (Set<DebugFlag>) request.getAttribute(DebugFlag.REQUEST_ATTRIBUTE_NAME);
        if (dFlag == null) {
            String[] debugValues = request.getParameterValues(DEBUG_PARAM_NAME);
            if (debugValues != null) {
                dFlag = EnumSet.noneOf(DebugFlag.class);
                for (String m: request.getParameterValues(DEBUG_PARAM_NAME)) {
                    try {
                        dFlag.add(DebugFlag.valueOf(m.toUpperCase()));
                    } catch (IllegalArgumentException e) {
                        // ignore
                    }
                }
                dFlag = Collections.unmodifiableSet(dFlag);
            } else {
                dFlag = Collections.emptySet();
            }
            request.setAttribute(DebugFlag.REQUEST_ATTRIBUTE_NAME, dFlag);
        }
        return dFlag;
    }
}