/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.msm.commons;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;

import com.day.cq.wcm.api.WCMException;
import com.day.cq.wcm.msm.api.LiveAction;
import com.day.cq.wcm.msm.api.LiveActionFactory;
import org.apache.sling.api.wrappers.ValueMapDecorator;

import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * Base implementation of a {@code LiveActionFactory} service
 *
 * @param <ActionType> Type casting ActionType which extends {@link LiveAction}
 */
public abstract class BaseActionFactory<ActionType extends LiveAction> implements LiveActionFactory<ActionType> {

    /**
     * Creates a new ActionType instance given the specified configuration.
     * 
     * @param config the {@code ValueMap} containing the action's configuration
     * @return a new action instance
     * @throws WCMException Throws WCMException
     */
    protected abstract ActionType newActionInstance(ValueMap config) throws WCMException;

    /**
     * Creates an action using the {@link #newActionInstance(ValueMap)} method. The configuration
     * passed to that method consists of the default configuration returned by
     * {@link #getFactoryConfig()}, overlayed by the properties on the given resource.
     *
     * @return a new action instance
     */
    public ActionType createAction(Resource resource) throws WCMException {
        ValueMap resourceMap = null;
        if (resource != null) {
            resourceMap = resource.adaptTo(ValueMap.class);
        }
        Map<String, Object> result = mergeConfigs(getFactoryConfig(), resourceMap);
        return newActionInstance(new ValueMapDecorator(result));
    }

    /**
     * <p>Returns the factory configuration. The returned value is the base of the configuration
     * that will be passed to {@link #newActionInstance(ValueMap)}.</p>
     *
     * <p>This method needs to be overridden by subclasses that want to make use of the
     * configuration override mechanism. The purpose of this dummy implementation is to provide
     * compatibility with already existing subclasses.</p>
     *
     * @see #createAction(Resource)
     * @return the factory configuration
     */
    protected Dictionary getFactoryConfig() {
        return null;
    }

    /**
     * Merges the specified default and override configurations.
     *
     * @param base The default configuration
     * @param overrides A map containing entries to override in the default configuration
     * @return A new configuration map containing the merge result
     */
    private Map<String, Object> mergeConfigs(Dictionary base, Map<String, Object> overrides) {
        Map<String, Object> result = new HashMap<String, Object>();
        if (base != null) {
            Enumeration enumeration = base.keys();
            while (enumeration.hasMoreElements()) {
                Object key = enumeration.nextElement();
                result.put(key.toString(), base.get(key));
            }
        }
        if (overrides != null) {
            result.putAll(overrides);
        }
        return result;
    }

}
