/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.workflow.process;

import com.adobe.granite.workflow.collection.ResourceCollection;
import com.adobe.granite.workflow.collection.ResourceCollectionManager;
import com.day.cq.dam.api.Asset;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.api.Revision;
import com.day.cq.wcm.api.WCMException;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.exec.WorkflowData;
import com.day.cq.workflow.exec.WorkflowProcess;
import com.day.cq.workflow.job.AbsoluteTimeoutHandler;
import com.day.cq.workflow.metadata.MetaDataMap;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.commons.json.JSONObject;
import org.apache.sling.jcr.api.SlingRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.version.Version;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Create Version")
public class CreateVersionProcess implements WorkflowProcess {

    /** the logger */
    private static final Logger log = LoggerFactory.getLogger(CreateVersionProcess.class);

    private static final String WORKFLOW_COLLECTION_PAGE_RESOURCE_TYPE = "cq/workflow/components/collection/page";

    public static final String TYPE_JCR_PATH = "JCR_PATH";
    public static final String TYPE_JCR_UUID = "JCR_UUID";

    @Reference
    private ResourceResolverFactory resolverFactory = null;

    @Reference
    private SlingRepository repository;

    @Reference
    private ResourceCollectionManager rcManager;

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap args) throws WorkflowException {
        try {
            Session session = workflowSession.getSession();
            WorkflowData data = workItem.getWorkflowData();
            String path = null;
            String type = data.getPayloadType();
            if (type.equals(TYPE_JCR_PATH) && data.getPayload() != null) {
                String payloadData = (String) data.getPayload();
                if (session.itemExists(payloadData)) {
                    path = payloadData;
                }
            } else if (data.getPayload() != null && type.equals(TYPE_JCR_UUID)) {
                Node node = session.getNodeByIdentifier((String) data.getPayload());
                path = node.getPath();
            }

            if (path != null) {
                // create version for page or asset
                ResourceResolver resolver = resolverFactory.getResourceResolver(Collections.singletonMap("user.jcr.session", (Object) session));
				Resource res = resolver.getResource(path);
                Map<String, String> revisionMap = new HashMap<String, String>();

                if (res.adaptTo(Page.class) != null) {
                    Page page = res.adaptTo(Page.class);
                    PageManager pageManager = page.getPageManager();

                    // Create a version of a single page
                    // Or a Workflow Package page
                    createPageVersion(page, workItem, pageManager, revisionMap);

                    // If the page contains a collection of resource (e.g. Workflow Package page)
                    if (page.getContentResource().isResourceType(WORKFLOW_COLLECTION_PAGE_RESOURCE_TYPE)) {
                        // check for resource collection
                        List<ResourceCollection> rcCollections = rcManager.getCollectionsForNode((Node) session.getItem(path));
                        List<String> paths = ResourceCollectionHelper.getPaths(path, rcCollections);

                        for (String collectionPath : paths) {
                            Resource collectionResource = resolver.getResource(collectionPath);

                            if (collectionResource.adaptTo(Page.class) != null) {
                                createPageVersion(collectionResource.adaptTo(Page.class), workItem, pageManager, revisionMap);
                            } else if (collectionResource.adaptTo(Asset.class) != null) {
                                createAssetVersion(collectionResource.adaptTo(Asset.class), workItem, revisionMap);
                            }
                        }
                    }
                } else if (res.adaptTo(Asset.class) != null) {
                    createAssetVersion(res.adaptTo(Asset.class), workItem, revisionMap);
                } else {
                    log.warn("Cannot create version of " + path);
                }

                // set version info in workflow data's metadata
                if (!revisionMap.isEmpty()) {
                    JSONObject revisionJs = new JSONObject();

                    for (String key :revisionMap.keySet()) {
                        revisionJs.put(key, revisionMap.get(key));
                    }

                    data.getMetaDataMap().put("versions", revisionJs.toString());
                    if (workItem.getWorkflowData().getMetaDataMap().get(AbsoluteTimeoutHandler.ABS_TIME, String.class) != null) {
                        Calendar cal = getTime(workItem);
                        data.getMetaDataMap().put("comment",
                                "Activate version " + revisionMap + " on " + cal.getTime().toString());
                    }
                    workflowSession.updateWorkflowData(workItem.getWorkflow(), data);
                }
            } else {
                log.warn("Cannot process payload. Path is null");
            }
        } catch (RepositoryException e) {
            throw new WorkflowException(e);
        } catch (WCMException e) {
            throw new WorkflowException(e);
        } catch (Exception e) {
            throw new WorkflowException(e);
        }
    }

    private void createPageVersion(Page page, WorkItem workItem, PageManager pageManager, Map<String, String> revisionMap) throws WCMException, RepositoryException {
        if (page == null) {
            return;
        }

        String versionLabel = createUniqueVersionLabel(pageManager.getRevisions(page.getPath(), null),
                getAbsTime(workItem));
        Revision rev = pageManager.createRevision(page, versionLabel, null);
        revisionMap.put(page.getContentResource().getPath(), rev.getVersion().getName());
    }

    private void createAssetVersion(Asset asset, WorkItem workItem, Map<String, String> revisionMap) throws Exception {
        if (asset == null) {
            return;
        }

        final String versionLabel = createUniqueVersionLabel(asset.getRevisions(null), getAbsTime(workItem));
        com.day.cq.dam.api.Revision rev = asset.createRevision(versionLabel, null);
        revisionMap.put(asset.getPath(), rev.getVersion().getName());
    }

    private String createUniqueVersionLabel(Collection<?> revisions, final String versionLabelHint)
            throws RepositoryException {

        if (versionLabelHint==null) {
            return null;
        }

        final List<Version> versions = new LinkedList<Version>();
        for (final Object o : revisions) {

            final Version v;
            if (o instanceof Revision) {
                v = ((Revision)o).getVersion();
            } else if (o instanceof com.day.cq.dam.api.Revision) {
                v = ((com.day.cq.dam.api.Revision)o).getVersion();
            } else {
                v = null;
            }

            if (null != v) {
                versions.add(v);
            }
        }

        String versionLabel = versionLabelHint;

        int count = 1;
        while (true) {

            boolean unique = true;
            for (final Version v : versions) {
                if (v.getContainingHistory().hasVersionLabel(versionLabel)) {
                    versionLabel = versionLabelHint + " (" + ++count + ")";
                    unique = false;
                    break;
                }
            }

            if (unique) {
                break;
            }
        }

        return versionLabel;
    }

    private String getAbsTime(WorkItem workItem) {
        if (workItem.getWorkflowData().getMetaDataMap().get(AbsoluteTimeoutHandler.ABS_TIME, String.class) != null) {
            Calendar cal = getTime(workItem);
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy.MM.dd HH.mm.ss");
            return "Scheduled Activation Time is " + formatter.format(cal.getTime());
        } else {
            return null;
        }
    }

    private Calendar getTime(WorkItem workItem) {
        Long time = workItem.getWorkflowData().getMetaDataMap().get(AbsoluteTimeoutHandler.ABS_TIME, Long.class);
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(time);
        return cal;
    }

}
