/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.query;

import com.day.crx.statistics.Report;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.NodeIterator;
import javax.jcr.Node;
import javax.jcr.query.QueryManager;
import java.util.Iterator;
import java.util.Map;
import java.util.Calendar;
import java.util.TreeMap;
import java.util.SortedMap;
import java.util.ArrayList;
import java.util.List;

import org.apache.jackrabbit.util.Text;
import org.apache.jackrabbit.util.ISO9075;

/**
 * <code>QueryHistoryReport</code> implements a report that returns historical
 * data about a given query.
 *
 * @author mreutegg
 */
public class QueryHistoryReport extends Report {

    /**
     * A user query.
     */
    private final String query;

    /**
     * Default period is 12 months
     */
    private int period = 12;

    /**
     * Creates a new report.
     *
     * @param dataPath the location where the query statistics are stored.
     * @param query    the user query.
     */
    public QueryHistoryReport(String dataPath, String query) {
        super(dataPath);
        this.query = query;
    }

    /**
     * {@inheritDoc}
     * Returns result rows with the following objects:
     * <ul>
     * <li>Path <code>String</code>, which indicates the month</li>
     * <li><code>Long</code> count (how many times the query was executed)</li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        // create buckets
        Query dummy = new Query(getDataPath(), "dummy", 0, 0);
        SortedMap buckets = new TreeMap();
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.MONTH, -(getPeriod() - 1));
        cal.set(Calendar.DAY_OF_MONTH, 1);
        for (int i = 0; i < getPeriod(); i++) {
            buckets.put(Text.getRelativeParent(dummy.getPath(), 2), new Long(0));
            dummy.setTimestamp(cal.getTimeInMillis());
            cal.add(Calendar.MONTH, 1);
        }
        QueryManager qm = session.getWorkspace().getQueryManager();
        StringBuffer stmt = new StringBuffer("/jcr:root");
        stmt.append(getDataPath());
        stmt.append("/*/*/*/");
        stmt.append(ISO9075.encode(query));
        NodeIterator nodes = qm.createQuery(stmt.toString(),
                javax.jcr.query.Query.XPATH).execute().getNodes();
        while (nodes.hasNext()) {
            Node n = nodes.nextNode();
            long count = n.getProperty("count").getLong();
            // get path of month node
            String path = n.getAncestor(n.getDepth() - 2).getPath();
            Long c = (Long) buckets.get(path);
            if (c != null) {
                buckets.put(path, new Long(c.longValue() + count));
            }
        }
        List data = new ArrayList();
        for (Iterator it = buckets.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry entry = (Map.Entry) it.next();
            String path = (String) entry.getKey();
            Long count = (Long) entry.getValue();
            data.add(new Object[]{path, count});
        }
        return data.iterator();
    }

    /**
     * @return the report period in number of days.
     */
    public int getPeriod() {
        return period;
    }

    /**
     * @param period the report period in number of days.
     */
    public void setPeriod(int period) {
        this.period = period;
    }
}
