/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.result;

import com.day.crx.statistics.Report;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.Value;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.util.Calendar;

/**
 * <code>RelatedQueriesReport</code> returns a list of related queries.
 *
 * @author mreutegg
 */
public class RelatedQueriesReport extends Report {

    /**
     * The query for which to return related queries.
     */
    private final String query;

    /**
     * Number of related queries to return at most. Default value: 5 queries.
     */
    private int size = 5;

    /**
     * Add OAK-specific "option(traversal ok)" to the statistics query so that no traversal warning is triggered
     */
    private final boolean traversalOk;

    /**
     * Creates a new report.
     *
     * @param dataPath the path where result statistics are stored.
     * @param query the user query.
     */
    public RelatedQueriesReport(String dataPath, String query) {
        this(dataPath, query, false);
    }

    /**
     * Creates a new report.
     *
     * @param dataPath the path where result statistics are stored.
     * @param query the user query.
     * @param traversalOk set to true to put "traveral ok" option in the statistics query
     */
    public RelatedQueriesReport(String dataPath, String query, boolean traversalOk) {
        super(dataPath);
        this.query = query.toLowerCase();
        this.traversalOk = traversalOk;
    }

    /**
     * {@inheritDoc}
     * Returns result rows with the following objects:
     * <ul>
     * <li>Related query <code>String</code></li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        ResultsByQueryReport report = new ResultsByQueryReport(getDataPath(), query, traversalOk);
        List related = new ArrayList();
        for (Iterator it = report.getResult(session); it.hasNext() && related.size() < getSize(); ) {
            Object[] data = (Object[]) it.next();
            StringBuffer path = new StringBuffer(getDataPath());
            path.append(data[0]);
            path.append("/");
            path.append(ResultSelectedPathBuilder.STATS_NAME);
            Item item = session.getItem(path.toString());
            if (item.isNode()) {
                Calendar cal = Calendar.getInstance();
                Node stats = (Node) item;
                // go back at most 3 years
                for (int i = 0; i < 3 && related.size() < getSize(); i++) {
                    String year = String.valueOf(cal.get(Calendar.YEAR));
                    cal.add(Calendar.YEAR, 1);
                    if (!stats.hasNode(year)) {
                        continue;
                    }
                    Node y = stats.getNode(year);
                    if (!y.hasProperty(ResultSelected.QUERIES)) {
                        continue;
                    }
                    Value[] values = y.getProperty(
                            ResultSelected.QUERIES).getValues();
                    for (int v = 0; v < values.length; v++) {
                        String query = values[v].getString().toLowerCase();
                        if (query.equals(this.query)) {
                            continue;
                        }
                        if (!related.contains(query)) {
                            related.add(query);
                        }
                    }
                }
            }
        }
        if (related.size() > getSize()) {
            related = related.subList(0, getSize());
        }
        for (int i = 0; i < related.size(); i++) {
            related.set(i, new Object[]{related.get(i)});
        }
        return related.iterator();
    }

    /**
     * @return the maximum number of queries to return.
     */
    public int getSize() {
        return size;
    }

    /**
     * @param size the maximum number of queries to return.
     */
    public void setSize(int size) {
        this.size = size;
    }
}
