/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.image;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Paint;
import java.awt.PaintContext;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.ColorModel;

/**
 * The <code>LineStyle</code> class implements the Communiqu� 2 line style.
 * This line style embraces both the stroke style (dashing, line width) and
 * the line color. To use this line style for line drawing, you would get the
 * color to set the <code>Paint</code> of the Java2D <code>Graphics2D</code>
 * and you would get the stroke to set the <code>Stroke</code>.
 * <p>
 * The line style is initialized using the paint only, in which case it defaults
 * to define a style with the given paint which is solid and has the width of
 * one point. The line style can also be initialized giving the paint, the line
 * width and the length of the dash and blank parts of a dashed line. See the
 * comments on the {@link #LineStyle(Paint, float, float, float)} constructor
 * for information on the possibilities.
 * <p>
 * The line style is immutable and final.
 *
 * @version $Revision$
 * @author fmeschbe
 * @since coati
 * @audience wad
 */
public final class LineStyle implements Paint, Stroke {

    /**
     * Style code for a solid line.
     */
    public static final int STYLE_SOLID  = 0;

    /**
     * Style code for a dotted line, where both the mark and the blank length
     * are one point.
     */
    public static final int STYLE_DOTTED = 1;

    /**
     * Style code for a dashed line, where the mark length is 3 points and the
     * blank length is 2 points.
     */
    public static final int STYLE_DASHED = 2;

    /**
     * Style code for a fancy dashed line, where both the mark and the blank
     * lentghs are specified with the constructor. This style code is set
     * by the constructor in this case. If this style code is set in the
     * constructor, when the mark length is set to zero, it has no effect and
     * the line is styled to be solid.
     */
    public static final int STYLE_FANCY  = 3;

    /**
     * The default color value, if the color parameter of either constructor
     * is <code>null</code>.
     */
    private static final Color DEFAULT_COLOR = Color.black;

    /** The default line style is a solid black line, 1px wide */
    public static final LineStyle DEFAULT = new LineStyle(DEFAULT_COLOR);

    /**
     * The default mark length, if the mark parameter to the constructor is
     * <code>NaN</code>. This leads to the interpretation of the blank parameter
     * as the style code.
     *
     * @see #LineStyle(Paint, float, float, float)
     */
    private static final float DEFAULT_MARK = 0;

    /**
     * The default blank length, if the blank parameter to the constructor is
     * <code>Nan</code>. In conjunction with the default mark parameters this
     * leads to the solid line being the default.
     *
     * @see #LineStyle(Paint, float, float, float)
     */
    private static final float DEFAULT_BLANK = STYLE_SOLID;

    /**
     * The default line width, if the width parameter to the constructor is
     * <code>NaN</code>, negative or zero.
     *
     * @see #LineStyle(Paint, float, float, float)
     */
    private static final float DEFAULT_WIDTH = 1;

    /**
     * The style value of the line, which is one of the predefined constants
     * above.
     */
    private final int style;

    /**
     * The length of the mark in points. This value is ignored, if the line
     * style is <code>STYLE_SOLID</code>.
     */
    private final float mark;

    /**
     * The length of the blank in points. This value is ignored, if the line
     * style is <code>STYLE_SOLID</code>.
     */
    private final float blank;

    /**
     * The width of the line in points.
     */
    private final float width;

    /**
     * The color of the line.
     */
    private final Paint paint;

    /**
     * The Java2D <code>Stroke</code> is cached after the first call to
     * {@link #createStrokedShape}.
     */
    private Stroke lineStroke = null;

    /**
     * The string representation of this line style is cached after the first
     * call to {@link #toString()}.
     *
     * @see #toString()
     */
    private String stringValue = null;

    /**
     * Creates a line style for a line with the given color. The line is solid
     * and has a width of one point.
     *
     * @param paint The <code>Paint</code> value to set. If this is
     *      <code>null</code>, the line color is defined to be black.
     */
    public LineStyle(Paint paint) {
	this(paint, Float.NaN, Float.NaN, Float.NaN);
    }

    /**
     * Creates a new line style with the given paint and stroking attributes.
     * The mark and blank parameters are special : If mark is zero, the blank
     * parameters is interpreted as a generic style and the mark and blank
     * values are reset accordingly :
     * <table>
     * <tr><td>STYLE_DOTTED<td>Dotted stroke with mark=1, blank=1
     * <tr><td>STYLE_DASHED<td>Dashed stroke with mark=3, blank=2
     * <tr><td>else<td>Solid stroke with no breaks, that is mark=1, blank=0
     * </table>
     * <p>
     * If the mark parameters is given as a non-zero value, the style is set
     * to STYLE_FANCY. If any of the parameters has a value as listed in the
     * following table, a default value is assumed :
     * <table>
     * <tr><td><code>paint == null</code><td>black color
     * <tr><td><code>isNaN(mark)</code><td>mark=0, prompting the above algorithm
     * <tr><td><code>isNaN(blank)</code><td>blank=STYLE_SOLID, which applies if
     * 		mark is zero
     * <tr><td><code>isNan(width)</code><td>width = 1
     * </table>
     * <p>
     * If mark is non-zero (or after the above listed treatment) a dashed stroke
     * is defined, where mark defines the point length of set points and blank
     * defines the point length of unset points, applied alternatively for the
     * length of the line to be drawn. If on the other hand mark is non-zero
     * but the blank length is zero, this defines a solid line.
     *
     * @param paint The paint value to set. If this is <code>null</code>, the
     * 		line color is defined to be black.
     * @param mark The length of the mark in points. If zero, negative or NaN
     * 		the blank parameter contains a style constant.
     * @param blank The length of the blank in points or a style constant if the
     * 		length is zero, negative or NaN. If blank is negative, zero or
     * 		NaN the line style is defined to be solid. The line style is
     * 		also defined to be solid, if the blank parameter is not a valid
     * 		style constant if mark is zero.
     * @param width The line width. If this is zero, negative or NaN, the
     * 		width defaults to 1 point.
     */
    public LineStyle(Paint paint, float mark, float blank, float width) {

	// check parameters
	int style;
	if (paint == null)                    paint = DEFAULT_COLOR;
	if (Float.isNaN(mark) || mark < 0)    mark  = DEFAULT_MARK;
	if (Float.isNaN(blank) || blank < 0)  blank = DEFAULT_BLANK;
	if (Float.isNaN(width) || width <= 0) width = DEFAULT_WIDTH;

	// blank is style if mark is zero
	if (mark == 0) {
	    switch ((int)blank) {
		case STYLE_DOTTED:
		    style = STYLE_DOTTED;
		    mark = 1 * width;
		    blank = 1 * width;
		    break;

		case STYLE_DASHED:
		    style = STYLE_DASHED;
		    mark = 3 * width;
		    blank = 2 * width;
		    break;

		default: /* solid */
		    style = STYLE_SOLID;
		    mark = 1;
		    blank = 0;
	    }
	} else {
	    // if mark and blank are both valid, this line is fancy, else solid
	    if (mark <= 0 || blank <= 0) {
		style = STYLE_SOLID;
		mark = 1;
		blank = 0;
	    } else {
		style = STYLE_FANCY;
	    }
	}

	this.paint = paint;
	this.style = style;
	this.mark = mark;
	this.blank = blank;
	this.width = width;
    }

    /**
     * Returns the style of this line style which is one of the predefined
     * style constants.
     * @return the style of this line style.
     */
    public int getStyle() {
	return style;
    }

    /**
     * Returns the mark length of this line style. This is either the mark
     * length used to construct the line style or the length of the mark
     * according to the style used to create the line style.
     *
     * @return the mark length of this line style.
     */
    public float getMark() {
	return mark;
    }

    /**
     * Returns the blank length of this line style. This is either the blank
     * length used to construct the line style or the length of the blank
     * according to the style used to create the line style.
     *
     * @return the blank length of this line style.
     */
    public float getBlank() {
	return blank;
    }

    /**
     * Returns the width of the line according to the line style.
     * @return the width of the line according to the line style.
     */
    public float getWidth() {
	return width;
    }

    //---------- Paint interface -----------------------------------------------

    /**
     * Creates and returns a {@link PaintContext} used to
     * generate the color pattern.
     * @param cm the {@link ColorModel} that receives the
     * <code>Paint</code> data. This is used only as a hint.
     * @param deviceBounds the device space bounding box
     *                     of the graphics primitive <being rendered
     * @param userBounds the user space bounding box
     *                     of the graphics primitive being rendered
     * @param xform the {@link AffineTransform} from user
     *      space into device space
     * @param hints the hint that the context object uses to
     *              choose between rendering alternatives
     * @return the <code>PaintContext</code> for
     *              generating color patterns
     * @see PaintContext
     */
    public PaintContext createContext(ColorModel cm, Rectangle deviceBounds,
	Rectangle2D userBounds, AffineTransform xform, RenderingHints hints) {
	return paint.createContext(cm, deviceBounds, userBounds, xform, hints);
    }

    /**
     * Returns the type of this <code>Transparency</code>.
     * @return the field type of this <code>Transparency</code>, which is
     *		either OPAQUE, BITMASK or TRANSLUCENT.
     */
    public int getTransparency() {
	return paint.getTransparency();
    }

    //---------- Stroke interface ----------------------------------------------

    /**
     * Returns an outline <code>Shape</code> which encloses the area that
     * should be painted when the <code>Shape</code> is stroked according
     * to the rules defined by the
     * object implementing the <code>Stroke</code> interface.
     * @param p a <code>Shape</code> to be stroked
     * @return the stroked outline <code>Shape</code>.
     */
    public Shape createStrokedShape(Shape p) {
	if (lineStroke == null) {
	    if (style == STYLE_SOLID) {

		lineStroke = new BasicStroke(width, BasicStroke.CAP_SQUARE,
		    BasicStroke.JOIN_MITER, 10);

	    } else {

		lineStroke = new BasicStroke(width, BasicStroke.CAP_SQUARE,
		    BasicStroke.JOIN_MITER, 10, new float[]{ mark, blank }, 0);

	    }
	}

	return lineStroke.createStrokedShape(p);
    }

    //---------- Object overwrite ----------------------------------------------

    /**
     * Returns a String representation of the LineStyle
     * @return a String representation of the LineStyle
     */
    public String toString() {
	if (stringValue == null) {
	    // not yet cached
	    StringBuffer buf = new StringBuffer();

	    buf.append("LineStyle: paint=0x");
	    buf.append(paint);
	    buf.append(", width=");
	    buf.append(width);

	    switch (style) {
		case STYLE_SOLID:
		    buf.append(", solid");
		    break;

		case STYLE_DOTTED:
		    buf.append(", dotted");
		    break;

		case STYLE_DASHED:
		    buf.append(", dashed");
		    break;

		case STYLE_FANCY:
		    buf.append(", fancy: mark=");
		    buf.append(mark);
		    buf.append(", blank=");
		    buf.append(blank);
		    break;
	    }

	    stringValue = buf.toString();
	}

	return stringValue;
    }

}
