/*************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.util;

import java.util.Stack;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The <code>FinalizerHandler</code> class implements the functionality to
 * register {@link Finalizer}s and call them at some point in time, such as
 * the end of the current <code>ExecutionContext</code>.
 * <p>
 * The registered {@link Finalizer}s will be called in the reverse sequence
 * of their registration. That is the last {@link Finalizer} registered will
 * get called first.
 *
 * @version $Revision: 1.13 $, $Date: 2004-08-22 06:56:09 +0200 (Sun, 22 Aug 2004) $
 * @author fmeschbe
 * @since antbear
 * Audience dev
 */
public class FinalizerHandler {

    /** default log */
    private static final Logger log =
        LoggerFactory.getLogger(FinalizerHandler.class);

    /** Keep the list of registered objects in this Stack. */
    private final Stack registered = new Stack();

    /**
     * Registers a <code>Finalizer</code>.
     * @param object the finalzier to register
     */
    public void registerObject(Finalizer object) {
        registered.push(object);
    }

    /**
     * Unregisters a <code>Finalizer</code> object. After unregistering the
     * objects <code>doFinalize()</code> method is not called anymore.
     *
     * @param object The <code>Finalizer</code> object to unregister.
     */
    public void unregister(Finalizer object) {
        registered.remove(object);
    }

    /**
     * Unregisters all registered <code>Finalizer</code> objects so that a
     * subsequent call to {@link #callFinalizers()} effectively does nothing.
     */
    public void unregisterAll() {
        registered.clear();
    }

    /**
     * Returns <code>true</code> if no {@link Finalizer} objects are registered
     * yet.
     * @return whether it's empty
     */
    public boolean isEmpty() {
        return registered.isEmpty();
    }

    /**
     * Calls all registered <code>Finalizer</code> {@link Finalizer#doFinalize()}
     * method, and removes them from the internal list.
     * <p>
     * The call to the {@link Finalizer#doFinalize()} is not expected to throw
     * any exceptions or even errors. In case a <code>Throwable</code> is thrown
     * this is logged as an error and processing continiues with the next
     * <code>Finalizer</code> to be finalized.
     */
    public void callFinalizers() {

        while (!registered.isEmpty()) {

            Finalizer fin = (Finalizer) registered.pop();

            try {

                fin.doFinalize();

            } catch (Throwable t) {

                // ignore anything thrown
                log.error("callFinalizers: Unexpected Exception/Error: {}",
                    t.toString());
                log.debug("dump", t);

            }
        }
    }
}
