/*************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.util;

import java.io.PrintStream;
import java.io.PrintWriter;

/**
 * An Exception that wraps another Throwable, and prints its wrapped exception's
 * stack trace as well as its own when printStackTrace is called.
 *
 * @version $Revision: 1.11 $, $Date: 2004-08-22 06:56:09 +0200 (Sun, 22 Aug 2004) $
 * @author benp
 * Audience core
 */
public abstract class WrappedException extends Exception {

    /** Root failure cause */
    protected Throwable rootException;

    /**
     * Class constructor that creates a <code>WrappedException</code> given
     * a message describing the failure cause
     * @param s description
     */
    public WrappedException(String s) {
        super(s);
    }

    /**
     * Class constructor that creates a <code>WrappedException</code> given
     * a message describing the failure cause and a root exception
     * @param s description
     * @param e root failure cause
     */
    public WrappedException(String s, Throwable e) {
        super(s);
	rootException = e;
    }

    /**
     * Class constructor that creates a <code>WrappedException</code> given
     * a root exception
     * @param t root failure cause
     */
    public WrappedException(Throwable t) {
        this(null, t);
    }

    /**
     * Returns the error message string of this exception.
     * @return error message string of this exception.
     */
    public String getMessage() {
        String s = super.getMessage();
	if (rootException == null) {
            return s;
	} else {
            String s2 = rootException.getMessage();
            return s == null ? s2 : s + ": " + s2;
	}
    }

    /**
     * Returns the root exception of this exception.
     * @return the root exception of this exception
     */
    public Throwable getRootException() {
        return rootException;
    }

    /**
     * Prints this <code>WrappedException</code> and its backtrace to the
     * standard error stream.
     */
    public void printStackTrace() {
	synchronized (System.err) {
	    super.printStackTrace();
            if (rootException != null) {
                rootException.printStackTrace();
            }
	}
    }

    /**
     * Prints this <code>WrappedException</code> and its backtrace to the
     * specified print stream.
     * @param s <code>PrintStream</code> to use for output
     */
    public void printStackTrace(PrintStream s) {
	synchronized (s) {
	    super.printStackTrace(s);
            if (rootException != null) {
                rootException.printStackTrace(s);
            }
	}
    }

    /**
     * Prints this <code>WrappedException</code> and its backtrace to
     * the specified print writer.
     * @param s <code>PrintWriter</code> to use for output
     * @since JDK1.1
     */
    public void printStackTrace(PrintWriter s) {
	synchronized (s) {
	    super.printStackTrace(s);
            if (rootException != null) {
                rootException.printStackTrace(s);
            }
	}
    }

}
