/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.launches.api;

import java.util.Calendar;
import java.util.Date;

/**
 * Launch status compared to production.
 */
public final class LaunchResourceStatus {
    
    public static enum LaunchStatusType{
       /**
        * New created resource compared to production
        */
       CREATED,
       /**
        * Unchanged resource compared to production
        */
       UNCHANGED,
       /**
        * Modified resource compared to production
        */
       MODIFIED,
       /**
        * Deleted resource compared to production
        */
       DELETED;
       
       /**
        * Get the launch status type value
        *
        * @return The launch promotion scope value
        */
       public String getValue() {
           return this.name().toLowerCase();
       }
       
       /**
        * @see java.lang.Enum#toString()
        */
       public String toString() {
           return getValue();
       }
    }

    
    /**
     * Path of the resource in launch
     */
    private final String resourcePath;
    
    /**
     * Title of the resource
     */
    private final String title;
    
    /**
     * The launch modification date.
     */
    private final Calendar launchModificationDate;
    
    /**
     * The production modification date.
     */
    private final Calendar productionModificationDate;
    
    /**
     * The user object of launch modification.
     */
    private final String launchUserId;
    
    /**
     * The user object of production modification.
     */
    private final String productionUserId;
    
    /**
     * Launch status type
     */
    private final LaunchStatusType type;
    
    public LaunchResourceStatus(LaunchStatusType type, String path, String title, Calendar launchLastModif, 
            Calendar prodLastModif, String launchUserId, String productionUserId){
        this.type = type;
        this.resourcePath = path;
        this.title = title;
        this.launchModificationDate = launchLastModif;
        this.productionModificationDate = prodLastModif;
        this.launchUserId = launchUserId;
        this.productionUserId = productionUserId;
    }
    
    public LaunchStatusType getType() {
        return type;
    }
    
    public String getResourcePath() {
        return resourcePath;
    }
    
    public String getTitle() {
        return title;
    }

    public Date getLaunchModificationDate() {
        return launchModificationDate != null ? launchModificationDate.getTime() : null;
    }

    public Date getProductionModificationDate() {
        return productionModificationDate != null ? productionModificationDate.getTime() : null;
    }

    public String getLaunchUserId() {
        return launchUserId;
    }

    public String getProductionUserId() {
        return productionUserId;
    }

    /**
     * Overridden to alter the preconditions when two launch statuses are
     * considered equal.
     *
     * @param o object to compare this object against
     * @return <code>true</code> if this object is considered equal to the
     *         other object, otherwise <code>false</code>
     */
    public final boolean equals(Object o) {
        if (o instanceof LaunchResourceStatus) {
            LaunchResourceStatus other = (LaunchResourceStatus) o;
            return other.resourcePath.equals(resourcePath) &&
                    other.type == type;
        }
        return false;
    }

    /**
     * Returns a hash code value for the object. Replaced in order to
     * return the same hashcode for two objects that are considered
     * equal according to the {@link #equals} method implemented above.
     *
     * @return a hash code value for this object.
     * @see java.lang.Object#equals(java.lang.Object)
     * @see java.util.Hashtable
     */
    public int hashCode() {
        return resourcePath.hashCode() ^ type.hashCode();
    }

}
