/*
 * File: CosObjectID.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.io.InputStream;

import com.adobe.internal.io.stream.InputByteStream;

/**
 * A light weight buffer to optimize the parsing of COS objects.
 */
final class CosParseBuf implements InputByteStream {
	private InputByteStream mBaseBuf = null;
	private int mBufferSize = 0;
	private long mOrigPos = 0;
	private int mBaseOffset = 0;
	private int mPos = 0;
	private int mEnd = 0;
	private byte[] mBuffer = null;
	
	// This parameter has been newly introduced to enable reading from this
	// Stream only till the buffer size. This is required for ObjectStreams in
	// which the buffer size actually denotes the size of the object that is to
	// be read and is useful in cases where two consecutive integers are written
	// without any delimiter.
	private boolean readTillBufferSize = false;

	/*
	 * Constructed from an underlying InputByteStream. The buffer chunk size
	 * would ideally be a perfect fit for the COS objects that are read into
	 * it for parsing. The actual size passed is determined by tweaking.
	 */
	CosParseBuf(InputByteStream baseBuf, int bufferSize)
		throws IOException
	{
		mBaseBuf = baseBuf;
		mBufferSize = bufferSize;
		mBuffer = new byte[mBufferSize];
		mOrigPos = baseBuf.getPosition();
		mEnd = baseBuf.read(mBuffer);
	}
	
	CosParseBuf(InputByteStream baseBuf, int bufferSize, boolean readTillBufferSize)
	throws IOException
{
	this(baseBuf, bufferSize);
	this.readTillBufferSize = readTillBufferSize;
}

	public int read()
		throws IOException
	{
		
		if(readTillBufferSize && (mPos == mBufferSize))			
			// If we have read till the buffer size and were required to
			// read till this point only then returning EOF
			return InputByteStream.EOF;
		if (mPos == mEnd) {
			
			mBaseOffset += mPos;			
			mEnd = mBaseBuf.read(mBuffer);
			if (mEnd <= 0)
				return InputByteStream.EOF;
			else
				mPos = 0;
		}
		return 0xFF & mBuffer[mPos++];
	}

	public int unget()
	{
		if(mPos > 0) --mPos;
		return mBuffer[mPos];
	}

	public long getPosition()
	{
		return mOrigPos + mBaseOffset + mPos;
	}

	public void close()
		throws IOException
	{
		mBaseBuf.seek(mOrigPos + mBaseOffset + mPos);
		mBaseBuf = null;
		mBuffer = null;
	}

	public boolean eof()
		throws IOException
	{
		if (mPos < mEnd)
			return false;
		return mBaseBuf.eof();
	}

	public long bytesAvailable()
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: bytesAvailable");
	}

	public boolean exists(byte[] key, long extent)
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: exists");
	}

	public  long indexOf(byte[] key)
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: indexOf");
	}

	public long length()
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: length");
	}

	public int read(byte[] bytes)
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: read");
	}

	public int read(byte[] bytes, int position, int length)
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: read");
	}

	public InputByteStream seek(long position)
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: seek");
	}

	public InputByteStream slice()
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: slice");
	}

	public InputByteStream slice(long begin, long length)
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: slice");
	}

	public InputStream toInputStream()
		throws IOException
	{
		throw new IOException("Unimplemented method call on CosParseBuf: toInputStream");
	}
}
