/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.tags;

import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.ResourceBundle;
import java.util.Set;

import org.apache.sling.api.resource.ValueMap;

/**
 * <code>ContentResourceBundle</code> implements a {@link ResourceBundle}, which
 * is backed by the <code>ValueMap</code> of a <code>Resource</code>. The
 * property names act as keys and the values of the properties are the localized
 * strings. The bundle implementation also uses an optional {@link #parent}
 * bundle as a fallback if a key is not found on the <code>ValueMap</code>.
 */
class ContentResourceBundle extends ResourceBundle {

    /**
     * The underlying value map from the Sling Resource.
     */
    private final ValueMap content;

    /**
     * Creates a new bundle based on <code>content</code>.
     *
     * @param content the underlying content Resource values.
     * @param parent an optional bundle used as fallback if a key is not found
     *            on the content node.
     */
    ContentResourceBundle(ValueMap content, ResourceBundle parent) {
        this.content = content;
        setParent(parent);
    }

    /**
     * {@inheritDoc}
     * <p/>
     * If there is a property with the given <code>key</code> the value of the
     * property is returned. If the property is multi-valued the first value is
     * returned, if there is no value (empty value array) an empty string is
     * returned. If the property does not exist <code>null</code> is returned.
     */
    protected Object handleGetObject(String key) {
        if (content.containsKey(key)) {
            return content.get(key, "");
        }

        return null;
    }

    /**
     * {@inheritDoc}
     * <p/>
     * The names of all properties on the underlying value map plus the keys of
     * the fallback resource bundle.
     */
    public Enumeration<String> getKeys() {

        // start with keys of this content
        Set<String> names = new HashSet<String>(content.keySet());

        // also add keys from parent if present
        ResourceBundle parent = this.parent;
        if (parent != null) {
            names.addAll(Collections.list(parent.getKeys()));
        }
        return Collections.enumeration(names);
    }
}
