/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api.model;

import java.io.Serializable;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.dam.scene7.api.constants.Scene7AssetType;

/**
 * Abstraction of a Scene7 Asset
 */
@ProviderType
public interface Scene7Asset extends Serializable {

    /**
     * Get the type of this asset
     *
     * @return A <code>Scene7AssetType</code> identifying this asset type
     */
    public abstract Scene7AssetType getAssetType();

    /**
     * Get the asset type as read from Scene7
     *
     * @return A <code>String</code> object containing the actual Scene7 type string
     */
    public abstract String getAssetTypeStr();

    /**
     * Get the asset's subtype string from Scene7 (if appliable)
     * 
     * @return A <code>String</code> object containing the Scene7 subtype string
     */
    public abstract String getAssetSubTypeStr();

    /**
     * Get the name of this asset
     *
     * @return <code>String</code> object containing the asset's name
     */
    public abstract String getName();

    /**
     * Get this asset handle
     *
     * @return <code>String</code> object with this asset's handle
     */
    public abstract String getAssetHandle();

    /**
     * Get the file name where the asset is stored
     *
     * @return <code>String</code> containing the file name of the asset
     */
    public abstract String getFileName();

    /**
     * Get the folder where this asset is stored
     *
     * @return <code>String</code> containing the folder path of the asset
     */
    public abstract String getFolder();

    /**
     * Get the Scene7 folder handle of the folder containing this asset
     *
     * @return <code>String</code> object containing a Scene7 folder handle
     */
    public abstract String getFolderHandle();

    /**
     * Get the user that created this asset
     *
     * @return <code>String</code> object containing the username of the user that created this asset
     */
    public abstract String getCreatedBy();

    /**
     * Get the user that last modified this asset
     *
     * @return <code>String</code> object containing the username of the user that last modified this asset
     */
    public abstract String getModifiedBy();

    /**
     * Get the creation date of this asset
     *
     * @return <code>Date</code> object with the creation time of the asset
     */
    public abstract Date getCreatedDate();

    /**
     * Get the last modified date of this asset
     *
     * @return <code>Date</code> object with the time the asset was last modified
     */
    public abstract Date getModifiedDate();

    /**
     * Get the original file uploaded to Scene7 for this asset
     *
     * @return <code>String</code> object containing the original file of this asset
     */
    public abstract String getOriginalFile();

    /**
     * Get the original path of the file uploaded to Scene7 for this asset
     *
     * @return <code>String</code> object containing the original path for this asset
     */
    public abstract String getOriginalPath();

    /**
     * Get the list of this asset's sub-assets
     *
     * @return a <code>List</code> with the <code>Scene7Asset</code> assets linked to this asset
     */
    public abstract List<Scene7Asset> getSubAssets();

    /**
     * Adds a new sub-asset to the current asset
     *
     * @param subAsset
     *            a <code>Scene7Asset</code> object to be added as a sub-asset of the current asset
     */
    public abstract void addSubAsset(Scene7Asset subAsset);

    /**
     * Gets the originator asset for this asset Should only return a valid asset if the current asset was generated from another uploaded
     * asset (e.g. a Template asset is usually generated after importing a PSD into Scene7; the original uploaded PSD will be the Template
     * asset's originator)
     *
     * @return a <code>Scene7Asset</code> object holding this asset's originator asset
     */
    public abstract Scene7Asset getOriginatorAsset();

    /**
     * Updates the originator asset of the current asset
     *
     * @param originator
     *            the new <code>Scene7Asset</code> originator of the current asset
     */
    public abstract void updateOriginatorAsset(Scene7Asset originator);

    /**
     * Get the width of this asset
     *
     * @return a <code>Long</code> object containing the width of this asset
     */
    public abstract Long getWidth();

    /**
     * Get the height of this asset
     *
     * @return a <code>Long</code> object containing the height of this asset
     */
    public abstract Long getHeight();

    /**
     * Get the url modified of this asset
     *
     * @return a <code>String</code> object containing the url modifier of this asset
     */
    public abstract String getUrlModifier();

    /**
     * Get the viewer preset type of this asset
     *
     * @return a <code>String</code> object containing the viewer preset type
     */
    public abstract String getViewerPresetType();

    /**
     * Get a map containing the viewer preset configuration for this asset
     *
     * @return a <code>Map</code> that store the name-value pairs found in the viewer preset configuration of the asset
     */
    public abstract Map<String, String> getViewerPresetConfigurationSettings();

    /**
     * Returns the root folder or an empty <code>String</code> if no folder is specified for this asset.
     *
     * @return path to Root folder
     */
    public abstract String getRootFolder();

    /**
     * Gets the publish status of this asset
     *
     * @return a <code>Boolean</code> indicating if this asset is published or not
     */
    public abstract boolean isPublished();

    /**
     * Gets the size in bytes of the file where the asset is stored
     *
     * @return an <code>Integer</code> containing the file size
     */
    public abstract Integer getFileSize();

    /**
     * Gets the video encoding preset id for this asset, if this is a video scene7 asset
     *
     * @return a <code>String</code> containing the video encoding preset for a video scene7 asset
     */
    public abstract String getVideoEncodingPresetId();

    public abstract Map<String, String> getAssetProperties();

    /**
     * Gets the <code>Scene7SmartCrop</code> object (if exists) associated with
     * this asset
     * 
     * @return <code>Scene7SmartCrop</code> object
     */
    public abstract Scene7SmartCrop getSmartCrop();
    /**
     * Returns a map with video properties such as duration, bit rate of the asset if its a video, null otherwise
     *
     * @return a <code>java.util.Map</code>
     */
    public Map<String, String> getVideoProperties();
}
