/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.replication;

import java.util.Iterator;
import java.util.List;

import javax.jcr.Session;

import org.osgi.annotation.versioning.ProviderType;

/**
 * The <code>Replicator</code> is the replication service.
 * It can be used to replicate content.
 */
@ProviderType
public interface Replicator {

    /**
     * Replication privilege
     * @since 5.5
     */
    String REPLICATE_PRIVILEGE = "{http://www.day.com/crx/1.0}replicate";

    /**
     * Triggers a new replication with supplied options.
     *
     * @param session user session
     * @param type The type of replication
     * @param path The path specified the content to be replicated
     * @throws ReplicationException if an error occurs
     */
    void replicate(Session session, ReplicationActionType type, String path)
            throws ReplicationException;

    /**
     * Triggers a new replication with supplied options.
     *
     * @param session user session
     * @param type The type of replication
     * @param path The path specified the content to be replicated
     * @param options Additional replication options
     * @throws ReplicationException if an error occurs
     */
    void replicate(Session session, ReplicationActionType type, String path,
                   ReplicationOptions options)
            throws ReplicationException;

    /**
     * Triggers a new replication with supplied options.
     *
     * @param session user session
     * @param type The type of replication
     * @param paths The paths specified the content to be replicated; it is recommended to consider 100 paths as maximum.<br>
     *   Up to that limit the system guarantees that the paths are replicated in a transactional manner. If this number is exceeded,
     *   the system is free to split up the provided paths into multiple chunks and replicate each chunk on its own; in that case 
     *   all the paths are replicated in a non-transactional path. The system will write log statements in this case. 
     * @param options Additional replication options
     * @throws ReplicationException if an error occurs
     */
    void replicate(Session session, ReplicationActionType type, String[] paths,
                   ReplicationOptions options)
            throws ReplicationException;

    /**
     * Checks if a user session has enough permissions for a provided replication type and path
     *
     * @param session user session
     * @param type The type of replication
     * @param path The path specified the content to be replicated
     * @throws ReplicationException if an error occurs
     * @since 5.5
     */
    void checkPermission(Session session, ReplicationActionType type, String path) throws ReplicationException;

    /**
     * Returns a replication status for the given path.
     * @param session session
     * @param path path to check
     * @return the replication status or <code>null</code> if not available.
     */
    ReplicationStatus getReplicationStatus(Session session, String path);

    /**
     * Returns the paths of all nodes for the given subtree path which are
     * activated.
     * 
     * @param session User session
     * @param path Path to check
     * @return Subtree paths for activated nodes
     * @throws ReplicationException If error occurs on retrieving node and its
     *             descendants
     */
    Iterator<String> getActivatedPaths(Session session, String path)
            throws ReplicationException;

    /**
     * Creates a {@link List} of {@link ReplicationContentFilter}s to be used by {@link ContentBuilder}s to filter
     * content for exclusion from the replication content being built. The filters in the list are not in a particular
     * order.
     *
     * @param action The {@link ReplicationAction} to build the filters contained in the list for.
     *
     * @return A {@link List} of {@link ReplicationContentFilter}s.
     * @deprecated
     */
    @Deprecated
    List<ReplicationContentFilter> createContentFilterChain(ReplicationAction action);

}
