/*
 *
 *	File: PSNameResolver.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.fontmanagement.platform;

import java.io.Serializable;

import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.fontmanagement.FontResolutionPriority;

/**
 * This interface provides access to the methods that a client can use to build a font set
 * that is used 
 * to resolve platform font names into actual {@link com.adobe.fontengine.font.Font} objects.
 * 
 * <h4>Concurrency</h4>
 * 
 * All classes that implement this interface are in general <b>not</b> guaranteed to be threadsafe.  Specific
 * implementations <i>may</i> offer tighter guarantees.
 */
public interface PlatformFontResolver extends Serializable
{
	/**
	 * Set the resolution mechanism that the PlatformNameResoulver should use to decide which
	 * font should be chosen when two fonts have the same platform name.
	 * <p> {@link com.adobe.fontengine.fontmanagement.FontResolutionPriority#INTELLIGENT_FIRST 
	 * FontResolutionPriority.INTELLIGENT_FIRST} specifies
	 * that an "intelligent" determination is made about the fonts and the "better" of the two font is chosen.
	 * If the "intelligent" resolver can make no distinction between the fonts then the first font  to have 
	 * been added first is chosen.
	 * <p> {@link com.adobe.fontengine.fontmanagement.FontResolutionPriority#INTELLIGENT_LAST
	 * FontResolutionPriority.INTELLIGENT_LAST} specifies
	 * that an "intelligent" determination is made about the fonts and the "better" of the two font is chosen.
	 * If the "intelligent" resolver can make no distinction between the fonts then the first font  to have 
	 * been added last is chosen.
	 * <p> {@link com.adobe.fontengine.fontmanagement.FontResolutionPriority#FIRST 
	 * FontResolutionPriority.FIRST} specifies
	 * the first font added to the font set that matches the platform attributes is chosen.
	 * <p> {@link com.adobe.fontengine.fontmanagement.FontResolutionPriority#LAST
	 * FontResolutionPriority.LAST} specifies
	 * the last font added to the font set that matches the platform attributes is chosen.
	 * @param priority The resolution mechanism  to use
	 * @return The old resolution priority setting.
	 */
	public abstract FontResolutionPriority setResolutionPriority(FontResolutionPriority priority);

	/**
	 * Adds a font to the set of fonts that are examined for resolving of platform names.
	 * If this font is indistinguishable from another font already in the font set then which
	 * of the two that is kept is determined by the settings of 
	 * {@link com.adobe.fontengine.fontmanagement.FontResolutionPriority FontResolutionPriority}.
	 * @param font the font to add to the fonts used for platform resolution
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 * @throws InvalidFontException
	 */
	public abstract void addFont(Font font) 
	throws UnsupportedFontException, InvalidFontException, FontLoadingException;

	/**
	 * Adds a font to the set of fonts that are examined for resolving of platform names.
	 * If this font's PlatformFontDescription is indistinguishable from another font already in 
	 * the font set then which of the two that is kept is determined by the settings of 
	 * {@link com.adobe.fontengine.fontmanagement.FontResolutionPriority FontResolutionPriority}.
	 * <p>Note that the PlatformFontDescription is NOT pushed down into the Font. It is only an
	 * alias in the database. Thus, calling 
	 * {@link com.adobe.fontengine.font.Font#getPostscriptFontDescription() font.getPostscriptFontDescription()}
	 * will not return the description 
	 * passed in through this method but will instead return the original platform description
	 * of the font.
	 * @param platformDesc the platform description of the font
	 * @param font the font to add to the fonts used for platform resolution
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 * @throws InvalidFontException
	 */
	public abstract void addFont(PlatformFontDescription platformDesc, Font font) 
	throws UnsupportedFontException, InvalidFontException, FontLoadingException;

	/**
	 * @return <code>true</code> if the font set is empty (has no fonts); <code>false</code> otherwise
	 */
	public abstract boolean isEmpty();

	/**
	 * Find a Font that matches the provided description. No analysis (e.g.,
	 * removal of a CMap extension) is done by this function. 
	 * @param searchAttributes the platform font attributes to search for
	 * @return the Font.
	 */
	public abstract Font findFont(PlatformFontSearchAttributes searchAttributes);
}