/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.workflow.launcher;

import org.osgi.annotation.versioning.ProviderType;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.List;


/**
 * The <code>ConfigEntry</code> represents <i>one</i> workflow launcher 
 * configuration entry
 */
@ProviderType
public class ConfigEntry {
    private String nodetype;
    private List<String> whereClauses;
    private int eventType;
    private String glob;
    private String workflow;
    private String id;
    private String description;
    private boolean enabled;
    private List<String> excludeList;
    private List<String> runModes;
    private List<String> features;
    private List<String> disabledFeatures;

    /**
     * Construct a ConfigEntry with the specified information.
     * @param eventType one of javax.jcr.observation.Event#NODE_ADDED
     * @param glob the path pattern, can be regexp, where the event should be watching
     * @param nodetype type of node to watch
     * @param whereClause node property conditions to watch
     * @param workflow the name of the workflow to trigger
     * @param id the id for this config
     * @param description description for this config
     * @param enabled true if the enabled, false for disabled
     * @param excludeList conditions to exclude from the event
     * @param runModes runmodes for this config
     * @deprecated
     */
    public ConfigEntry(int eventType, String glob, String nodetype,
                       String whereClause, String workflow, String id,
                       String description, boolean enabled,
                       List<String> excludeList, List<String> runModes) {
        this.eventType = eventType;
        this.glob = glob;
        this.nodetype = nodetype;
        List<String> conditions = new ArrayList<String>(1);
        conditions.add(whereClause);
        this.whereClauses = conditions;
        this.workflow = workflow;
        this.id = id;
        this.description = description;
        this.enabled = enabled;
        this.excludeList = excludeList;
        this.runModes = runModes;
    }

    /**
     * Construct a ConfigEntry with the specified information.
     * @param eventType one of javax.jcr.observation.Event#NODE_ADDED
     * @param glob the path pattern, can be regexp, where the event should be watching
     * @param nodetype type of node to watch
     * @param whereClauses list of node property conditions.  All conditions must be TRUE to launch the workflow.  A single condition entry can contain || for OR conditions
     * @param workflow the name of the workflow to trigger
     * @param id the id for this config
     * @param description description for this config
     * @param enabled true if the enabled, false for disabled
     * @param excludeList conditions to exclude from the event
     * @param runModes runmodes for this config
     * @param features features which must be enabled for this launcher to be active
     * @param disabledFeatures features which must be disabled for this launcher to be active
     */
    public ConfigEntry(int eventType, String glob, String nodetype,
                       List<String> whereClauses, String workflow, String id,
                       String description, boolean enabled,
                       List<String> excludeList, List<String> runModes,
                       List<String>features, List<String>disabledFeatures) {
        this.eventType = eventType;
        this.glob = glob;
        this.nodetype = nodetype;
        this.whereClauses = whereClauses;
        this.workflow = workflow;
        this.id = id;
        this.description = description;
        this.enabled = enabled;
        this.excludeList = excludeList;
        this.runModes = runModes;
        this.features = features;
        this.disabledFeatures = disabledFeatures;
    }

    /**
     * Return the name of the workflow to trigger.
     * @return workflow name.
     */
    public String getWorkflow() {
        return workflow;
    }

    /**
     * Set the name of the workflow to trigger.
     * @param workflow workflow name
     */
    public void setWorkflow(String workflow) {
        this.workflow = workflow;
    }

    /**
     * Return the JCR Event Type that triggers this config.
     * @return event type
     */
    public int getEventType() {
        return eventType;
    }

    /**
     * Set the event type to watch.
     * @param eventType event type to listen to.
     */
    public void setEventType(int eventType) {
        this.eventType = eventType;
    }

    /**
     * Returns the path pattern that triggers this launcher configuration.
     * @return path pattern.
     */
    public String getGlob() {
        return glob;
    }

    /**
     * Set the path pattern to trigger this launcher configuration.
     * @param glob path pattern.
     */
    public void setGlob(String glob) {
        this.glob = glob;
    }

    /**
     * Get the node type associated with this launcher.
     * @return node type
     */
    public String getNodetype() {
        return nodetype;
    }

    /**
     * Set the node type associated with this launcher.
     * @param nodetype the node type
     */
    public void setNodetype(String nodetype) {
        this.nodetype = nodetype;
    }

    /**
     * Return the where clause for this trigger.
     * The where clause is a condition of node properties,
     * e.g. approved==true
     * @deprecated use #getWhereClauses()
     * @return where clause
     */
    public String getWhereClause() {
        String whereClause = "";
        if (whereClauses != null && whereClauses.size() > 0) {
            whereClause = whereClauses.get(0);
        }
        return whereClause;
    }

    /**
     * Return the where clause for this trigger.
     * The where clause is a condition of node properties,
     * e.g. approved==true
     *
     * @return where clause
     */
    public List<String> getWhereClauses() {
        return whereClauses;
    }

    /**
     * Set the where clause for this trigger.
     * @param whereClause condition
     * @deprecated use #setWhereClauses
     */
    public void setWhereClause(String whereClause) {
        if (whereClauses == null) {
            whereClauses = new ArrayList<String>(1);
        }
        whereClauses.set(0, whereClause);
    }

    /**
     * Set the where clause for this trigger.
     * @param whereClauses conditions
     */
    public void setWhereClauses(List<String> whereClauses) {
        this.whereClauses = whereClauses;
    }

    /**
     * Return the ID of this trigger.
     * @return id
     */
    public String getId() {
        return id;
    }

    /**
     * Set the Id of this trigger
     * @param id trigger id.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * Get the description for this launcher configuration.
     * @return description
     */
    public String getDescription() {
        return description;
    }

    /**
     * Set the description for this launcher configuration.
     * @param description a new description.
     */
    public void setDescription(String description) {
        this.description = description;
    }

    private String getWhereClauseString() {
        String whereClause = "";
        boolean added = false;
        if (whereClauses != null) {
            for (String where : whereClauses) {
                if (StringUtils.isBlank(where)) {
                    continue;
                }

                if (added) {
                    whereClause += " && ";
                }
                whereClause += where;
                added = true;
            }
        }

        return whereClause;
    }

    private String getFeaturesString() {
        String featuresString = "";
        boolean added = false;
        if (this.features != null) {
            for (String feature : this.features) {
                if (added) {
                    featuresString += " && ";
                }
                featuresString += feature;
                added = true;
            }
        }

        return featuresString;
    }

    private String getDisabledFeaturesString() {
        String featuresString = "";
        boolean added = false;
        if (this.disabledFeatures != null) {
            for (String feature : this.disabledFeatures) {
                if (added) {
                    featuresString += " && ";
                }
                featuresString += "!" + feature;
                added = true;
            }
        }

        return featuresString;
    }

    public int hashCode() {
        String code = String.valueOf(eventType) + nodetype + getWhereClauseString() + glob + workflow + getFeaturesString() + getDisabledFeaturesString();
        return code.hashCode();
    }

    /**
     * Returns if this launcher configuration is enabled or not.
     * @return true if the configuration is enabled, false otherwise.
     */
    public boolean isEnabled() {
        return enabled;
    }

    /**
     * Enables or disables this launcher configuration.
     * @param enabled true to enable this configuration, false otherwise.
     */
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    /**
     * Returns a list of exclude conditions.
     * If conditions in this list evaluate to true the launcher configuration does not trigger.
     * @return list of conditions
     */
    public List<String> getExcludeList() {
        return excludeList;
    }

    /**
     * Set the list of exclude conditions.
     * @param excludeList new exclude conditions list.
     */
    public void setExcludeList(List<String> excludeList) {
        this.excludeList = excludeList;
    }

    /**
     * Return the list of runmode associated with this launcher configuration.
     * @return list of run modes.
     */
    public List<String> getRunModes() {
        return this.runModes;
    }

    /**
     * Set the run moodes associated with this launcher configuration.
     * @param runModes list of runmodes.
     */
    public void setRunModes(List<String> runModes) {
        this.runModes = runModes;
    }

    /**
     * return a list of features which must be enabled for this launcher configuration to be enabled
     * @return list of features for this launcher
     */
    public List<String> getFeatures() {
        return features;
    }

    /**
     * return a list of features which must be disabled or not found for this launcher configuration to be enabled
     * @return list of features for this launcher
     */
    public List<String> getDisabledFeatures() {
        return disabledFeatures;
    }

    /**
     * sets a list of features which must be enabled for this launcher configuration to be enabled

     * @param features list of features for this launcher
     */
    public void setFeatures(List<String> features) {
        this.features = features;
    }

    /**
     * sets a list of features which must be disabled for this launcher configuration to be enabled

     * @param disabledFeatures list of features which must be disabled
     */
    public void setDisabledFeatures(List<String> disabledFeatures) {
        this.disabledFeatures = disabledFeatures;
    }

    @Override
    public String toString() {
        return "ConfigEntry{" +
                "nodetype='" + nodetype + '\'' +
                ", whereClause='" + getWhereClauseString() + '\'' +
                ", eventType=" + eventType +
                ", glob='" + glob + '\'' +
                ", workflow='" + workflow + '\'' +
                ", id='" + id + '\'' +
                ", description='" + description + '\'' +
                ", enabled=" + enabled +
                ", excludeList'=" + excludeList + '\'' +
                ", runModes='" + runModes + '\'' +
                ", features='" + getFeaturesString() + '\'' +
                ", disabledFeatures='" + getDisabledFeaturesString() + '\'' +
                '}';
    }
}
