/* ****************************************************************************
 *
 *	File: CosCloneMgr.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import com.adobe.internal.io.ByteWriterFactory.Fixed;
import com.adobe.internal.io.stream.IO;
import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.io.stream.StreamManager;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;

public class CosCloneMgr
{
	private CosDocument mTarget;
	private HashMap<CosObjectRefAdapter, CosObjectRefAdapter> mClonedCosObjects;
	/**
	 * While shallow cloning, only streams which occur as value corresponding to 
	 * these names in a CosDictionary are deep copied. 
	 */
	private HashSet<ASName> streamsToClone;

	/**
	 * Constructor associated with the target document.
	 */
	public CosCloneMgr(CosDocument target)
	{
		mTarget = target;
		mClonedCosObjects = new HashMap<CosObjectRefAdapter, CosObjectRefAdapter>();
		streamsToClone = new HashSet<ASName>();
	}
	
	/**
	 * Constructor associated with the target document.
	 */
	public CosCloneMgr(CosDocument target, HashSet<ASName> streamsToClone)
	{
		this(target);
		if(streamsToClone != null)
			this.streamsToClone.addAll(streamsToClone);		
	}

	private CosObject cloneCosArray(CosArray cosObj)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray newCosArray = mTarget.createCosArray();
		if(cosObj.isIndirect())
			mClonedCosObjects.put(CosObjectRefAdapter.newInstance(cosObj), CosObjectRefAdapter.newInstance(newCosArray));
		for (int i = 0; i < cosObj.size(); i++) {
			newCosArray.add(clone(cosObj.get(i)));
		}
		return newCosArray;
	}

	private CosObject cloneCosDictionary(CosDictionary original, CosDictionary clone)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if(original.isIndirect())
			mClonedCosObjects.put(CosObjectRefAdapter.newInstance(original), CosObjectRefAdapter.newInstance(clone));
		Iterator<Map.Entry<ASName, CosObject>> iter = original.entrySet().iterator();
		CosDocument doc = original.getDocument();
		try {
			while(iter.hasNext()) {
				Map.Entry<ASName, CosObject> entry = iter.next();
				CosObject obj = entry.getValue();
				if (obj instanceof CosObjectRef)
					obj = doc.resolveReference((CosObjectRef)obj);					
				clone.put(entry.getKey(), clone(obj));
			}
		}
		catch (IOException e) {
			throw new PDFIOException(e);
		}
		return clone;
	}

	private CosObject cloneCosDictionary(CosDictionary cosObj)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		LinkedHashMap newMap = new LinkedHashMap(cosObj.size());
		CosDictionary newCosDictionary = mTarget.createCosDictionary(newMap);
		return cloneCosDictionary(cosObj, newCosDictionary);
	}

	private CosObject cloneCosStream(CosStream cosObj)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		CosStream cloneStream = mTarget.createCosStream();
		cloneStream = (CosStream)(cloneCosDictionary(cosObj, cloneStream));
		CosArray outputFilterList = cosObj.getOutputFiltersList();
		// Watson#3655750. If filters have been added to a cosstream, before saving they
		// are available in output filter list (and not part of cosdict) hence they
		// have to be cloned seperately.
		if(outputFilterList != null)
		{
			CosArray clonedOutputFilterList = (CosArray) cloneCosArray(outputFilterList);
			cloneStream.setOutputFiltersList(clonedOutputFilterList);
		}
		cloneStreamdata(cosObj, cloneStream);
		return cloneStream;
	}

	/**
	 * Clones the specified object and returns cloned object.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject clone(CosObject cosObj)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (cosObj instanceof CosBoolean)
			return mTarget.createCosBoolean(cosObj.booleanValue());
		if (cosObj instanceof CosName)
			return mTarget.createCosName(cosObj.nameValue());
		if (cosObj instanceof CosNull)
			return mTarget.createCosNull();
		if (cosObj instanceof CosNumeric)
			return mTarget.createCosNumeric((CosNumeric)cosObj);
		if (cosObj instanceof CosString) {
			CosString str = mTarget.createCosString(((CosString)cosObj).byteArrayValue());
			str.setWriteHex(((CosString)cosObj).getWriteHex());
			return str;
		}
		if (mClonedCosObjects.containsKey(CosObjectRefAdapter.newInstance(cosObj)))
			return mClonedCosObjects.get(CosObjectRefAdapter.newInstance(cosObj)).getObject();
		if (cosObj instanceof CosArray)
			return cloneCosArray((CosArray)cosObj);
		if (cosObj instanceof CosStream)
			try {
				return cloneCosStream((CosStream)cosObj);
			} catch (IOException e) {
				throw new PDFIOException(e);
			}
		if (cosObj instanceof CosDictionary)
			return cloneCosDictionary((CosDictionary)cosObj);
		return null;
	}
	
	/**
	 * Clones the specified object and returns cloned object.
	 * If this is a stream then it doesn't perform deep copying.
	 * @param cosObj CosObject which is to be cloned
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject shallowClone(CosObject cosObj)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		// Sending false as default value to signify that 
		// if this cosobject is a stream then do not deep copy it.
		try {
			return shallowClone(cosObj, false);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}
	
	/**
	 * This method clones the cosobject ignoring streams if copyStream is false .
	 * @param cosObj Cosobject which is to be cloned
	 * @param copyStream Set as true if deep copy of stream is to be made   
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException 
	 */
	private CosObject shallowClone(CosObject cosObj, boolean copyStream)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (cosObj instanceof CosBoolean)
			return mTarget.createCosBoolean(cosObj.booleanValue());
		if (cosObj instanceof CosName)
			return mTarget.createCosName(cosObj.nameValue());
		if (cosObj instanceof CosNull)
			return mTarget.createCosNull();
		if (cosObj instanceof CosNumeric)
			return mTarget.createCosNumeric((CosNumeric)cosObj);
		if (cosObj instanceof CosString) {
			CosString str = mTarget.createCosString(((CosString)cosObj).byteArrayValue());
			str.setWriteHex(((CosString)cosObj).getWriteHex());
			return str;
		}
		if (cosObj instanceof CosObjectRef)
			return shallowClone(mTarget.resolveReference((CosObjectRef)cosObj));
		
		
		if (cosObj instanceof CosArray)
			return shallowCloneCosArray((CosArray)cosObj);
		if (cosObj instanceof CosStream)
		{
			if(copyStream)
			{
				try {
					return shallowCloneCosStream((CosStream)cosObj);
				} catch (IOException e) {
					throw new PDFIOException(e);
				}
			}
			else
			{
				return cosObj; // No cloning	
			}
					
		}
		if (cosObj instanceof CosDictionary)
			return shallowCloneCosDictionary((CosDictionary)cosObj);
		return null;
	}
	
	private CosObject shallowCloneCosArray(CosArray cosObj)
	throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		CosArray newCosArray = mTarget.createCosArray();
		for (int i = 0; i < cosObj.size(); i++) {
			newCosArray.add(shallowClone(cosObj.get(i)));
		}
		return newCosArray;
	}

	private CosObject shallowCloneCosDictionary(CosDictionary cosObj)
	throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		LinkedHashMap newMap = new LinkedHashMap(cosObj.size());
		CosDictionary newCosDictionary = mTarget.createCosDictionary(newMap);
		return shallowCloneCosDictionary(cosObj, newCosDictionary);
	}
	
	private CosObject shallowCloneCosDictionary(CosDictionary original, CosDictionary clone)
	throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		Iterator<Map.Entry<ASName, CosObject>> iter = original.entrySet().iterator();
		
		while(iter.hasNext()) {
			Map.Entry<ASName, CosObject> entry = iter.next();
			clone.put(entry.getKey(), shallowClone(entry.getValue(), streamsToClone != null && streamsToClone.contains(entry.getKey())));
		}
		return clone;
	}
	
	private CosObject shallowCloneCosStream(CosStream cosObj)
	throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{	
		CosStream cloneStream = mTarget.createCosStream();
		cloneStream = (CosStream)(shallowCloneCosDictionary(cosObj, cloneStream));
		cloneStreamdata(cosObj, cloneStream);
		return cloneStream;		
	}
	
	// Copies data from source cosstream to cloned stream 
	private void cloneStreamdata(CosStream cosObj, CosStream cloneStream)
	throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		boolean encoded = cosObj.isEncoded();
		InputByteStream dataStream = null;
		if(encoded)
			dataStream = cosObj.getStreamEncoded();
		else
			dataStream = cosObj.getStreamDecoded();
		if (dataStream != null) {
			// Must be sure to use the StreamManager on the destination side
			StreamManager streamManager = cloneStream.getStreamManager();
			OutputByteStream clonedData = streamManager.getOutputByteStreamEncryptedDocument(Fixed.FIXED, dataStream.length());
			IO.copy(dataStream, clonedData);
			if (encoded)
				cloneStream.newDataEncoded(clonedData.closeAndConvert());
			else
				cloneStream.newDataDecoded(clonedData.closeAndConvert());
			dataStream.close(); // must close the stream
		}		
	}
	
	public CosObject removeMapping(CosObject cosObj)
	{
		CosObjectRefAdapter object = mClonedCosObjects.remove(CosObjectRefAdapter.newInstance(cosObj));
		if(object == null)
			return null;
		return object.getObject();
	}
}
