/*
 * File: CosObjectInfo.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.lang.ref.SoftReference;
import java.util.Iterator;
import java.util.List;

import com.adobe.internal.io.stream.IO;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.util.StringOps;

/**
 * Contains the information for lazy parsing of cos objects and
 * for noting the position of objects in the output file so that
 * the xref table can be generated.
 */
public final class CosObjectInfo {
	/* object states */
	private static final int isFree = 0;		// unused
	private static final int isAssigned = 1;	// assigned but unrooted and uninstantiated
	private static final int isAddressed = 2;	// has storage address but not loaded
	private static final int isLoaded = 3;		// has storage address and loaded
	private static final int isDirty = 4;		// instantiated and dirty but unrooted

	/* fields */
	private int mState;				// object state
	private int mObjNum;				// indirect object number
	private int mObjGen;				// indirect object generation
	private long mPos;				// object storage offset
	private CosDocument mDoc;			// corresponding CosDocument
	private Object mObj;				// pointer to object or softref if instantiated
	private CosObjectRef mRef;			// cache the object reference
	private CosObjectInfo mObjStmInfo;		// Info of parent CosObjectStream if compressed
	private int mStreamNdx;				// index in stream if compressed
	private boolean mIsObjStm;			// true if the associated object is an object stream
	private boolean mWriteCompressed;		// true if the associated object will be written into an object stream
	private long mNextObjPos = CosToken.DEFAULT_NEXT_OBJ_POS;// the position of the next object in document stream.

	/**
	 * Constructor for CosObjectInfo
	 * @param objNum - indirect object number
	 * @param objGen - indirect object generation
	 */
	CosObjectInfo(CosDocument doc, int objNum, int objGen)
	{
		mDoc = doc;
		mState = isAssigned;
		mObjNum = objNum;
		mObjGen = objGen;
		mPos = 0;
	}

	/**
	 * Constructor a free CosObjectInfo
	 */
	CosObjectInfo()
	{
		mState = isFree;
		mObjNum = 0;
		mObjGen = 65535;
		mPos = 0;
	}

	/* methods */

	public int getObjNum()
	{
		return mObjNum;
	}

	void setObjNum(int objNum)
	{
		mObjNum = objNum;
	}

	public int getObjGen()
	{
		return mObjGen;
	}

	void setIsObjStm(boolean isObjStm)
	{
		mIsObjStm = isObjStm;
	}

	boolean isObjStm()
	{
		return mIsObjStm;
	}

	void setWriteCompressed(boolean writeCompressed)
	{
		mWriteCompressed = writeCompressed;
	}

	boolean isWriteCompressed()
	{
		return mWriteCompressed;
	}

	void setObjGen(int objGen)
	{
		mObjGen = objGen;
	}

	boolean isCompressed()
	{
		return mObjStmInfo != null;
	}

	public CosObject getObject()
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		return getObject(true);
	}

	public CosObject getObject(boolean loadIfRequired)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		CosObject obj = mDoc != null ? mDoc.getRepairedValue(this.getObjNum()) : null;
		if(obj != null)
			return obj;// the object was repaired so return the repaired object.
		if (mObj == null || mObj instanceof CosObject)
			return (CosObject)mObj;
		obj = (CosObject)((SoftReference)mObj).get();
		if (loadIfRequired && (obj == null)) {
			mState = isAddressed;
			obj = mDoc.getXRef().getIndirectObject(this);
			mObj = new SoftReference(obj);
		}
		return obj;
	}

	void setObject(CosObject obj)
	{
		mDoc = obj.getDocument();
		if (mDoc.isCacheEnabled())
			mObj = new SoftReference(obj);
		else
			mObj = obj;
	}

	void clearObject()
	{
		mObj = null;
	}

	public long getPos()
	{
		return mPos;
	}

	void setPosInternal(long pos)
	{
		mPos = pos;
	}

	void setPos(long pos)
	{
		if (mPos != pos) {
			if (mPos != 0) {
				// We're moving it
				Object obj = mObj;
				if (obj instanceof SoftReference)
					obj = ((SoftReference)mObj).get();
				if (obj instanceof CosStream)
					((CosStream)obj).adjustPos(pos - mPos);
			}
			mPos = pos;
		}
	}
	
	/**
	 * Sets the offset of the next object.
	 * @param pos
	 */
	void setNextObjPos(long pos)
	{
		mNextObjPos = pos;
	}
	
	/**
	 * Returns the offset of next object.
	 * @return long
	 */
	long getNextObjPos()
	{
		return mNextObjPos;
	}

	public int getStreamNdx()
	{
		return mStreamNdx;
	}

	void setStreamNdx(int ndx)
	{
		mStreamNdx = ndx;
	}

	public CosObjectInfo getStreamInfo()
	{
		return mObjStmInfo;
	}

	void setStreamInfo(CosObjectInfo stm)
	{
		mObjStmInfo = stm;
		mWriteCompressed = false;
	}

	public CosObjectRef getRef()
	{
		return mRef;
	}

	void setRef(CosObjectRef ref)
	{
		mRef = ref;
	}

	boolean isFree()
	{
		return mState == isFree;
	}

	void markFree()
	{
		mState = isFree;
		mObj = null;
	}

	boolean isAssigned()
	{
		return mState == isAssigned;
	}

	void markAssigned()
	{
		mState = isAssigned;
	}

	boolean isAddressed()
	{
		return mState == isAddressed;
	}

	void markAddressed()
	{
		mState = isAddressed;
	}

	boolean isLoaded()
	{
		return mState == isLoaded;
	}

	void markLoaded()
	{
		mState = isLoaded;
	}

	boolean isDirty()
	{
		return mState == isDirty;
	}

	void markDirty()
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		mState = isDirty;
		if (mDoc != null)
			mDoc.markDirty();
		if (mObj == null || mObj instanceof SoftReference)
			mObj = getObject();
	}

	public int getState()
	{
		return mState;
	}

	void setState(int state)
	{
		mState = state;
	}

	/**
	 * If this object is can be flushed, flush it.
	 * Returns false if dirty.
	 */
	boolean flushCachedObject()
	{
		if (isDirty())
			return false;
		if (isLoaded()) {
			markAddressed();
			mObj = null;
		}
		return true;
	}

	/**
	 * Mark object as not dirty if it was dirty.
	 * Returns false if not dirty.
	 */
	boolean markNotDirty()
	{
		if (!isDirty())
			return false;
		markLoaded();
		if (mObj instanceof CosObject)
			mObj = new SoftReference(mObj);
		return true;
	}

	long writeIndirectObject(OutputByteStream obs, long pos)
	throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		return writeIndirectObject(obs, pos, false);
	}
	
	long writeIndirectObject(OutputByteStream obs, long pos, boolean saveToCopy)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		if (isFree())
			return pos;
		obs.write(StringOps.toByteArray(Integer.toString(mObjNum)));
		obs.write(' ');
		obs.write(StringOps.toByteArray(Integer.toString(mObjGen)));
		obs.write(StringOps.toByteArray(" obj\n"));
		CosObject obj = getObject();
		loadStrings(obj, null, null);
		mPos = pos;
		mObjStmInfo = null;
		mWriteCompressed = false;
		mStreamNdx = 0;
		mNextObjPos = CosToken.DEFAULT_NEXT_OBJ_POS;
		if(obj instanceof CosStream)
		{
			((CosStream)obj).writeOut(obs, false, false, saveToCopy);
		}
		else
		{
			obj.writeOut(obs);
		}
		obs.write(StringOps.toByteArray("\nendobj\n"));
		return obs.getPosition();
	}

	void loadStrings(CosObject obj, CosObject parent, ASName key)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		if (obj instanceof CosString) {
			// EchoSign bug#3304111 : If the string is expected to be encrypted but is not actually encrypted then we shall skip it if
			// user wants us to do so.
			if(parent != null && ((CosString)obj).getIsEncrypted() && mDoc.getOptions().skipCorruptObjects()) {
				try {
					((CosString)obj).byteArrayValue();
				} catch(Exception e){
					// This may happen when this string is present as un-encrypted while it was expected to be encrypted.
					CosString repairedString = new CosString(mDoc, new byte[]{}, 0, 0, false, null);
					repairedString.setParentObj((CosDictionary) parent);
					((CosDictionary)parent).setRepairedValue(key, repairedString);
				}
			} else// otherwise just go with the normal routine.
				((CosString)obj).byteArrayValue();
		} else if (obj instanceof CosArray) {
			for (int i = 0; i < ((CosArray)obj).size(); i++) {
				if (((CosArray)obj).getRef(i) == null) {
					CosObject child = ((CosArray)obj).get(i);
					if (!child.isIndirect())
						loadStrings(child, null, null);
				}
			}
		} else if (obj instanceof CosDictionary) {
			List<ASName> keys = ((CosDictionary)obj).getKeys();
			Iterator<ASName> iter = keys.iterator();
			while (iter.hasNext()) {
				key = iter.next();
				if (((CosDictionary)obj).getRef(key) == null) {
					CosObject child = ((CosDictionary)obj).get(key);
					if (!child.isIndirect()){
						loadStrings(child, obj, key);
					}
				}
			}
		}
	}

	void writeXRefTableEntry(OutputByteStream obs)
		throws IOException
	{
		if (isFree()) {
			writeInt(obs, 0, 10);
			obs.write(' ');
			writeInt(obs, 65535, 5);
			obs.write(StringOps.toByteArray(" f\r\n"));
		} else {
			writeInt(obs, mPos, 10);
			obs.write(' ');
			writeInt(obs, mObjGen, 5);
			obs.write(StringOps.toByteArray(" n\r\n"));
		}
	}

	void writeInt(OutputByteStream obs, long num, int size)
		throws IOException
	{
		int i;
		int maxDig = 10;
		if (size > maxDig)
			size = maxDig;
		char[] digits = new char[maxDig];
		int work = 1000000000;
		for (i = 0; i < maxDig-1; i++) {
			digits[i] = (char)('0' + num/work);
			num %= work;
			work /= 10;
		}
		digits[i] = (char)('0' + num);
		for (i = maxDig - size; i < maxDig; i++)
			obs.write(digits[i]);
	}

	void writeXRefStreamEntry(OutputByteStream buf, int[] fieldsizes)
		throws PDFCosParseException, IOException, PDFSecurityException
	{
		if (isCompressed()) {
			buf.write(IO.longToByteArray(2, fieldsizes[0]));
			buf.write(IO.longToByteArray(mObjStmInfo.getObjNum(), fieldsizes[1]));
			buf.write(IO.longToByteArray(mStreamNdx, fieldsizes[2]));
		} else {
			if (isFree()) {
				buf.write(IO.longToByteArray(0, fieldsizes[0]));
				buf.write(IO.longToByteArray(0, fieldsizes[1]));
				buf.write(IO.longToByteArray(65535, fieldsizes[2]));
			} else {
				buf.write(IO.longToByteArray(1, fieldsizes[0]));
				buf.write(IO.longToByteArray(mPos, fieldsizes[1]));
				buf.write(IO.longToByteArray(mObjGen, fieldsizes[2]));
			}
		}
	}

	/*
	 * Clone values from template to this
	 */
	public void copyValuesFrom(CosObjectInfo template)
	{
		mState = template.mState;
		mObjNum = template.mObjNum;
		mObjGen = template.mObjGen;
		mPos = template.mPos;
		mDoc = template.mDoc;
		mObj = template.mObj;
		mRef = template.mRef;
		mObjStmInfo = template.mObjStmInfo;
		mStreamNdx = template.mStreamNdx;
		mIsObjStm = template.mIsObjStm;
		mWriteCompressed = template.mWriteCompressed;
		mNextObjPos = template.mNextObjPos;
	}
	
	@Override
	public String toString()
	{
		StringBuilder builder = new StringBuilder();
		builder.append("[");
		builder.append(this.mObjNum);
		builder.append(":");
		builder.append(this.mObjGen);
		builder.append("] = ");
		switch(this.mState)
		{
		case isAddressed:
			builder.append("addressed");
			break;
		case isAssigned:
			builder.append("assigned");
			break;
		case isDirty:
			builder.append("dirty");
			break;
		case isFree:
			builder.append("free");
			break;
		case isLoaded:
			builder.append("loaded");
			break;
		}
		builder.append(" at 0x");
		builder.append(Long.toHexString(this.mPos));
		
		return builder.toString();
	}
}
